
#include <SceneOverlayPanels.h>

#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOverlayGroup.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoTransform.h>

/*
The class is building a scene graph composed of panels on which are aggrageted different co-planar geometries with of without transparency.
Each panel requires its specific SoOverlayGroup in which the geometries must added from back to front. 
*/

#define NUM_PTS_PROFILE 4
const int numPanels = NUM_PTS_PROFILE - 1;
const int numPolylines = 4;


SoSelection*
SceneOverlayPanels::getSceneGraph()
{
  if ( m_scene == nullptr )
    m_scene = getMultiPanelScene();
  return m_scene.ptr();
}

void
SceneOverlayPanels::generatedProfile()
{
  float x0 = -NUM_PTS_PROFILE / 2;
  for ( int i = 0; i < NUM_PTS_PROFILE; i++ )
  {
    float x = x0 + i;
    float z = -.5f + ( float )rand() / ( float )RAND_MAX;
    m_profile.push_back( SbVec3f( x, 0, z ) );
  }
}

SoMaterial*
SceneOverlayPanels::getRandomMat()
{
  float R = ( float )rand() / ( float )RAND_MAX * .6f + .4f;
  float G = ( float )rand() / ( float )RAND_MAX * .6f + .4f;
  float B = ( float )rand() / ( float )RAND_MAX * .6f + .4f;
  auto rMat = new SoMaterial;
  rMat->diffuseColor = SbVec3f( R, G, B );
  rMat->transparency = 0.f;
  return rMat;
}

SoFaceSet*
SceneOverlayPanels::getFence( int overlayIndex, int panelIndex, float bottom, float range )
{

  auto strip = new SoFaceSet;
  if ( panelIndex < numPanels )
  {

    auto vertices = new SoVertexProperty;
    strip->vertexProperty = vertices;

    SbVec3f p1 = m_profile[panelIndex];
    SbVec3f p2 = m_profile[panelIndex + 1];
    SbVec3f normal = p2 - p1;
    vertices->vertex.set1Value( 0, p1[0], bottom, p1[2] );
    vertices->vertex.set1Value( 1, p2[0], bottom, p2[2] );
    vertices->vertex.set1Value( 2, p2[0], bottom + range, p2[2] );
    vertices->vertex.set1Value( 3, p1[0], bottom + range, p1[2] );
    vertices->normal.set1Value( 0, -normal[2], 0, normal[0] );
    strip->numVertices.set1Value( 0, 4 );

    vertices->normalBinding = SoVertexProperty::Binding::PER_FACE;

    std::string name = "Panel_" + std::to_string( overlayIndex ) + "_" + std::to_string( panelIndex );
    strip->setName( name.c_str() );
  }
  return strip;
}

SoLineSet*
SceneOverlayPanels::getFenceLine( int overlayIndex, int panelIndex, float bottom, float range )
{

  auto polyLine = new SoLineSet;
  if ( panelIndex < numPanels )
  {
    auto nptsPerFace = 10;
    SoVertexProperty* vertices = new SoVertexProperty;

    int pindex = 0;
    SbVec3f p1 = m_profile[panelIndex];
    SbVec3f p2 = m_profile[panelIndex + 1];

    vertices->vertex.set1Value( pindex++, p1[0], bottom + range / 2, p1[2] );

    for ( int i = 0; i < nptsPerFace - 2; i++ )
    {
      float y = bottom + range * ( float )rand() / ( float )RAND_MAX;
      float step = 1.f / ( float )( nptsPerFace - 1 ) * ( i + 1 );
      SbVec3f p = p1 + ( p2 - p1 ) * step;
      vertices->vertex.set1Value( pindex++, p[0], y, p[2] );
    }
    vertices->vertex.set1Value( pindex++, p2[0], bottom + range / 2, p2[2] );

    polyLine->vertexProperty = vertices;

    std::string name = "Line_" + std::to_string( overlayIndex ) + "_" + std::to_string( panelIndex );
    polyLine->setName( name.c_str() );
  }
  return polyLine;
}

SoTransform*
SceneOverlayPanels::getPanelNamePosition( int panelIndex, float bottom, float range )
{

  auto namePosition = new SoTransform;
  if ( panelIndex < numPanels )
  {
    SoVertexProperty* vertices = new SoVertexProperty;

    SbVec3f p1 = m_profile[panelIndex];
    SbVec3f p2 = m_profile[panelIndex + 1];

    SbVec3f p = p1 + ( p2 - p1 ) * 0.1f;
    namePosition->translation = SbVec3f( p[0], bottom + range * .9f, p[2] );

    namePosition->rotation = SbRotation( SbVec3f( 1, 0, 0 ), p2 - p1 );
  }
  return namePosition;
}

SoSelection*
SceneOverlayPanels::getMultiPanelScene()
{
  generatedProfile();

  auto scene = new SoSelection;
  auto sh = new SoShapeHints;
  scene->addChild( sh );
  sh->vertexOrdering = SoShapeHints::VertexOrdering::COUNTERCLOCKWISE;

  auto font = new SoFont;
  font->size = 0.1f;
  scene->addChild( font );

  auto overlaySep = new SoSeparator;
  scene->addChild( overlaySep );

  for ( int i = 0; i < numPanels; i++ )
  {
    auto overlay = new SoOverlayGroup;
    overlaySep->addChild( overlay );
  }

  auto mat = new SoMaterial;
  mat->diffuseColor = SbVec3f( 0.8f, 0.7f, 0.1f );

  int overlayIndex = 0;
  for ( int i = 0; i < numPanels; i++ )
  {
    SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );

    overlay->addChild( mat );
    overlay->addChild( getFence( overlayIndex, i, -1.f, 2.f ) );
  }
  overlayIndex++;

  mat = new SoMaterial;
  mat->diffuseColor = SbVec3f( 0, 0.3f, 0.9f );
  auto ds = new SoDrawStyle;
  ds->lineWidth = 3.f;
  for ( int i = 0; i < numPanels; i++ )
  {
    SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );
    float bottom = -0.9f;
    float range = 1.8f;
    overlay->addChild( ds );
    overlay->addChild( mat );
    overlay->addChild( getFenceLine( overlayIndex, i, bottom, range ) );
  }
  overlayIndex++;

  mat = new SoMaterial;
  mat->diffuseColor = SbVec3f( 0.9f, 0, 0.2f );
  mat->transparency = 0.f;
  for ( int i = 0; i < numPanels; i++ )
  {
    SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );

    overlay->addChild( mat );
    overlay->addChild( getFence( overlayIndex, i, -0.1f, .7f ) );
  }
  overlayIndex++;

  for ( int nl = 0; nl < numPolylines / 2; nl++ )
  {
    auto ds = new SoDrawStyle;
    ds->lineWidth = nl % 3 ? 3.f : 2.f;
    ds->linePattern = nl % 2 ? 0xf00f : 0xffff;

    mat = getRandomMat();
    for ( int i = 0; i < numPanels; i++ )
    {
      SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );
      float bottom = -0.9f + 0.1f * nl;
      float range = 1.2f;
      overlay->addChild( mat );
      overlay->addChild( ds );
      overlay->addChild( getFenceLine( overlayIndex, i, bottom, range ) );
    }
    overlayIndex++;

  }


  mat = new SoMaterial;
  mat->diffuseColor = SbVec3f( 0, 0.7f, 0.2f );
  mat->transparency = 0.5f;
  for ( int i = 0; i < numPanels; i++ )
  {
    SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );

    overlay->addChild( mat );
    overlay->addChild( getFence( overlayIndex, i, -0.6f, 0.7f ) );
  }
  overlayIndex++;

  for ( int nl = 0; nl < numPolylines / 2; nl++ )
  {
    auto ds = new SoDrawStyle;
    ds->lineWidth = nl % 3 ? 3.f : 2.f;
    ds->linePattern = nl % 2 ? 0xf00f : 0xffff;

    mat = getRandomMat();
    for ( int i = 0; i < numPanels; i++ )
    {
      SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );
      float bottom = -0.6f + 0.1f * nl;
      float range = 1.2f;
      overlay->addChild( mat );
      overlay->addChild( ds );
      overlay->addChild( getFenceLine( overlayIndex, i, bottom, range ) );
    }
    overlayIndex++;
  }

  mat = new SoMaterial;
  mat->emissiveColor = SbVec3f( 1, 1, 1 );
  for ( int i = 0; i < numPanels; i++ )
  {
    SoOverlayGroup* overlay = static_cast<SoOverlayGroup*>( overlaySep->getChild( i ) );
    float bottom = -1.f;
    float range = 2.f;

    auto textSep = new SoSeparator;
    overlay->addChild( textSep );

    textSep->addChild( mat );
    textSep->addChild( getPanelNamePosition( i, bottom, range ) );

    auto panelName = new SoText3;
    textSep->addChild( panelName );
    std::string pname = "Panel " + std::to_string( i );
    panelName->string = pname;
  }

  return scene;
}

