#include <Scene.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShaderObject.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoOrthographicCamera.h>

Scene::Scene()
{
  m_origin = SbVec3f( NUM_POINTS_PER_AXIS * POINT_SPACE, 0.f, NUM_POINTS_PER_AXIS * POINT_SPACE ) / 2;
  const int numPoints = NUM_POINTS_PER_AXIS * NUM_POINTS_PER_AXIS;
  std::vector<SbVec3f> points( numPoints );
  std::vector<SbVec3f> normals( numPoints );

  int pIndex = 0;
  for ( int j = 0; j < NUM_POINTS_PER_AXIS; ++j )
  {
    for ( int i = 0; i < NUM_POINTS_PER_AXIS; ++i )
    {
      points[pIndex] = getPoint( i, j );
      SbVec3f normal = computeNormal( points[pIndex], getPoint( i - 1, j - 1 ), getPoint( i + 1, j - 1 ) );
      normal.normalize();
      normals[pIndex] = normal;
      pIndex++;
    }
  }

  m_root = new SoSeparator;
  m_drawStyle = new SoDrawStyle;
  m_drawStyle->pointSize = 1.0f;
  m_root->addChild( m_drawStyle );

  SoMaterial* material = new SoMaterial;
  material->diffuseColor.setValue( 1.0f, 1.0f, 1.0f );
  m_root->addChild( material );

  SoShaderProgram* shader = new SoShaderProgram;
  SoShaderObject* shaderObject = shader->setVertexShader( 0, "$OIVHOME/examples/source/Inventor/Features/PointCloud/shaders/Sprite_vert.glsl" );
  shader->setFragmentShader( 1, "$OIVHOME/examples/source/Inventor/Features/PointCloud/shaders/Sprite_frag.glsl" );

  SoShaderParameter1f* spBasePointSize = shaderObject->addShaderParameter1f( "basePointSize", 1.0f );
  spBasePointSize->value.connectFrom( &m_drawStyle->pointSize );

  m_spSizeVarying = shaderObject->addShaderParameter1i( "sizeVarying", true );

  m_spColorVarying = shaderObject->addShaderParameter1i( "colorVarying", true );

  m_root->addChild( shader );

  SoPointSet* ps = new SoPointSet;
  SoVertexProperty* vp = new SoVertexProperty;
  ps->vertexProperty = vp;
  vp->vertex.setNum( numPoints );
  vp->vertex.setValues( 0, numPoints, points.data() );
  vp->normal.setValues( 0, numPoints, normals.data() );
  vp->materialBinding = SoVertexProperty::PER_VERTEX;

  m_root->addChild( ps );

  SoSeparator* annotationSep = new SoSeparator;
  SoAnnotation* annotation = new SoAnnotation;
  SoOrthographicCamera* camera = new SoOrthographicCamera;
  camera->nearDistance = -1.0f;
  camera->farDistance = 1.0f;
  camera->position = SbVec3f(0.99f, -0.95f, 0.0f);
  camera->viewportMapping.setValue(SoCamera::LEAVE_ALONE);
  annotation->addChild(camera);

  SoMaterial* textMaterial = new SoMaterial;
  textMaterial->diffuseColor = SbColor(1.0f, 0.0f, 0.0f);
  annotation->addChild(textMaterial);

  SoPickStyle *pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;
  annotation->addChild(pickStyle);

  SoTextProperty* textProperty = new SoTextProperty;
  textProperty->style = SoTextProperty::Style::BACK_FRAME;
  textProperty->styleColors.set1Value(SoTextProperty::StyleColor::BACK_FRAME_COLOR, SbVec4f(1.0f, 1.0f, 1.0f, 1.0f));
  annotation->addChild(textProperty);

  SoFont* fontNode = new SoFont();
  fontNode->size = 15;
  fontNode->renderStyle = SoFont::TEXTURE;
  annotation->addChild(fontNode);

  m_text = new SoText2;
  annotation->addChild(m_text.ptr());
  m_text->string = "Picked point: none";

  annotationSep->addChild(annotation);

  m_root->addChild(annotationSep);
}

void
Scene::setText(const SbString& text)
{
  m_text->string = text;
}

SbVec3f
Scene::getPoint( int i, int j )
{
  SbVec3f p = SbVec3f( i * 10.f, 0.f, j * 10.f ) - m_origin;
  p[1] = NUM_POINTS_PER_AXIS / 1.f * ( float )( cos( ( p - m_origin ).length() / ( NUM_POINTS_PER_AXIS * 10 ) * M_PI ) );
  return p;
}

SbVec3f
Scene::computeNormal( const SbVec3f& p0, const SbVec3f& p1, const SbVec3f& p2 )
{
  SbVec3f vec0 = p1 - p0;
  SbVec3f vec1 = p2 - p0;
  return vec1.cross( vec0 );
}

SoSeparator*
Scene::getRoot() const
{
  return m_root.ptr();
}

SoShaderParameter1i*
Scene::getSpColorVarying() const
{
  return m_spColorVarying;
}

SoShaderParameter1i*
Scene::getSpSizeVarying() const
{
  return m_spSizeVarying;
}

SoDrawStyle*
Scene::getDrawStyle() const
{
  return m_drawStyle;
}

void Scene::clear(){
  m_root = nullptr;
}
