#include <Inventor/SoDB.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/ViewerComponents/Qt/QtHelper.h>
#include <Inventor/ViewerComponents/Qt/RenderAreaOrbiter.h>
#include <Inventor/actions/SoRayPickAction.h>
#include <Inventor/details/SoPointDetail.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoShaderParameter.h>
#include <Inventor/nodes/SoStereoCamera.h>
#include <QApplication>
#include <QCheckBox>
#include <QMainWindow>
#include <QVBoxLayout>
#include <Scene.h>
#include "DoubleSpinBoxSliderWidget.h"

void
rayPickCallback(void* userData, SoEventCallback *eventCB)
{
  const SoMouseButtonEvent* btnEvent = static_cast<const SoMouseButtonEvent*>(eventCB->getEvent());
  if (!SO_MOUSE_PRESS_EVENT(btnEvent, BUTTON1))
		return;

  const SbVec2s& pos = btnEvent->getPosition();
  const SoHandleEventAction* action = eventCB->getAction();
  const SbViewportRegion& viewport = action->getViewportRegion();

  SoRayPickAction* rpa = new SoRayPickAction(viewport);
  SoSceneManager* sceneManager = action->getSceneManager();
  rpa->setSceneManager(sceneManager);
  rpa->setPoint(pos);

  Scene* scene = static_cast<Scene*>(userData);
  rpa->apply(sceneManager->getSceneGraph());
  SoPickedPoint* pickedPoint = rpa->getPickedPoint();
  const SoDetail* detail = pickedPoint ? pickedPoint->getDetail() : nullptr;

  static SbString baseStr = "Picked point: ";
  const SoPointDetail* pointDetail = dynamic_cast<const SoPointDetail*>(detail);
  SbString pickedStr = "none";
  if (pointDetail)
    pickedStr = SbString("#") + SbString(pointDetail->getCoordinateIndex());

  scene->setText(baseStr + pickedStr);
}

int
main( int argc, char** argv )
{
  QtHelper::addPlatformPluginsPath();

  QApplication app( argc, argv );

  SoDB::init();

  QMainWindow* window = new QMainWindow;

  QWidget* centralWidget = new QWidget( window );
  centralWidget->setLayout( new QVBoxLayout );
  centralWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  centralWidget->layout()->setSpacing( 0 );

  QWidget* topWidget = new QWidget( centralWidget );
  centralWidget->layout()->addWidget( topWidget );
  topWidget->setLayout( new QHBoxLayout );
  topWidget->layout()->setContentsMargins( 0, 0, 0, 0 );
  topWidget->setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Maximum );


    DoubleSpinBoxSliderWidget* spinBox = new DoubleSpinBoxSliderWidget( topWidget );
    topWidget->layout()->addWidget( spinBox );
    spinBox->setSingleStep( 0.1 );
    spinBox->setRange( 1, 20 );
    spinBox->setLabel( "Reference point size" );

    QCheckBox* sizeVaryingCB = new QCheckBox( topWidget );
    topWidget->layout()->addWidget( sizeVaryingCB );
    sizeVaryingCB->setText( "Varying point size" );
    sizeVaryingCB->setChecked( true );


    QCheckBox* colorVaryingCB = new QCheckBox( topWidget );
    topWidget->layout()->addWidget( colorVaryingCB );
    colorVaryingCB->setText( "Varying color" );
    colorVaryingCB->setChecked( true );

  qobject_cast<QBoxLayout*>( topWidget->layout() )->addStretch();


  RenderAreaOrbiter* renderArea = new RenderAreaOrbiter( window );

  const float grayVal = 0.99f;
  renderArea->setClearColor( SbColorRGBA( grayVal, grayVal, grayVal, 1.0f ) );

  centralWidget->layout()->addWidget( renderArea->getContainerWidget() );
  window->setCentralWidget( centralWidget );


  Scene scene;
  SoSeparator* root = scene.getRoot();
  SoEventCallback* rayPickCB = new SoEventCallback;
  rayPickCB->addEventCallback<SoMouseButtonEvent>(rayPickCallback, &scene);
  root->addChild(rayPickCB);

  renderArea->setSceneGraph( root );
  renderArea->viewAll( SbViewportRegion() );
  renderArea->setAntialiasingMode( SoSceneManager::AntialiasingMode::AUTO );
  renderArea->setAntialiasingQuality( 1 );

  SoStereoCamera* cam = dynamic_cast<SoStereoCamera*>( renderArea->getSceneInteractor()->getCamera() );
  cam->position = SbVec3f( 5004.26f, -895.473f, -3690.13f );
  cam->orientation = SbRotation( SbVec3f( 0.0571383f, -0.994446f, -0.0883862f ), 4.28429f );
  cam->focalDistance = 5000.0f;


  window->setCentralWidget( centralWidget );
  window->resize( 949, 614 );
  window->show();

  // Connections
  QObject::connect( spinBox, &DoubleSpinBoxSliderWidget::valueChanged, [&scene]( double value ) { scene.getDrawStyle()->pointSize = value; } );
  QObject::connect( sizeVaryingCB, &QCheckBox::clicked, [&scene]( bool checked ) { scene.getSpSizeVarying()->value = checked; } );
  QObject::connect( colorVaryingCB, &QCheckBox::clicked, [&scene]( bool checked ) { scene.getSpColorVarying()->value = checked; } );

  spinBox->setValue( 2.0f );

  app.exec();

  // We free the memory.
  scene.clear();
  delete window;
  SoDB::finish();
  return 0;
}
