/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

///////////////////////////////////////////////////////////////////////
// 
// This example shows you how to code within a vertex shader program
// an animation in order to speed up the rendering.
//
// Source code involved with shaders is framed with 
// BEGIN SHADER and END SHADER.
//
// A dialog window allows to select which shading language to use when
// this example is executed.
//
///////////////////////////////////////////////////////////////////////

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoVertexProperty.h> 
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoQuadMesh.h> 
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoSwitch.h>  
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderParameter.h>

#include <Inventor/sensors/SoOneShotSensor.h> 
#include <Inventor/events/SoKeyboardEvent.h>
#include <DialogViz/SoDialogVizAll.h>

#define MAX_ANIM_STEP   10000
#define TITLE           "Animated Flag Shader"
#define DIALOGVIZ_FILE  "$OIVHOME/examples/source/Inventor/Features/Shaders/AnimatedFlag/AnimatedFlagDialog.iv"

// The default value must be set to the default selected language
// i.e. default DialogViz radio buttons selected item
SoShaderObject::SourceType   ShadingLanguageUsed = SoShaderObject::GLSL_PROGRAM;

Widget                MyWindow;

// Flag's properties
int                   NumPolyPerRowFlag = 300; // Initialization of its complexity
SbVec2f               DimensionFlag(9.0f,6.9f);  // Dimensions

SoSeparator          *VertexShaderSep;
SoSwitch             *InfoSwitch;
SoVertexProperty     *FlagVertexProperty;
SoQuadMesh           *FlagGeom;
SoShaderParameter1f  *AnimStep;                 // Shader's Parameter
SoTexture2           *LogoTexture;
SoMessageDialog      *ErrorDialog = NULL ;

/*---------------------------------------------------------------------------*/

void
configureShader()
{
  /*********************** BEGIN SHADER ****************************/

  // Initialize and set the vertex shader
  SoVertexShader * vertexShader = new SoVertexShader;
  
  // We set the shader source file, depending on which shading language has been chosen
  switch(ShadingLanguageUsed) {
  case SoShaderObject::GLSL_PROGRAM:
    LogoTexture->filename.setValue("$OIVHOME/examples/data/Inventor/Textures/VSGBannerGLSL.png");
    vertexShader->sourceProgram.setValue("$OIVHOME/examples/data/Inventor/Shaders/AnimatedFlagVtx.glsl");
    AnimStep->name  = "AnimStep";
    break;
  default:
    exit(EXIT_FAILURE);
  }
 
  // Set the parameter
  vertexShader->parameter.set1Value(0, AnimStep);

  // Initialize and set the shader program
  SoShaderProgram * shaderProgram = new SoShaderProgram;
  shaderProgram->shaderObject.setValue(vertexShader);

  // We must insert the shaderProgram before any shape to be drawn
  VertexShaderSep->insertChild(shaderProgram, 0);

  /*********************** END SHADER ******************************/
}/*---------------------------------------------------------------------------*/

void
fpsCB(float fps, void *, SoXtViewer *viewer) 
{
  SbString titleStr(TITLE);
  SbString fpsStr((int)fps);
  titleStr += " : ";
  titleStr += fpsStr;
  titleStr += " fps";

  titleStr += " / ";
  titleStr += (NumPolyPerRowFlag+1)*(NumPolyPerRowFlag+1)*2;
  titleStr += " triangles";
  
  viewer->setTitle(titleStr.getString());
}/*---------------------------------------------------------------------------*/

// Fill the vertex and texture coordinates fields of a SoVertexProperty 
void 
fillCoords(const SbVec2f &dimension, int numPolyPerRow,
           float repeatTex, SoVertexProperty * vertexProperty) 
{
  int f = 0;
  
  // Dimension of the SoQuadMesh using the SoVertexProperty  
  float lenght = dimension[0];
  float width  = dimension[1];
  
  float stepLength = lenght/numPolyPerRow;
  float stepWidth  = width/numPolyPerRow;
  
  int numVertexPerRow = numPolyPerRow+1;
  int size = numVertexPerRow*numVertexPerRow;
  
  // Set vertices' positions  
  SbVec3f *vertices = new SbVec3f[size];
  
  for (int i=0; i<numVertexPerRow;i++) {
    for (int j=0; j<numVertexPerRow;j++) {
      vertices[f++].setValue(j*stepLength, 8.1f+i*stepWidth, 0);
    }
  }
  
  vertexProperty->vertex.setValues(0,size, vertices);
  
  // Set vertices' texture coordinates
  SbVec2f tmpVec;
  SbVec2f *texCoord = new SbVec2f[size];
  
  for (int j=0; j<size;j++) {
    tmpVec[0]  = (vertices[j][0]/lenght)*repeatTex;
    tmpVec[1]  = -0.175f+(vertices[j][1]/width)*repeatTex;
    texCoord[j] = tmpVec;  
  }
  
  vertexProperty->texCoord.setValues(0, size, texCoord);
  
  delete [] vertices;
  delete [] texCoord;
}/*---------------------------------------------------------------------------*/

// Define the keys that allow users to interact with the application
void 
keyPressCB(void *, SoEventCallback * eventCB)
{
  const SoEvent * event = eventCB->getEvent();
  
  // Switch Help ON and OFF 
  if(SO_KEY_PRESS_EVENT(event, H)) {
    if(InfoSwitch->whichChild.getValue() == SO_SWITCH_ALL)
      InfoSwitch->whichChild = SO_SWITCH_NONE;
    else
      InfoSwitch->whichChild = SO_SWITCH_ALL;
  }
  
  // Increase the flag's complexity 
  if (SO_KEY_PRESS_EVENT(event, PAD_ADD )) {
    NumPolyPerRowFlag = NumPolyPerRowFlag+10;
    fillCoords(DimensionFlag, NumPolyPerRowFlag, 1.0, FlagVertexProperty);
    
    FlagGeom->verticesPerRow = NumPolyPerRowFlag+1;
    FlagGeom->verticesPerColumn = NumPolyPerRowFlag+1;
  }

  // Reduce the flag's complexity
  if (SO_KEY_PRESS_EVENT(event, PAD_SUBTRACT )) {
    if (NumPolyPerRowFlag > 10)
      NumPolyPerRowFlag = NumPolyPerRowFlag-10;
    fillCoords(DimensionFlag, NumPolyPerRowFlag, 1.0, FlagVertexProperty);
   
    FlagGeom->verticesPerRow = NumPolyPerRowFlag+1;
    FlagGeom->verticesPerColumn = NumPolyPerRowFlag+1;
  }  
}/*---------------------------------------------------------------------------*/

// Modify periodically the value of the AnimStep parameter
static void 
timerIncrease(void* /*data*/,SoSensor *sensor) 
{
  /*********************** BEGIN SHADER ****************************/
  if (AnimStep->value.getValue() < MAX_ANIM_STEP)
    AnimStep->value = AnimStep->value.getValue() + 0.3f;
  else 
    AnimStep->value = 0.0f;
  /*********************** END SHADER ******************************/
  
  sensor->schedule();
}/*---------------------------------------------------------------------------*/

// This function displays the Help Menu
SoSwitch * 
displayInfo() 
{
  // Create a SoSwitch to toggle the Help Menu on and off
  SoSwitch * infoSwitch = new SoSwitch;
  infoSwitch->ref();
  infoSwitch->whichChild = SO_SWITCH_ALL;
  
  SoSeparator * infoSep = new SoSeparator;
  
  SoPickStyle * pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;
  infoSep->addChild(pickStyle);
  
  infoSwitch->addChild(infoSep);
  
  // Light the text
  SoLightModel * lModel = new SoLightModel;
  lModel->model = SoLightModel::BASE_COLOR;
  infoSep->addChild(lModel);
  
  // Set the text's font
  SoFont * fontInfo = new SoFont;
  fontInfo->name = "Courier New";
  fontInfo->size = 12;
  infoSep->addChild(fontInfo);
  
  // Set the text's color
  SoBaseColor * infoColor = new SoBaseColor;
  infoColor->rgb.setValue(SbColor(1.0f, 0.0f, 0.0f));
  infoSep->addChild(infoColor);
  
  // Position the text
  SoTranslation * transInfo = new SoTranslation;
  transInfo->translation.setValue(-0.95f, 0.95f, 0.0f);
  infoSep->addChild(transInfo);
  
  // Set the text
  SoText2 *infoText = new SoText2;
  infoText->string.set1Value(0, "H  : Toggle this display");
  infoText->string.set1Value(1, "+/-: Increase/Decrease the flag's complexity");
  infoSep->addChild(infoText);
  
  infoSwitch->unrefNoDelete();
  return infoSwitch;
}/*---------------------------------------------------------------------------*/

// Main function
int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt
  MyWindow = SoXt::init(argv[0]);
  SoDialogViz::init();

  bool fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);

  SoSeparator * root = new SoSeparator;
  root->ref();
  
  // Display Info
  InfoSwitch = displayInfo();
  
  SoPerspectiveCamera * camera = new SoPerspectiveCamera;
    
  // Create the flag 
  SoShapeHints * hints = new SoShapeHints;
  hints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  
  // Fill the flag's vertexProperty
  FlagVertexProperty = new SoVertexProperty;
  FlagVertexProperty->normalBinding = SoVertexProperty::OVERALL;
  FlagVertexProperty->materialBinding = SoVertexProperty::OVERALL;
  fillCoords(DimensionFlag, NumPolyPerRowFlag, 1.0f, FlagVertexProperty);
  
  // Put the flag's geometry in a Separator
  SoSeparator *flagGeomSep= new SoSeparator; 
  
  FlagGeom = new SoQuadMesh;
  FlagGeom->verticesPerRow = NumPolyPerRowFlag+1;
  FlagGeom->verticesPerColumn = NumPolyPerRowFlag+1;
  FlagGeom->vertexProperty.setValue(FlagVertexProperty);
  
  LogoTexture = new SoTexture2;
  LogoTexture->model = SoTexture2::DECAL;

  flagGeomSep->addChild(LogoTexture);
  flagGeomSep->addChild(FlagGeom);
  
  // Create the pole
  SoMaterial *poleMaterial = new SoMaterial();
  poleMaterial->diffuseColor.setValue(1.0f, 1.0f, 0.0f);
  
  SoTransform *poleTransf = new SoTransform();
  poleTransf->scaleFactor.setValue(0.20f, 9.0f, 0.20f);
  poleTransf->translation.setValue(-0.20f, 6.2f, 0.0f);
  
  SoSeparator *poleSep = new SoSeparator();
  
  SoCylinder *pole = new SoCylinder();
  
  poleSep->addChild(poleMaterial);
  poleSep->addChild(poleTransf);
  poleSep->addChild(pole);
  
  /*********************** BEGIN SHADER ****************************/
  // Create a separator for the shader  
  VertexShaderSep = new SoSeparator;
  AnimStep = new SoShaderParameter1f;
  AnimStep->ref();
  AnimStep->value = 0.0f;

  // Set the shader program
  configureShader();
  /************************ END SHADER *****************************/
  
  // Add the flag's geometry to the shader's separator to apply the shader to it 
  VertexShaderSep->addChild(flagGeomSep);
  
  // Create a separator for the different parts of the flag  
  SoSeparator * flagSep = new SoSeparator();
  
  // Add the flag's geometry animated by the shader
  // and add the pole 
  flagSep->addChild(VertexShaderSep);
  flagSep->addChild(poleSep); 
  
  // Track the keyboard events
  SoEventCallback * eventCB = new SoEventCallback;
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), keyPressCB, NULL);
    
  // Create an invisible cube englobing the entire flag to avoid clipping effect.
  SoSeparator * noClippingSep = new SoSeparator; 
  
  SoCube *eliminateClipping = new SoCube;
  eliminateClipping->height.setValue(20);
  eliminateClipping->width.setValue(20);
  eliminateClipping->depth.setValue(20);
  
  SoTranslation *noClippingTrans = new SoTranslation;
  noClippingTrans->translation.setValue(0.0f, 6.0f, 0.0f);
  
  SoDrawStyle *invisible =  new SoDrawStyle;
  invisible->style.setValue(SoDrawStyle::INVISIBLE);
  
  noClippingSep->addChild(invisible);
  noClippingSep->addChild(noClippingTrans);
  noClippingSep->addChild(eliminateClipping);
  
  // Build the scene graph
  root->addChild(InfoSwitch);
  root->addChild(eventCB); 
  root->addChild(camera);
  root->addChild(hints);
  root->addChild(flagSep);
  root->addChild(noClippingSep);
  
  // Create a viewer
  SoXtExaminerViewer * eViewer = new SoXtExaminerViewer(MyWindow);
  camera->viewAll(root,eViewer->getViewportRegion());
  
  // Attach and show viewer
  eViewer->setSceneGraph(root);
  eViewer->setTitle(TITLE);
  eViewer->setSize(SbVec2s(500, 500));
  if (fullDemo) eViewer->setFramesPerSecondCallback(fpsCB);
  eViewer->show();

  // Schedule the animation.
  SoOneShotSensor *tictac = new SoOneShotSensor(timerIncrease, NULL);
  if (fullDemo) tictac->schedule();

  // Show the viewer
  SoXt::show(MyWindow);

  // Loop forever
  SoXt::mainLoop();
  
  if(ErrorDialog)
    ErrorDialog->destroy() ;

  root->unref();
  delete eViewer;
  delete tictac;

  SoXt::finish();

  return EXIT_SUCCESS;
}/*---------------------------------------------------------------------------*/


