/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
///////////////////////////////////////////////////////////////////////
//
// This example shows you how to code with a vertex and a geometry
// shader the display of a 3D point field bars.
//
// Source code involved with shaders is framed with
// BEGIN SHADER and END SHADER.
//
///////////////////////////////////////////////////////////////////////

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoNormal.h>
#include <Inventor/nodes/SoNormalBinding.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoGeometryShader.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderParameter.h>
#include <Inventor/nodes/SoIndexedTriangleStripSet.h>

#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/sensors/SoOneShotSensor.h>

#include <Inventor/SoWinApp.h>


#include <DialogViz/SoDialogVizAll.h>

#define TITLE "Geometry Shaders"

#define SIZE_WC 1.0f
#define DEFAULT_HISTO_COMPLEXITY 4

#ifdef WIN32
#define sprintf sprintf_s
#endif

enum InforStrID
{
  TOGGLE_DISPLAY_ID=0,
  COMPLEXITY_ID,
  TOGGLE_ANIM_ID,
  TOGGLE_SHADER_ID
};

#define HISTO_COMPLEXITY_STR     "+/-: Increase/Decrease the histogram's complexity"
#define GEOM_SHADER_ON_STR       "G  : Toggle geometry shader (On)"
#define GEOM_SHADER_OFF_STR      "G  : Toggle geometry shader (Off)"
#define ANIM_HISTO_ON_STR        "A  : Toggle histogram animation (On)"
#define ANIM_HISTO_OFF_STR       "A  : Toggle histogram animation (Off)"

SoPointSet          *HistoGeomShaderShape;
SoCoordinate3       *HistoGeomShaderCoord;
SoNormal            *HistoGeomShaderNormal;
SoShaderParameter1f *HistoGeomShaderWidth;

SoNormal                   *HistoGeomCPUNormal;
SoIndexedTriangleStripSet  *HistoGeomCPUShape;
SoCoordinate3              *HistoGeomCPUCoord;

SoCube              *SceneBBox;
SoSwitch            *InfoSwitch;
SoSwitch            *GeomShaderOrCPUSwitch;
int                  HistoComplexity=DEFAULT_HISTO_COMPLEXITY;
bool                 IsGeometryShader=true;
bool                 IsAnimation=false;
SoText2             *InfoText;
SoOneShotSensor     *AnimationSensor;

/*---------------------------------------------------------------------------*/

void
fpsCB(float fps, void *, SoXtViewer *viewer)
{
  SbString titleStr( TITLE );
  SbString fpsStr( (int)fps );
  titleStr += " : ";
  titleStr += fpsStr;
  titleStr += " fps";

  viewer->setTitle( titleStr.getString() );
}/*---------------------------------------------------------------------------*/

void
updateComplexityInfoStr()
{
  char str[256];
  sprintf( str, "%s (%d triangles)", HISTO_COMPLEXITY_STR, HistoComplexity*HistoComplexity*12 );
  InfoText->string.set1Value( COMPLEXITY_ID, str);

}/*---------------------------------------------------------------------------*/

// This function displays the Help Menu
SoSwitch *
displayInfo()
{
  // Create a SoSwitch to toggle the Help Menu on and off
  SoSwitch * infoSwitch = new SoSwitch;
  infoSwitch->ref();
  infoSwitch->whichChild = SO_SWITCH_ALL;

  SoSeparator * infoSep = new SoSeparator;
  infoSep->boundingBoxIgnoring = true;

  SoPickStyle * pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;
  infoSep->addChild( pickStyle );

  infoSwitch->addChild( infoSep );

  // Light the text
  SoLightModel * lModel = new SoLightModel;
  lModel->model = SoLightModel::BASE_COLOR;
  infoSep->addChild( lModel );

  // Set the text's font
  SoFont * fontInfo = new SoFont;
  fontInfo->name = "Courier New";
  fontInfo->size = 12;
  infoSep->addChild( fontInfo );

  // Set the text's color
  SoBaseColor * infoColor = new SoBaseColor;
  infoColor->rgb.setValue( SbColor( 1.0f, 0.0f, 0.0f ) );
  infoSep->addChild( infoColor );

  // Position the text
  SoTranslation * transInfo = new SoTranslation;
  transInfo->translation.setValue( -0.95f, 0.95f, 0.0f );
  infoSep->addChild( transInfo );

  // Set the text
  InfoText = new SoText2;

  InfoText->string.set1Value( TOGGLE_DISPLAY_ID,  "H  : Toggle this display" );
  InfoText->string.set1Value( TOGGLE_ANIM_ID,     ANIM_HISTO_OFF_STR );
  InfoText->string.set1Value( TOGGLE_SHADER_ID,   GEOM_SHADER_ON_STR );
  infoSep->addChild(InfoText);

  updateComplexityInfoStr();

  infoSwitch->unrefNoDelete();
  return infoSwitch;
}/*---------------------------------------------------------------------------*/

void
generateHistogramsShader( int gridSize )
{
  float pointSpace = SIZE_WC / ( gridSize-1 );

  HistoGeomShaderWidth->value = pointSpace * 0.2f;
  HistoGeomShaderShape->numPoints = gridSize * gridSize;
  HistoGeomShaderCoord->point.setNum( gridSize * gridSize );
  HistoGeomShaderNormal->vector.setNum(gridSize * gridSize);

  SbVec3f *coord = HistoGeomShaderCoord->point.startEditing();

  float yCoord = 0.0 ;

  for ( int i=0; i < gridSize; i++ )
  {
    float xCoord = 0.0 ;
    for ( int j=0; j < gridSize; j++ )
    {
      coord->setValue( xCoord, yCoord, (rand() % 100 + 1) / 100.0f );
      coord++;
      xCoord += pointSpace;
    }
    yCoord += pointSpace;
  }

  HistoGeomShaderCoord->point.finishEditing();
}/*---------------------------------------------------------------------------*/

void
generateHistogramsCPU( int gridSize )
{
  float pointSpace = SIZE_WC / ( gridSize-1 );

  float BarWidth = pointSpace * 0.2f;
  HistoGeomCPUCoord->point.setNum( gridSize * gridSize * 8 );       // 8 points per bar (6*4)
  HistoGeomCPUShape->normalIndex.setNum( gridSize * gridSize * 6 ); // 6 normals per bar
  HistoGeomCPUShape->coordIndex.setNum( gridSize * gridSize * 30 ); // 6 strips with 5 indices per bar
  HistoGeomCPUNormal->vector.setNum( 6 );                           // 6 normals per bar

  HistoGeomCPUNormal->vector.set1Value( 0, SbVec3f( 0, -1, 0 ) );
  HistoGeomCPUNormal->vector.set1Value( 1, SbVec3f( 1, 0, 0 )  );
  HistoGeomCPUNormal->vector.set1Value( 2, SbVec3f( 0, 1, 0 )  );
  HistoGeomCPUNormal->vector.set1Value( 3, SbVec3f( -1, 0, 0 ) );
  HistoGeomCPUNormal->vector.set1Value( 4, SbVec3f( 0, 0, 1 )  );
  HistoGeomCPUNormal->vector.set1Value( 5, SbVec3f( 0, 0, -1 ) );

  SbVec3f *coord = HistoGeomCPUCoord->point.startEditing();
  int32_t *normalIndex = HistoGeomCPUShape->normalIndex.startEditing();
  int32_t *coordIndex = HistoGeomCPUShape->coordIndex.startEditing();

  SbVec3f pts( 0, 0, 0 ) ;
  int32_t curCoordInd = 0;

  for ( int i=0; i < gridSize; i++ )
  {
    pts[0] = 0 ;
    for ( int j=0; j < gridSize; j++ )
    {
      pts[2] = (rand() % 100 + 1) / 100.0f;

      // Normals
      normalIndex[0] = 0;
      normalIndex[1] = 1;
      normalIndex[2] = 2;
      normalIndex[3] = 3;
      normalIndex[4] = 4;
      normalIndex[5] = 5;
      normalIndex += 6 ;

      // The eight vertices
      coord[0].setValue( pts[0] - BarWidth, pts[1] - BarWidth, 0 );
      coord[1].setValue( pts[0] - BarWidth, pts[1] - BarWidth, pts[2] );
      coord[2].setValue( pts[0] + BarWidth, pts[1] - BarWidth, 0 );
      coord[3].setValue( pts[0] + BarWidth, pts[1] - BarWidth, pts[2] );
      coord[4].setValue( pts[0] + BarWidth, pts[1] + BarWidth, 0 );
      coord[5].setValue( pts[0] + BarWidth, pts[1] + BarWidth, pts[2] );
      coord[6].setValue( pts[0] - BarWidth, pts[1] + BarWidth, 0 );
      coord[7].setValue( pts[0] - BarWidth, pts[1] + BarWidth, pts[2] );
      coord +=8;

      // 1st face
      coordIndex[0] = curCoordInd;
      coordIndex[1] = curCoordInd+1;
      coordIndex[2] = curCoordInd+2;
      coordIndex[3] = curCoordInd+3;
      coordIndex[4] = -1;
      coordIndex += 5;

      // 2nd face
      coordIndex[0] = curCoordInd+2;
      coordIndex[1] = curCoordInd+3;
      coordIndex[2] = curCoordInd+4;
      coordIndex[3] = curCoordInd+5;
      coordIndex[4] = -1;
      coordIndex += 5;

      // 3rd face
      coordIndex[0] = curCoordInd+4;
      coordIndex[1] = curCoordInd+5;
      coordIndex[2] = curCoordInd+6;
      coordIndex[3] = curCoordInd+7;
      coordIndex[4] = -1;
      coordIndex += 5;

      // 4th face
      coordIndex[0] = curCoordInd+7;
      coordIndex[1] = curCoordInd+6;
      coordIndex[2] = curCoordInd+1;
      coordIndex[3] = curCoordInd;
      coordIndex[4] = -1;
      coordIndex += 5;

      // 5th face
      coordIndex[0] = curCoordInd+1;
      coordIndex[1] = curCoordInd+3;
      coordIndex[2] = curCoordInd+7;
      coordIndex[3] = curCoordInd+5;
      coordIndex[4] = -1;
      coordIndex += 5;

      // 6th face
      coordIndex[0] = curCoordInd;
      coordIndex[1] = curCoordInd+2;
      coordIndex[2] = curCoordInd+6;
      coordIndex[3] = curCoordInd+4;
      coordIndex[4] = -1;
      coordIndex += 5;

      curCoordInd += 8;
      pts[0] += pointSpace;
    }
    pts[1] += pointSpace;
  }

  HistoGeomCPUCoord->point.finishEditing();
  HistoGeomCPUShape->normalIndex.finishEditing();
  HistoGeomCPUShape->coordIndex.finishEditing();

}/*---------------------------------------------------------------------------*/

void
generateHistogramsGeom( int gridSize  )
{
  SceneBBox->width  = 1 +  2 * SIZE_WC / ( gridSize-1 ) * 0.2f;
  SceneBBox->height = SceneBBox->width.getValue() ;

  if ( IsGeometryShader )
    generateHistogramsShader( HistoComplexity );
  else
    generateHistogramsCPU( HistoComplexity );
}/*---------------------------------------------------------------------------*/

// Define the keys that allow users to interact with the application
void
keyPressCB( void *, SoEventCallback * eventCB )
{
  const SoEvent * event = eventCB->getEvent();

  // Switch Help ON and OFF
  if ( SO_KEY_PRESS_EVENT( event, H ) )
  {
    if ( InfoSwitch->whichChild.getValue() == SO_SWITCH_ALL )
      InfoSwitch->whichChild = SO_SWITCH_NONE;
    else
      InfoSwitch->whichChild = SO_SWITCH_ALL;
  }

  // Increase the flag's complexity
  else if ( SO_KEY_PRESS_EVENT(event, PAD_ADD ) )
  {
     HistoComplexity *= 2;
     generateHistogramsGeom( HistoComplexity );
     updateComplexityInfoStr();
  }

  // Reduce the flag's complexity
  else if ( SO_KEY_PRESS_EVENT(event, PAD_SUBTRACT ) )
  {
    if (HistoComplexity > 2)
      HistoComplexity /= 2 ;

     generateHistogramsGeom( HistoComplexity );
     updateComplexityInfoStr();
  }

  else if ( SO_KEY_PRESS_EVENT(event, A ) )
  {
    IsAnimation = !IsAnimation;
    if ( IsAnimation )
    {
      InfoText->string.set1Value( TOGGLE_ANIM_ID, ANIM_HISTO_ON_STR );
      AnimationSensor->schedule();
    }
    else
    {
      InfoText->string.set1Value( TOGGLE_ANIM_ID, ANIM_HISTO_OFF_STR );
      AnimationSensor->unschedule();
    }

  }

  else if ( SO_KEY_PRESS_EVENT(event, G ) )
  {
    IsGeometryShader = !IsGeometryShader;
    generateHistogramsGeom( HistoComplexity );
    if ( IsGeometryShader )
    {
      InfoText->string.set1Value( TOGGLE_SHADER_ID, GEOM_SHADER_ON_STR );
      GeomShaderOrCPUSwitch->whichChild = 0;
    }
    else
    {
      InfoText->string.set1Value( TOGGLE_SHADER_ID, GEOM_SHADER_OFF_STR );
      GeomShaderOrCPUSwitch->whichChild = 1;
    }
  }
}/*---------------------------------------------------------------------------*/

static void
playAnimStep( void* /*data*/, SoSensor *sensor )
{
  generateHistogramsGeom( HistoComplexity );
  sensor->schedule();
}/*---------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init( argv[0] );

  bool fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);
#ifdef __linux
  if (!fullDemo) srand(32768);
#endif

  SoDialogViz::init();

  InfoSwitch = displayInfo();

  // Track the keyboard events
  SoEventCallback * eventCB = new SoEventCallback;
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), keyPressCB, NULL);

  AnimationSensor = new SoOneShotSensor( playAnimStep, NULL );

  SoPerspectiveCamera * camera = new SoPerspectiveCamera;

  SoSeparator *root = new SoSeparator;

  SoMessageDialog *errorDialog = NULL ;

  if( !SoGeometryShader::isSupported(SoShaderObject::GLSL_PROGRAM) )
  {
    errorDialog = new SoMessageDialog ;
    errorDialog->title = "GeometryShader Warning";
    errorDialog->type = SoMessageDialog::MD_WARNING;
    errorDialog->label = "Geometry Shader is not supported by your graphic board !";
    errorDialog->show() ;
  }

  // Useful in order Inventor computes a correct scene bounding box to
  // adjust the camera clipping planes.
  SoSeparator *SceneBBoxSep = new SoSeparator;
  SoDrawStyle *ds = new SoDrawStyle;
  ds->style = SoDrawStyle::INVISIBLE;
  SceneBBox = new SoCube ;
  SceneBBox->depth  = 1;

  SoTranslation *translation = new SoTranslation;
  translation->translation.setValue( 0.5f, 0.5f, 0.5f );

  SceneBBoxSep->addChild( ds );
  SceneBBoxSep->addChild( translation );
  SceneBBoxSep->addChild( SceneBBox );

  /*********************** BEGIN SHADER ****************************/
  /************* The geometry is generated onto the GPU ************/

  // Initialize the vertex and geometry shader
  SoVertexShader *vertexShader = new SoVertexShader;
  SoGeometryShader *geometryShader = new SoGeometryShader;
  SoFragmentShader *fragmentShader = new SoFragmentShader;

  vertexShader->sourceProgram.setValue( "$OIVHOME/examples/data/Inventor/Shaders/histoShaderVtx.glsl" );
  geometryShader->sourceProgram.setValue( "$OIVHOME/examples/data/Inventor/Shaders/histoShaderGeom.glsl" );
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/data/Inventor/Shaders/histoShaderFrag.glsl" );

  // Initialize and set the shader program
  SoShaderProgram *shaderProgram = new SoShaderProgram;
  shaderProgram->shaderObject.set1Value( 0, vertexShader );
  shaderProgram->shaderObject.set1Value( 1, geometryShader );
  shaderProgram->shaderObject.set1Value( 2, fragmentShader );

  // Uniform parameter to define the bar width
  HistoGeomShaderWidth = new SoShaderParameter1f;
  HistoGeomShaderWidth->name = "BarWidth";
  geometryShader->parameter.setValue( HistoGeomShaderWidth );

  // Input geometry shader primitive (SoPointSet)
  HistoGeomShaderShape = new SoPointSet;
  HistoGeomShaderCoord = new SoCoordinate3;
  // If no normals are specified points will be rendered without lighting
  // we don't care about the values specified because the real ones are computed
  // in the shader
  HistoGeomShaderNormal = new SoNormal;
  HistoGeomShaderNormal->vector.setNum(16);

  SoSeparator *geomShaderSep = new SoSeparator;
  geomShaderSep->boundingBoxIgnoring = true ;
  geomShaderSep->addChild( shaderProgram );
  geomShaderSep->addChild( HistoGeomShaderCoord );
  geomShaderSep->addChild( HistoGeomShaderNormal );
  geomShaderSep->addChild( HistoGeomShaderShape );

  /************************ END SHADER *****************************/

  // Geometry on the CPU
  HistoGeomCPUShape = new SoIndexedTriangleStripSet;
  HistoGeomCPUCoord = new SoCoordinate3;

  SoNormalBinding *normalBinding = new SoNormalBinding;
  normalBinding->value = SoNormalBinding::PER_PART_INDEXED;
  HistoGeomCPUNormal = new SoNormal;

  SoSeparator *geomCPUSep = new SoSeparator;
  geomCPUSep->boundingBoxIgnoring = true ;
  geomCPUSep->addChild( normalBinding );
  geomCPUSep->addChild( HistoGeomCPUNormal );
  geomCPUSep->addChild( HistoGeomCPUCoord );
  geomCPUSep->addChild( HistoGeomCPUShape );

  GeomShaderOrCPUSwitch = new SoSwitch;
  GeomShaderOrCPUSwitch->addChild( geomShaderSep );
  GeomShaderOrCPUSwitch->addChild( geomCPUSep );
  GeomShaderOrCPUSwitch->whichChild = 0; // Geometry shader by default.

  generateHistogramsGeom( HistoComplexity );

  // Build the scene graph
  root->ref();
  root->addChild( InfoSwitch );
  root->addChild( eventCB );
  root->addChild( camera );
  root->addChild( SceneBBoxSep );
  root->addChild( GeomShaderOrCPUSwitch );

  // Create a viewer
  // Attach and show viewer
  SoXtExaminerViewer *examinerViewer = new SoXtExaminerViewer( myWindow );
  examinerViewer->setSceneGraph( root );
  examinerViewer->setTitle( TITLE );
  examinerViewer->setSize( SbVec2s( 500, 500 ) );
  if (fullDemo) examinerViewer->setFramesPerSecondCallback( fpsCB );

  examinerViewer->setCamera( camera );
  examinerViewer->show();

  camera->viewAll( SceneBBoxSep, examinerViewer->getViewportRegion() );

  SoXt::show( myWindow );

  // Loop forever
  SoXt::mainLoop();

  root->unref();
  delete examinerViewer;

  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}/*----------------------------------------------------------------------------*/

