/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

///////////////////////////////////////////////////////////////////////
// 
// This example shows you how to code with a vertex and a fragment
// shader a per pixel lighting.
// The left sphere is lit with a shader program where as the right sphere
// is lit using the classical OpenGL PHONG lighting.
//
// Source code involved with shaders is framed with 
// BEGIN SHADER and END SHADER.
//
// A dialog window allows to select which shading language to use when
// this example is executed.
//
///////////////////////////////////////////////////////////////////////

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtMaterialEditor.h>

#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderParameter.h>

#include <Inventor/manips/SoPointLightManip.h> 
#include <Inventor/sensors/SoNodeSensor.h> 
#include <DialogViz/SoDialogVizAll.h>
#include <Inventor/helpers/SbFileHelper.h>


#define TITLE          "Per Pixel Lighing Shaders"

// The default value must be set to the default selected language
// i.e. default DialogViz radio buttons selected item
SoShaderObject::SourceType       ShadingLanguageUsed = SoShaderObject::GLSL_PROGRAM;

// This string will be set to the name of the chosen shading language.
// It will be displayed in the title of our viewer window.
SbString              TitleShadLang;

Widget               MyWindow;

SoSeparator         *ShaderSep;
SoTopLevelDialog    *LanguageChoiceDialogRoot;
SoXtMaterialEditor  *MatEditor;
SoPointLightManip   *PointLight;
SoShaderParameter3f *LightPosition;
SoVertexShader      *VertexShader;
SoFragmentShader    *FragmentShader;
SoXtExaminerViewer  *EViewer;
SoMessageDialog     *ErrorDialog = NULL ;


/*---------------------------------------------------------------------------*/

void
configureShaders()
{ 
  /*********************** BEGIN SHADER ****************************/
  // Initialize and set the vertex shader
  VertexShader = new SoVertexShader;
  
  // Initialize and set the fragment shader
  FragmentShader = new SoFragmentShader;
  // The parameter is not needed with GLSL. Its value can be retrieved
  // within GLSL thanks to its built-in variables.
  if(ShadingLanguageUsed != SoShaderObject::GLSL_PROGRAM)
    FragmentShader->parameter.set1Value(0, LightPosition);

  switch(ShadingLanguageUsed) {
  case SoShaderObject::GLSL_PROGRAM:
    // Specify the vertex and fragment programs
    VertexShader->sourceProgram.setValue(SbFileHelper::expandString("$OIVHOME/examples/data/Inventor/Shaders/PixelLightingVtx.glsl"));
    FragmentShader->sourceProgram.setValue(SbFileHelper::expandString("$OIVHOME/examples/data/Inventor/Shaders/PixelLightingFrag.glsl"));
    TitleShadLang = "GLSL";
    break;

  default:
    exit(EXIT_FAILURE);
  }

  // Initialize and set the shader program
  SoShaderProgram *shaderProgram = new SoShaderProgram;
  shaderProgram->shaderObject.set1Value(0, VertexShader);
  shaderProgram->shaderObject.set1Value(1, FragmentShader);
  
  // Add the shader program to the separator
  ShaderSep->insertChild(shaderProgram, 0);

  /*********************** END SHADER ****************************/
}/*---------------------------------------------------------------------------*/

void
fpsCB(float fps, void *, SoXtViewer *viewer) 
{
  SbString titleStr(TITLE);
  SbString fpsStr((int)fps);
  titleStr += " : ";
  titleStr += fpsStr;
  titleStr += " fps (with ";
  titleStr += TitleShadLang;
  titleStr += ")";
  viewer->setTitle(titleStr.getString());
}/*---------------------------------------------------------------------------*/

// Update the shader light position parameter in eye space.
void 
updateLightPosShaderParam()
{
  /*********************** BEGIN SHADER ****************************/
  // No parameter is used with GLSL. See above or in the shader code.
  if(ShadingLanguageUsed == SoShaderObject::GLSL_PROGRAM)
    return;
  /************************ END SHADER *****************************/

  SbVec3f lightPos;

  SbMatrix affine, proj;
  EViewer->getCamera()->getViewVolume().getMatrices(affine, proj);
  affine.multVecMatrix(PointLight->location.getValue(), lightPos);

  /*********************** BEGIN SHADER ****************************/
  LightPosition->value = lightPos;
  /************************ END SHADER *****************************/
}/*---------------------------------------------------------------------------*/

// Call at each camera motion
void
cameraSensorCB(void *, SoSensor *)
{
  updateLightPosShaderParam();
}/*---------------------------------------------------------------------------*/

// Call at each light motion
void
pointLightCB(void *, SoDragger *)
{
  updateLightPosShaderParam();
}/*---------------------------------------------------------------------------*/

// Auditor for the 'OK' button
class ConfirmLanguageChoiceAuditor : public SoDialogPushButtonAuditor {
  void dialogPushButton(SoDialogPushButton* /*cpt*/) 
  {
    // Check if supported
    if(!SoVertexShader::isSupported(ShadingLanguageUsed) ||
       !SoFragmentShader::isSupported(ShadingLanguageUsed)) {
      ErrorDialog = new SoMessageDialog ;
      ErrorDialog->title = "PixelLighting Warning";
      ErrorDialog->type = SoMessageDialog::MD_WARNING;

      if ( ShadingLanguageUsed == SoShaderObject::GLSL_PROGRAM )
      {
        ErrorDialog->label = "GLSL is not supported by your graphics board!";
      }

      ErrorDialog->show() ;
    }

    configureShaders();
    // Show the material editor and viewer windows
    MatEditor->show();
    SoXt::show(MyWindow);
    // Hide the DialogViz window
    LanguageChoiceDialogRoot->hide();
  }
};/*---------------------------------------------------------------------------*/

// Auditor for the 'Exit' button
class ExitApplicationAuditor : public SoDialogPushButtonAuditor {
  void dialogPushButton(SoDialogPushButton* /*cpt*/) 
  {
    exit(EXIT_SUCCESS);
  }
};/*---------------------------------------------------------------------------*/

// Auditor for the 'Select shading language to use' button
// Updates ShadingLanguageUsed's value
class SelectShadingLanguageAuditor : public SoDialogChoiceAuditor {
  void dialogChoice(SoDialogChoice * cpt) 
  {
    int32_t selectedShapeItem = cpt->selectedItem.getValue();
    switch(selectedShapeItem) {
    case 0:  ShadingLanguageUsed = SoShaderObject::GLSL_PROGRAM;    break;
    }
  }
};/*---------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt
  MyWindow = SoXt::init(argv[0]);
  SoDialogViz::init();

  bool fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);

  SoSeparator *root = new SoSeparator;
  root->ref();
  
  SoComplexity *complexity = new SoComplexity;
  complexity->value = 0.55f;

  // Point light
  PointLight = new SoPointLightManip;
  PointLight->location.setValue(2.0f, 2.0f, 6.0f);
  PointLight->getDragger()->addValueChangedCallback(pointLightCB, NULL);

  // Create a separator for the shaders  
  ShaderSep = new SoSeparator;
  
  /*********************** BEGIN SHADER ****************************/
  LightPosition = new SoShaderParameter3f;
  LightPosition->ref();
  /************************ END SHADER *****************************/
  
  // Create a separator for the sphere
  SoSeparator *sphereSep = new SoSeparator;
  sphereSep->addChild(new SoSphere);
  
  // Add the sphere to the shaders' separator to apply the shaders to it.
  ShaderSep->addChild(sphereSep);
    
  // Create a second sphere that will be lit by Inventor without using shader.
  SoSeparator *sphereSep2 = new SoSeparator;
  
  SoTranslation *sphereTrans = new SoTranslation;
  sphereTrans->translation.setValue(2.5f, 0.0f, 0.0f);
  
  sphereSep2->addChild(sphereTrans);
  sphereSep2->addChild(new SoSphere);

  // Material applied to the model
  SoMaterial *modelMat = new SoMaterial;
  modelMat->ambientColor.setValue(0.2f, 0.2f, 0.2f);
  modelMat->diffuseColor.setValue(0.3f, 0.3f, 0.9f);
  modelMat->specularColor.setValue(0.6f, 0.6f, 0.8f);
  modelMat->emissiveColor.setValue(0.0f, 0.0f, 0.0f);
  modelMat->shininess.setValue(0.2f);
  
  // Build the scene graph
  root->addChild(modelMat);
  root->addChild(complexity);
  root->addChild(PointLight);
  root->addChild(ShaderSep);
  root->addChild(sphereSep2);
  
  // Create a viewer
  EViewer = new SoXtExaminerViewer(MyWindow);
  
  // Attach and show viewer
  EViewer->setSceneGraph(root);
  EViewer->setTitle(TITLE);
  EViewer->setHeadlight(FALSE);
  EViewer->setSize(SbVec2s(500, 500));
  if (fullDemo) EViewer->setFramesPerSecondCallback(fpsCB);
  EViewer->show();

  // Track camera motion
  SoNodeSensor *cameraSensor = new SoNodeSensor(cameraSensorCB, NULL);
  cameraSensor->attach(EViewer->getCamera());

  // Update the light position
  pointLightCB(NULL, NULL);

  // Material editor
  MatEditor = new SoXtMaterialEditor;
  MatEditor->attach(modelMat);
    
  // DialogViz
  SbString DIALOGVIZ_FILE = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/Shaders/PixelLighting/PixelLightingDialog.iv");
  LanguageChoiceDialogRoot = (SoTopLevelDialog *)SoDialogViz::loadFromFile(DIALOGVIZ_FILE);
  LanguageChoiceDialogRoot->buildDialog(MyWindow, FALSE);
  SoDialogRadioButtons * langChoiceButton = (SoDialogRadioButtons *)LanguageChoiceDialogRoot->searchForAuditorId("LanguageChoiceButtons");
  SoDialogPushButton   * okButton         = (SoDialogPushButton *)LanguageChoiceDialogRoot->searchForAuditorId("OKButton");
  SoDialogPushButton   * exitButton       = (SoDialogPushButton *)LanguageChoiceDialogRoot->searchForAuditorId("ExitButton");
  okButton->addAuditor(new ConfirmLanguageChoiceAuditor());
  exitButton->addAuditor(new ExitApplicationAuditor());
  langChoiceButton->addAuditor(new SelectShadingLanguageAuditor());
  LanguageChoiceDialogRoot->show();

  // Loop forever
  SoXt::mainLoop();
  
  if(ErrorDialog)
    ErrorDialog->destroy() ;

  LanguageChoiceDialogRoot->close();
  root->unref();
  delete cameraSensor;
  delete MatEditor;
  delete EViewer;
  SoDialogViz::finish();
  SoXt::finish();

  return EXIT_SUCCESS;
}/*---------------------------------------------------------------------------*/


