/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (feb 2013)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoGeometryShader.h>
#include <Inventor/nodes/SoTessellationControlShader.h>
#include <Inventor/nodes/SoTessellationEvaluationShader.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoIndexedPointSet.h>
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/nodes/SoBufferedShape.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoSwitch.h>

//Include used for the GUI.
#include <DialogViz/dialog/SoDialogViz.h>
#include <DialogViz/dialog/SoDialogRealSlider.h>
#include <DialogViz/dialog/SoDialogRadioButtons.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>
#include <DialogViz/auditors/SoDialogAuditor.h>
#include <DialogViz/dialog/SoDialogCustom.h>

// Tessellation shaders parameters
SoShaderParameter1f* g_TessLevelInner = NULL;
SoShaderParameter1f* g_TessLevelOuter = NULL;
SoSwitch* g_shapeType = NULL;

const int faces[] = {
  2, 1, 0,
  3, 2, 0,
  4, 3, 0,
  5, 4, 0,
  1, 5, 0,
  11, 6,  7,
  11, 7,  8,
  11, 8,  9,
  11, 9,  10,
  11, 10, 6,
  1, 2, 6,
  2, 3, 7,
  3, 4, 8,
  4, 5, 9,
  5, 1, 10,
  2,  7, 6,
  3,  8, 7,
  4,  9, 8,
  5, 10, 9,
  1, 6, 10,
};


static const float vertices[12][3] =
{
 {0.000f,  0.000f,  1.000f},
 {0.894f,  0.000f,  0.447f},
 {0.276f,  0.851f,  0.447f},
 {-0.724f,  0.526f,  0.447f},
 {-0.724f, -0.526f,  0.447f},
 {0.276f, -0.851f,  0.447f},
 {0.724f,  0.526f, -0.447f},
 {-0.276f,  0.851f, -0.447f},
 {-0.894f,  0.000f, -0.447f},
 {-0.276f, -0.851f, -0.447f},
 {0.724f, -0.526f, -0.447f},
 {0.000f,  0.000f, -1.000f}
};

SoBufferedShape *
makeIcosahedronBufferedShape()
{
  SoBufferedShape *pShape = new SoBufferedShape;
  int facesSize = sizeof(faces)/sizeof(faces[0]);
  SoRef<SoCpuBufferObject> vertBuffer = new SoCpuBufferObject();
  vertBuffer->setSize( facesSize * 3 * sizeof(vertices[0][0]) );

  float* vert = (float*)vertBuffer->map(SoBufferObject::READ_ONLY);

  for (int i = 0; i < facesSize; ++i)
  {
    for (int j = 0; j < 3; ++j)
    {
      vert[3*i+j] = vertices[faces[i]][j];
    }
  }

  pShape->shapeType = SoBufferedShape::TRIANGLES;
  pShape->numVertices.set1Value( 0, facesSize );
  pShape->vertexBuffer.setValue( vertBuffer.ptr() );

  return pShape;
}

SoBufferedShape *
makeIcosahedronIndexedBufferedShape()
{
  SoBufferedShape *pShape = new SoBufferedShape;

  SoRef<SoCpuBufferObject> vertBuffer = new SoCpuBufferObject((void*)vertices, sizeof(vertices));
  SoRef<SoCpuBufferObject> indBuffer = new SoCpuBufferObject((void*)faces, sizeof(faces));

  pShape->shapeType = SoBufferedShape::TRIANGLES;
  pShape->numVertices.set1Value( 0, sizeof(faces)/sizeof(int) );
  pShape->vertexBuffer.setValue( vertBuffer.ptr() );
  pShape->indexBuffer.setValue( indBuffer.ptr() );

  return pShape;
}

SoIndexedPointSet *
makeIcosahedronIndexedPointSet()
{
  SoIndexedPointSet* pShape = new SoIndexedPointSet;
  SoVertexProperty* pVertexProperty = new SoVertexProperty;

  pVertexProperty->vertex.setValues(0, sizeof(vertices)/sizeof(vertices[0]), vertices);
  pShape->coordIndex.setValues(0, sizeof(faces)/sizeof(faces[0]), faces);

  pShape->vertexProperty = pVertexProperty;

  return pShape;
}

SoPointSet *
makeIcosahedronPointSet()
{
  SoPointSet* pShape = new SoPointSet;
  SoVertexProperty* pVertexProperty = new SoVertexProperty;

  std::vector<SbVec3f> v;
  for (size_t i = 0; i < sizeof(faces)/sizeof(faces[0]); ++i)
    v.push_back(SbVec3f(vertices[faces[i]]));

  pVertexProperty->vertex.setValues(0, int(v.size()), &*v.begin());
  pShape->vertexProperty = pVertexProperty;
  return pShape;
}


SoShaderProgram*
configureShaders()
{
  // Initialize and set the shader program
  SoShaderProgram *shaderProgram = new SoShaderProgram;
  shaderProgram->setVertexShader(0,"$OIVHOME/examples/source/Inventor/Features/Shaders/TessellationShader/TessShaderVtx.glsl");

  SoTessellationControlShader* TessellationControlShader = shaderProgram->setTessellationControlShader(1,"$OIVHOME/examples/source/Inventor/Features/Shaders/TessellationShader/TessShaderTessControl.glsl");
  shaderProgram->setTessellationEvaluationShader(2,"$OIVHOME/examples/source/Inventor/Features/Shaders/TessellationShader/TessShaderTessEvaluation.glsl");
  shaderProgram->setGeometryShader(3,"$OIVHOME/examples/source/Inventor/Features/Shaders/TessellationShader/TessShaderGeom.glsl");
  shaderProgram->setFragmentShader(4,"$OIVHOME/examples/source/Inventor/Features/Shaders/TessellationShader/TessShaderFrag.glsl");

  g_TessLevelOuter = TessellationControlShader->addShaderParameter<SoShaderParameter1f>("TessLevelOuter",1.0f);
  g_TessLevelInner = TessellationControlShader->addShaderParameter<SoShaderParameter1f>("TessLevelInner",1.0f);

  // Setup the patch length for tesselation.
  shaderProgram->patchLength = 3;

  return shaderProgram;
}

// Class used to interpret event from the GUI.
class auditorClass : public SoDialogAuditor
{
  // When slider value has changed.
  void dialogRealSlider(SoDialogRealSlider* cpt)
  {
    float value = cpt->value.getValue();
    if (cpt->auditorID.getValue() == "TessellationInner")
    {
      g_TessLevelInner->value.setValue(value);
    }
    else if (cpt->auditorID.getValue() == "TessellationOuter")
    {
      g_TessLevelOuter->value.setValue(value);
    }
  }

  void dialogRadioButtons(SoDialogRadioButtons* cpt)
  {
    if (cpt->auditorID.getValue() == "ShapeType")
    {
      g_shapeType->whichChild = cpt->selectedItem.getValue();
    }
  }
};

int
main(int /*argc*/, char **argv)
{
  Widget window = SoXt::init(argv[0]);

  // DialogViz functions used to build the GUI.
  ////////////////////////////////////////////
  SoDialogViz::init();
  SoTopLevelDialog* topLevelDialog = (SoTopLevelDialog*) SoDialogViz::loadFromFile("$OIVHOME/examples/source/Inventor/Features/Shaders/TessellationShader/interface.iv");
  topLevelDialog->buildDialog(window, TRUE);
  SoDialogCustom* customNode = (SoDialogCustom*)topLevelDialog->searchForAuditorId("viewer");
  Widget myWindow = customNode->getWidget();
  auditorClass* auditor = new auditorClass;
  topLevelDialog->label.setValue("Tessellation shaders");
  topLevelDialog->addAuditor(auditor);
  topLevelDialog->show();
  ////////////////////////////////////////////
  // End of DialogViz function

  SoSeparator *root = new SoSeparator;
  root->ref();

  SoSeparator* ShaderSep = new SoSeparator;
  root->addChild(ShaderSep);

  // Add shader program
  ShaderSep->addChild(configureShaders());

  // Add shape
  g_shapeType = new SoSwitch;
  g_shapeType->addChild(makeIcosahedronIndexedBufferedShape());
  g_shapeType->addChild(makeIcosahedronBufferedShape());
  g_shapeType->addChild(makeIcosahedronIndexedPointSet());
  g_shapeType->addChild(makeIcosahedronPointSet());

  g_shapeType->whichChild = 0;
  ShaderSep->addChild(g_shapeType);

  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setBackgroundColor(SbColor(.5f, .5f, .5f));
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Tessellation shader on Icosahedron");
  myViewer->show();
  myViewer->viewAll();

  SoXt::show(window);
  SoXt::mainLoop();

  delete auditor;
  root->unref();
  delete myViewer;
  SoDialogViz::finish();
  SoXt::finish();
  return EXIT_SUCCESS;
}
