/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

///////////////////////////////////////////////////////////////////////
// 
// This example shows you how to code with a vertex and a fragment
// shader a cartoon rendering.
//
// Source code involved with shaders is framed with 
// BEGIN SHADER and END SHADER.
//
// A dialog window allows to select which shading language to use when
// this example is executed.
//
///////////////////////////////////////////////////////////////////////

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtMaterialEditor.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoDirectionalLight.h> 
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderParameter.h>

#include <Inventor/sensors/SoNodeSensor.h> 
#include <Inventor/SoWinApp.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/helpers/SbFileHelper.h>

#define TITLE          "Toon Shading Shaders"

SbString dialogvizFile = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/Shaders/ToonShading/ToonShadingDialog.iv");

// The default value must be set to the default selected language
// i.e. default DialogViz radio buttons selected item
SoShaderObject::SourceType       ShadingLanguageUsed = SoShaderObject::GLSL_PROGRAM;

// This string will be set to the name of the chosen shading language.
// It will be displayed in the title of our viewer window.
SbString             TitleShadLang;

Widget               MyWindow;

SoSeparator         *ShaderSep;
SoTopLevelDialog    *LanguageChoiceDialogRoot;
SoXtExaminerViewer  *EViewer;
SoXtMaterialEditor  *MatEditor;
// Shader's Parameters
SoShaderParameter3f *LightDirection;
SoVertexShader      *VertexShader;
SoFragmentShader    *FragmentShader;
SoNodeSensor        *CameraSensor;
SoMessageDialog     *ErrorDialog = NULL ;

/*---------------------------------------------------------------------------*/

void
configureShaders()
{
  /*********************** BEGIN SHADER ****************************/
  // Initialize the vertex and fragment vertex shader
  VertexShader   = new SoVertexShader;
  FragmentShader = new SoFragmentShader;
 
  VertexShader->parameter.setValue(LightDirection);

  if ( ShadingLanguageUsed == SoShaderObject::GLSL_PROGRAM )
  {
    VertexShader->sourceProgram.setValue("$OIVHOME/examples/data/Inventor/Shaders/ToonShadingVtx.glsl");
    FragmentShader->sourceProgram.setValue("$OIVHOME/examples/data/Inventor/Shaders/ToonShadingFrag.glsl");
    // Sets the parameters names
    LightDirection->name = "lightDirection";
    TitleShadLang = "GLSL";
  }

  // Initialize and set the shader program
  SoShaderProgram *shaderProgram = new SoShaderProgram;  
  shaderProgram->shaderObject.set1Value(0, VertexShader);
  shaderProgram->shaderObject.set1Value(1, FragmentShader);
  
  // Add the shader program to the separator
  ShaderSep->insertChild(shaderProgram, 0);

  /************************ END SHADER *****************************/
}/*---------------------------------------------------------------------------*/

void
fpsCB(float fps, void *, SoXtViewer *viewer) 
{
  SbString titleStr(TITLE);
  SbString fpsStr((int)fps);
  titleStr += " : ";
  titleStr += fpsStr;
  titleStr += " fps (with ";
  titleStr += TitleShadLang;
  titleStr += ")";

  viewer->setTitle(titleStr.getString());
}/*---------------------------------------------------------------------------*/

void
cameraSensorCB(void *, SoSensor *)
{
  // Retrieve the headlight direction in model space.
  SbVec3f lightDir = EViewer->getHeadlight()->direction.getValue();

  // Transform the headlight direction in world space.
  const SbRotation &cameraRot = EViewer->getCamera()->orientation.getValue();
  cameraRot.multVec(lightDir, lightDir);

  // Transform the headlight direction in eye space (camera space)
  SbMatrix affine, proj;
  EViewer->getCamera()->getViewVolume().getMatrices(affine, proj);
  affine.multDirMatrix(lightDir, lightDir);

  /*********************** BEGIN SHADER ****************************/
  LightDirection->value = -lightDir;
  /************************ END SHADER *****************************/
}/*---------------------------------------------------------------------------*/

// Auditor for the 'OK' button
class ConfirmLanguageChoiceAuditor : public SoDialogPushButtonAuditor {
  void dialogPushButton(SoDialogPushButton* /*cpt*/) 
  {
    // Check if supported
    if(!SoVertexShader::isSupported(ShadingLanguageUsed) ||
       !SoFragmentShader::isSupported(ShadingLanguageUsed)) {
      ErrorDialog = new SoMessageDialog ;
      ErrorDialog->title = "ToonShading Warning";
      ErrorDialog->type = SoMessageDialog::MD_WARNING;
      
      if ( ShadingLanguageUsed == SoShaderObject::GLSL_PROGRAM )
      {
        ErrorDialog->label = "GLSL is not supported by your graphics board!";
      }
      
      ErrorDialog->show() ;
    }

    configureShaders();
    // Show the material editor and viewer windows
    SoXt::show(MyWindow);
    MatEditor->show();
    // Hide the DialogViz window
    LanguageChoiceDialogRoot->hide();
  }
};/*---------------------------------------------------------------------------*/

// Auditor for the 'Exit' button
class ExitApplicationAuditor : public SoDialogPushButtonAuditor {
  void dialogPushButton(SoDialogPushButton* /*cpt*/) 
  {
    exit(EXIT_SUCCESS);
  }
};/*---------------------------------------------------------------------------*/

// Auditor for the 'Select shading language to use' button
// Updates ShadingLanguageUsed's value
class SelectShadingLanguageAuditor : public SoDialogChoiceAuditor {
  void dialogChoice(SoDialogChoice * cpt) 
  {
    int32_t selectedShapeItem = cpt->selectedItem.getValue();
    switch(selectedShapeItem) {
    case 0:  ShadingLanguageUsed = SoShaderObject::GLSL_PROGRAM;    break;
    }
  }
};/*---------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
  // Initialize Inventor and Xt
  MyWindow = SoXt::init(argv[0]);
  SoDialogViz::init();

  bool fullDemo = !(argc >= 2 && argv[1] && strcmp(argv[1], "-noanim") == 0);

  SoSeparator *root = new SoSeparator;
  root->ref();
  
  SoComplexity *complexity = new SoComplexity;
  complexity->textureQuality = 0.1f; // We force Inventor to use nearest texture filtering
  
  // Create the toon texture
  SoTexture2 *toonTexture = new SoTexture2;
  toonTexture->filename.setValue(SbFileHelper::expandString( "$OIVHOME/examples/data/Inventor/Textures/ToonTex.png"));

  // Import an Inventor model
  SoInput myInput;
	myInput.openFile("$OIVHOME/examples/data/Inventor/Models/canstick.iv");
 
  SoSeparator *model = SoDB::readAll(&myInput);

  // Apply the toon texture to the model
  model->insertChild(toonTexture, 0);

  /*********************** BEGIN SHADER ****************************/
  // Create a separator for the shaders  
  ShaderSep = new SoSeparator;
  LightDirection = new SoShaderParameter3f;
  LightDirection->ref();
  /************************ END SHADER *****************************/
   
  // Material applied to the model
  SoMaterial *modelMat = new SoMaterial;
  modelMat->shininess = 0.1f;
  modelMat->diffuseColor.setValue(0.3f, 0.3f, 0.9f);
  modelMat->specularColor.setValue(1.0f, 1.0f, 1.0f);

  // Add the model to the shaders' separator to apply the shaders to it
  ShaderSep->addChild(modelMat);
  ShaderSep->addChild(model);
  
  // Build the scene graph
  root->addChild(complexity);
  root->addChild(ShaderSep);
  
  // Create a viewer
  // Attach and show viewer
  EViewer = new SoXtExaminerViewer(MyWindow);
  EViewer->setSceneGraph(root);
  EViewer->setTitle(TITLE);
  EViewer->setHeadlight(FALSE);
  EViewer->setSize(SbVec2s(500, 500));
  if (fullDemo) EViewer->setFramesPerSecondCallback(fpsCB);
  EViewer->show();

  // Material editor
  MatEditor = new SoXtMaterialEditor;
  MatEditor->attach(modelMat);

  // Track camera motion
  CameraSensor = new SoNodeSensor(cameraSensorCB, NULL);
  CameraSensor->attach(EViewer->getCamera());
  
  // DialogViz
  LanguageChoiceDialogRoot = (SoTopLevelDialog *)SoDialogViz::loadFromFile(dialogvizFile);
  LanguageChoiceDialogRoot->buildDialog(MyWindow, FALSE);
  LanguageChoiceDialogRoot->show();
  SoDialogRadioButtons * langChoiceButton = (SoDialogRadioButtons *)LanguageChoiceDialogRoot->searchForAuditorId("LanguageChoiceButtons");
  SoDialogPushButton   * okButton         = (SoDialogPushButton *)LanguageChoiceDialogRoot->searchForAuditorId("OKButton");
  SoDialogPushButton   * exitButton       = (SoDialogPushButton *)LanguageChoiceDialogRoot->searchForAuditorId("ExitButton");
  ConfirmLanguageChoiceAuditor* _okButtonAud = new ConfirmLanguageChoiceAuditor();
  okButton->addAuditor(_okButtonAud);
  ExitApplicationAuditor* _exitButtonAud = new ExitApplicationAuditor();
  exitButton->addAuditor(_exitButtonAud);
  SelectShadingLanguageAuditor* _langChoiceButtonAud = new SelectShadingLanguageAuditor();
  langChoiceButton->addAuditor(_langChoiceButtonAud);

  // Loop forever
  SoXt::mainLoop();

  if(ErrorDialog)
    ErrorDialog->destroy() ;

  delete _okButtonAud;
  delete _exitButtonAud;
  delete _langChoiceButtonAud;
  LanguageChoiceDialogRoot->close();
  root->unref();
  LightDirection->unref();
  delete MatEditor;
  delete CameraSensor;
  delete EViewer;
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}/*----------------------------------------------------------------------------*/

