/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : D. DALLA-ROSA (April 2011)
**=======================================================================*/

#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/image/SoHDRImageRW.h>
#include <Inventor/image/SoJPEGImageRW.h>
#include <Inventor/image/SoPNGImageRW.h>
#include <Inventor/image/SoRasterImageFile.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoRenderToTarget.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <DialogViz/SoDialogVizAll.h>

SoSwitch* g_textureSwitch = NULL;
SoRenderToTarget* g_renderToTexture = NULL;
SoTopLevelDialog* g_topLevelDialog = NULL;
SoXtExaminerViewer* g_mainViewer = NULL;

//#define USE_FLOAT_COLORBUFFER

#define DEFAULT_SIZE 128


class MyAuditorClass : public SoDialogAuditor
{
  void dialogRealSlider( SoDialogRealSlider* slider )
  {
    if (slider->auditorID.getValue() == SbString("antialiasing"))
    {
      g_renderToTexture->antialiasingQuality = slider->value.getValue();
    }
  }

  void dialogIntegerSlider( SoDialogIntegerSlider* slider )
  {
    if (slider->auditorID.getValue() == SbString("viewportSize"))
    {
      int value = slider->value.getValue();
      // When the user set 0 we use the viewer's viewport, 0 doesn't make sense.
      if (value == 0)
        value = -1;
      g_renderToTexture->size = SbVec2i32(value, value);
    }
  }
  
  void dialogComboBox( SoDialogComboBox* box )
  {
    if (box->auditorID.getValue() == SbString("texture"))
    {
      g_textureSwitch->whichChild = box->selectedItem.getValue();
    }
  }

  void dialogPushButton( SoDialogPushButton* button )
  {
    if (button->auditorID.getValue() == SbString("saveButton"))
    {
      SoDialogComboBox* cb = (SoDialogComboBox*)g_topLevelDialog->searchForAuditorId("outputType");

      if (cb)
      {
        SbString format = cb->items[cb->selectedItem.getValue()];
        int idx = g_textureSwitch->whichChild.getValue();
        if (idx >= 0)
        {
          // Get the currently displayed texture to save
          SoNode* node = g_textureSwitch->getChild(idx);
          if (node->isOfType(SoTexture2::getClassTypeId()))
          {
            SoTexture2* texture = (SoTexture2*)node;
            bool isHdri = false;
            // We don't convert to unsigned byte if we want HDR output.
            if (format == "HDR")
              isHdri = true;

            // We enable the readback mode only when we want to save the result.
            // Readback can cost a lot of perfs.
            g_renderToTexture->modes.set1Value(SoRenderToTarget::COLOR0, SoRenderToTarget::TARGET_COPY);
            g_renderToTexture->modes.set1Value(SoRenderToTarget::DEPTH, SoRenderToTarget::TARGET_COPY);
            g_renderToTexture->enableFragmentsQuery = TRUE;

            g_mainViewer->render();

            // Restore the default mode.
            g_renderToTexture->modes.set1Value(SoRenderToTarget::COLOR0, SoRenderToTarget::AUTO);
            g_renderToTexture->modes.set1Value(SoRenderToTarget::DEPTH, SoRenderToTarget::AUTO);

            g_renderToTexture->enableFragmentsQuery = FALSE;


            SbRasterImage* rasterImage = texture->image.toRasterImage(!isHdri);

            SbString filename = "output." + format;

            SoRasterImageRW* imageRW = NULL;
            SoRasterImageIO* imageIO = new SoRasterImageFile(filename);

            if (format == "HDR")
            {
              // HDR support only FLOAT input, there is no reason
              // to write unsigned bytes data to HDR format.
              if (rasterImage->getDataType() != SbDataType::FLOAT)
              {
                SoDebugError::post("MyAuditorClass::dialogPushButton", 
                  "HDR output is not available if the color buffer is not a FLOAT buffer");
                delete rasterImage;
                delete imageIO;
                return;
              }

              imageRW = new SoHDRImageRW;
              // HDR does not support alpha
              if ((rasterImage->getComponents() == SbRasterImage::LUMINANCE_TRANSPARENCY) ||
                  (rasterImage->getComponents() == SbRasterImage::RGB_TRANSPARENCY))
                rasterImage->removeAlphaChannel();
            }
            else
              if (format == "PNG")
                imageRW = new SoPNGImageRW;
              else
                if (format == "JPEG")
                {
                  imageRW = new SoJPEGImageRW;
                  // JPEG does not support alpha
                  rasterImage->removeAlphaChannel();
                }

            if (imageRW->open(imageIO, SoRasterImageRW::OPEN_WRITE))
            {
              if (imageRW->write(rasterImage))
                imageRW->writeFooter();

              imageRW->close();
            }
            else
              SoDebugError::post("MyAuditorClass::dialogPushButton", 
              "Error cannot open %s for, aborting saving action", filename.toLatin1());

            delete imageRW;
            delete imageIO;
            delete rasterImage;
          }
        }
      }
    }
  }

  void menuFileSelection( SoMenuFileSelection* sel )
  {
    SbString filename = sel->filename.getValue();

    SoInput input;
    if (input.openFile(filename))
    {
      SoNode* node = SoDB::readAll(&input);
      g_renderToTexture->removeAllChildren();
      if (node)
        g_renderToTexture->addChild(node);
      input.closeFile();
    }
  }
};


Widget
buildInterface(Widget window)
{
   g_topLevelDialog = (SoTopLevelDialog*) SoDialogViz::loadFromFile(SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Features/SimpleRenderToTarget/SimpleRenderToTargetDialog.iv"));
  
  if (g_topLevelDialog) { 
    MyAuditorClass *myAuditor = new MyAuditorClass;
    g_topLevelDialog->addAuditor( myAuditor );

    SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

    g_topLevelDialog->buildDialog( window, TRUE );
    g_topLevelDialog->show();

    return customNode ? customNode->getWidget() : window;
  }

  return NULL;
}

int main(int /*argc*/, char** argv)
{
  // Initialize Inventor. This returns a main window to use.
  // If unsuccessful, exit.
  Widget mainWindow = SoXt::init(argv[0]); // pass the app name
  SoDialogViz::init();

  if (mainWindow == NULL) 
    exit(1);

  if ( !SoRenderToTarget::isSupported() ) 
  {
    fprintf(stderr, "This demo cannot be run because SoRenderToTarget is not available on this platforms!\n");
    exit( 1 );
  }

  // Set up viewer
  Widget parent = buildInterface(mainWindow);

  g_mainViewer = new SoXtExaminerViewer(parent);

  SoSeparator* root = new SoSeparator;
  root->ref();

  root->addChild(new SoGradientBackground);

  SoTexture2* rgbaTexture = new SoTexture2;
  SoTexture2* depthTexture = new SoTexture2;


  rgbaTexture->minFilter = SoTexture2::LINEAR;
  rgbaTexture->magFilter = SoTexture2::LINEAR;
  depthTexture->minFilter = SoTexture2::LINEAR;
  depthTexture->magFilter = SoTexture2::LINEAR;

#if defined(USE_FLOAT_COLORBUFFER)
  rgbaTexture->internalFormat = SoTexture::RGBA_FLOAT32;
#endif


  g_renderToTexture = new SoRenderToTarget;
  g_renderToTexture->targets.set1Value(SoRenderToTarget::COLOR0, rgbaTexture);
  g_renderToTexture->targets.set1Value(SoRenderToTarget::DEPTH, depthTexture);
  g_renderToTexture->size.setValue(SbVec2i32(DEFAULT_SIZE, DEFAULT_SIZE));

  SoGroup* rttScene = new SoGroup;
  {
    SoGradientBackground* bgrd = new SoGradientBackground;
    bgrd->color0 = SbVec3f(0.8f, 0.7f, 0.7f);
    bgrd->color1 = SbVec3f(0.3f, 0.1f, 0.0f);
    rttScene->addChild(bgrd);
    rttScene->addChild(new SoSphere());
  }

  g_renderToTexture->addChild(rttScene);

  root->addChild(g_renderToTexture);

  SoSeparator* scene = new SoSeparator();
  {
   g_textureSwitch = new SoSwitch;
   scene->addChild(g_textureSwitch);

   g_textureSwitch->addChild(rgbaTexture);
   g_textureSwitch->addChild(depthTexture);
   g_textureSwitch->whichChild = 0;
   
   scene->addChild(new SoCube);
  }

  root->addChild(scene);

  g_mainViewer->setSceneGraph(root);
  g_mainViewer->show();        
  
  SoXt::show(mainWindow);  // Display main window   
  
  g_mainViewer->viewAll();

  SoXt::mainLoop();

  root->unref();
  delete g_mainViewer;
  g_topLevelDialog->close();

  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


