/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/actions/SoGetPrimitiveCountAction.h>
#include <Inventor/actions/SoGlobalSimplifyAction.h>
#include <Inventor/actions/SoReorganizeAction.h>
#include <Inventor/actions/SoShapeSimplifyAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/io/STL/SoSTLWriteAction.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/simplifier/SoDecimator.h>

#include <DialogViz/auditors/SoDialogAuditor.h>
#include <DialogViz/dialog/SoDialogCustom.h>
#include <DialogViz/dialog/SoDialogPushButton.h>
#include <DialogViz/dialog/SoDialogComboBox.h>
#include <DialogViz/dialog/SoDialogCheckBox.h>
#include <DialogViz/dialog/SoDialogRadioButtons.h>
#include <DialogViz/dialog/SoDialogIntegerSlider.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>
#include <DialogViz/dialog/SoDialogLabel.h>
#include <DialogViz/dialog/SoMenuCheckBox.h>
#include <DialogViz/dialog/SoMenuFileSelection.h>

// Viewers
SoXtExaminerViewer* g_originalShapeViewer = NULL;
SoXtExaminerViewer* g_simplifiedShapeViewer = NULL;

// Dialog
SoDialogIntegerSlider* g_decimatorTarget = NULL;
SoDialogLabel* g_originalNumTrianglesLabel = NULL;
SoDialogLabel* g_simplifiedNumTrianglesLabel = NULL;
SoDialogLabel* g_statusLabel = NULL;
SoDialogComboBox* g_simplificationTypeButton = NULL;
SoDialogPushButton* g_simplifyButton = NULL;

// Nodes
SoSeparator* g_simplifiedScene = NULL;
SoSeparator* g_originalScene = NULL;
SoShapeHints* g_simplifiedShapeHints = NULL;
SoShapeHints* g_originalShapeHints = NULL;
SoRef<SoSeparator> g_initialModel;

// Number of triangles in the original scene
int32_t g_originalNumTriangles = 0;

// Current number of triangles in the simplified scene
int32_t g_currentNumTriangles = 0;

// Remove all cameras in the given scene to prevent any conflict with the viewer's camera
void removeCameras( SoSeparator* scene )
{
  SoSearchAction searchAction;
  searchAction.setType( SoCamera::getClassTypeId() );
  searchAction.setInterest( SoSearchAction::ALL );
  searchAction.setSearchingAll( TRUE );
  searchAction.apply( scene );
  SoPathList& pathList = searchAction.getPaths();
  const int numPaths = pathList.getLength();
  for ( int i = 0; i < numPaths; i++ )
  {
    SoFullPath* path = (SoFullPath*) pathList[i];
    SoCamera* camera = (SoCamera*) path->getTail();
    SoGroup* cameraParent = dynamic_cast<SoGroup*>( path->getNodeFromTail( 1 ) );
    if ( cameraParent != NULL )
      cameraParent->removeChild( camera );
  }
}

// Computes the number of triangles in the given scene
int32_t getNumTriangles( SoSeparator* scene )
{
  SoGetPrimitiveCountAction countAction;
  countAction.apply( scene );
  return countAction.getTriangleCount();
}

// Updates num triangles text labels
void updateNumTrianglesLabel( SoDialogLabel* label, int32_t numTriangles )
{
  label->label.setValue( SbString( SbString("(") + numTriangles + " triangles)" ).getString() );
}

// Update decimation target slider min/max
void updateTargetSlider( int32_t numTriangles )
{
  g_currentNumTriangles = numTriangles;
  const int32_t sliderMin = g_decimatorTarget->min.getValue();
  if ( numTriangles <= sliderMin )
    numTriangles = sliderMin + 1;
  g_decimatorTarget->max.setValue( numTriangles );
  g_decimatorTarget->value.setValue( numTriangles );
}

// Launch simplification with a given target
void performSimplification( int32_t decimationTarget )
{
  SoDecimator* decimer = new SoDecimator;

  SoSimplifyAction* simplifyAction = NULL;
  switch ( g_simplificationTypeButton->selectedItem.getValue() )
  {
  case 0:
    simplifyAction = new SoGlobalSimplifyAction( decimer );
    break;
  case 1:
    simplifyAction = new SoReorganizeAction( decimer );
    break;
  case 2:
    simplifyAction = new SoShapeSimplifyAction( decimer );
    break;
  default:
    break;
  }

  if ( simplifyAction == NULL )
  {
    delete decimer;
    return;
  }

  const float percentage = float( decimationTarget ) / float( g_currentNumTriangles );
  simplifyAction->setSimplificationLevels( 1, &percentage );
  simplifyAction->setMinTriangles( 1 );

  // Update status label
  g_statusLabel->label.setValue( "Simplifying..." );

  // Apply and time decimation
  SbElapsedTime time;
  simplifyAction->apply( g_simplifiedScene );
  const double elapsedTime = time.getElapsed();

  // Update status label
  SbString labelStr = "Simplification took ";
  if ( elapsedTime > 60.0 )
  {
    const int elapsedMinutes = int( elapsedTime / 60.0 );
    const double elapsedSeconds = elapsedTime - 60.0 * double( elapsedMinutes );
    labelStr += SbString().setNum( elapsedMinutes ) + " minutes and " + SbString().setNum( elapsedSeconds ) + " seconds";
  }
  else if ( elapsedTime < 1.0 )
  {
    labelStr += SbString().setNum( 1000.0 * elapsedTime ) + " milliseconds";
  }
  else
  {
    labelStr += SbString().setNum( elapsedTime ) + " seconds";
  }
  g_statusLabel->label.setValue( labelStr.getString() );

  // Replace scene in global and reorganized cases
  if ( simplifyAction->isOfType( SoGlobalSimplifyAction::getClassTypeId() ) )
  {
    g_simplifiedScene->removeAllChildren();
    SoGlobalSimplifyAction* globalSimplifyAction = static_cast<SoGlobalSimplifyAction*>( simplifyAction );
    g_simplifiedScene->addChild( globalSimplifyAction->getSimplifiedSceneGraph() );
  }
  else if ( simplifyAction->isOfType( SoReorganizeAction::getClassTypeId() ) )
  {
    g_simplifiedScene->removeAllChildren();
    SoReorganizeAction* reorganizeAction = static_cast<SoReorganizeAction*>(simplifyAction);
    g_simplifiedScene->addChild( reorganizeAction->getSimplifiedSceneGraph() );
  }

  delete simplifyAction;
  delete decimer;

  // Update slider min/max and text labels
  const int32_t numTriangles = getNumTriangles( g_simplifiedScene );
  updateTargetSlider( numTriangles );
  updateNumTrianglesLabel( g_simplifiedNumTrianglesLabel, numTriangles );
}

// Restore the original model in the simplified shape viewer
void resetModel()
{
  g_originalScene->removeAllChildren();
  g_originalScene->addChild( g_initialModel.ptr() );
  g_simplifiedScene->removeAllChildren();
  g_simplifiedScene->addChild( g_initialModel.ptr()->copy() );

  updateTargetSlider( g_originalNumTriangles );
  updateNumTrianglesLabel( g_simplifiedNumTrianglesLabel, g_originalNumTriangles );
  g_statusLabel->label.setValue( " " );
}

// Read a scene graph file and return the corresponding separator
SoSeparator* readFile( const char* filename )
{
  // Open the input file
  SoInput mySceneInput;
  if ( !mySceneInput.openFile( filename ) )
  {
    std::cerr << "Cannot open file " << filename << std::endl;
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator* myGraph = SoDB::readAll( &mySceneInput );
  if ( myGraph == NULL )
  {
    std::cerr << "Problem reading file" << std::endl;
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

// Load the scene contained in the given filename in both viewers
void loadModel( const SbString& filename )
{
  // Read and prepare model scene
  g_initialModel = readFile( filename.getString() );
  removeCameras( g_initialModel.ptr() );

  // Update labels
  g_originalNumTriangles = getNumTriangles( g_initialModel.ptr() );
  updateNumTrianglesLabel( g_originalNumTrianglesLabel, g_originalNumTriangles );

  resetModel();
  g_originalShapeViewer->viewAll();
}

// Class used to interpret events from the GUI.
class AuditorClass : public SoDialogAuditor
{
  void dialogPushButton( SoDialogPushButton* cpt )
  {
    if ( cpt->auditorID.getValue() == "perform_simplification" )
    {
      performSimplification( g_decimatorTarget->value.getValue() );
      cpt->enable.setValue( FALSE );
    }
    else if ( cpt->auditorID.getValue() == "reset_model" )
    {
      resetModel();
      g_simplifyButton->enable.setValue( TRUE );
    }
  }

  void menuCheckBox( SoMenuCheckBox* cpt )
  {
    if ( cpt->auditorID.getValue() == "lighting" )
    {
      const SoShapeHints::VertexOrdering type = cpt->state.getValue() ? SoShapeHints::COUNTERCLOCKWISE : SoShapeHints::UNKNOWN_ORDERING;
      g_simplifiedShapeHints->vertexOrdering.setValue( type );
      g_originalShapeHints->vertexOrdering.setValue( type );
    }
    else if ( cpt->auditorID.getValue() == "hidden_line" )
    {
      static const SoXtViewer::DrawType types[] = { SoXtViewer::INTERACTIVE, SoXtViewer::STILL };

      const SoXtViewer::DrawStyle style = cpt->state.getValue() ? SoXtViewer::VIEW_HIDDEN_LINE : SoXtViewer::VIEW_AS_IS;
      for ( int i = 0; i < 2; i++ )
      {
        g_simplifiedShapeViewer->setDrawStyle( types[i], style );
        g_originalShapeViewer->setDrawStyle( types[i], style );
      }
    }
  }

  void dialogIntegerSlider( SoDialogIntegerSlider* cpt )
  {
    if ( cpt->auditorID.getValue() == "decimation_target" )
    {
      g_simplifyButton->enable.setValue( TRUE );
    }
  }

  void menuFileSelection( SoMenuFileSelection* cpt )
  {
    const SbString filePath = cpt->fileDirectory.getValue() + "/" + cpt->filename.getValue();
    const SbString& fileName = cpt->filename.getValue();

    if ( cpt->auditorID.getValue() == "open_model" )
    {
      loadModel( filePath );
      g_simplifyButton->enable.setValue( TRUE );
    }
    else if ( cpt->auditorID.getValue() == "save_model" )
    {
      if ( fileName.contains( ".iv" ) )
      {
        SoWriteAction writeAction;
        SoOutput* output = writeAction.getOutput();
        output->openFile( filePath );
        writeAction.apply( g_simplifiedScene->getChild( 0 ) );
        output->closeFile();
      }
      else if ( fileName.contains( ".stl" ) )
      {
        SoSTLWriteAction writeAction;
        writeAction.setOutputFileName( filePath );
        writeAction.apply( g_simplifiedScene->getChild( 0 ) );
      }
    }
  }
};

////////////////////////////////////////////////////////////////////////
// main function
int
main( int argc, char** argv )
{
  // Create the window
  Widget window = SoXt::init( argv[0] );
  if ( !window )
    return 1;

  // DialogViz functions used to build the GUI.
  ////////////////////////////////////////////
  SoDialogViz::init();
  SoRef<SoTopLevelDialog> topLevelDialog =
    (SoTopLevelDialog*) SoDialogViz::loadFromFile( "$OIVHOME/examples/source/Inventor/Features/Simplification/InteractiveSimplification/InteractiveSimplification_interface.iv" );
  topLevelDialog->buildDialog( window, TRUE );
  SoDialogCustom* customOriginalViewer   = (SoDialogCustom*) topLevelDialog->searchForAuditorId( "original_shape_viewer"   );
  SoDialogCustom* customSimplifiedViewer = (SoDialogCustom*) topLevelDialog->searchForAuditorId( "simplified_shape_viewer" );
  AuditorClass* auditor = new AuditorClass;
  topLevelDialog->label.setValue( "Interactive Simplification" );
  topLevelDialog->addAuditor( auditor );
  topLevelDialog->show();

  // Dialog buttons
  g_decimatorTarget             = dynamic_cast<SoDialogIntegerSlider*>(topLevelDialog->searchForAuditorId( "decimation_target"        ));
  g_simplifiedNumTrianglesLabel = dynamic_cast<SoDialogLabel*>        (topLevelDialog->searchForAuditorId( "simplified_num_triangles" ));
  g_originalNumTrianglesLabel   = dynamic_cast<SoDialogLabel*>        (topLevelDialog->searchForAuditorId( "original_num_triangles"   ));
  g_simplificationTypeButton    = dynamic_cast<SoDialogComboBox*>     (topLevelDialog->searchForAuditorId( "simplification_type"      ));
  g_statusLabel                 = dynamic_cast<SoDialogLabel*>        (topLevelDialog->searchForAuditorId( "status"                   ));
  g_simplifyButton              = dynamic_cast<SoDialogPushButton*>   (topLevelDialog->searchForAuditorId( "perform_simplification"   ));
  ////////////////////////////////////////////


  // Simplified Shape Scene
  ////////////////////////////////////////////
  SoRef<SoSeparator> simplifiedRoot = readFile( "$OIVHOME/examples/source/Inventor/Features/Simplification/InteractiveSimplification/InteractiveSimplification_simplified_scene.iv" );
  if ( simplifiedRoot.ptr() == NULL )
    return 1;

  // Nodes
  g_simplifiedScene      = dynamic_cast<SoSeparator*> (SoNode::getByName( SbName( "SIMPLIFIED_SCENE"       ) ));
  g_simplifiedShapeHints = dynamic_cast<SoShapeHints*>(SoNode::getByName( SbName( "SIMPLIFIED_SHAPE_HINTS" ) ));

  // Set up simplified shape viewer:
  g_simplifiedShapeViewer = new SoXtExaminerViewer( customSimplifiedViewer->getWidget() );
  g_simplifiedShapeViewer->setBorder( FALSE );
  g_simplifiedShapeViewer->setDecoration( FALSE );
  g_simplifiedShapeViewer->setAnimationEnabled( FALSE );
  g_simplifiedShapeViewer->setTransparencyType( SoGLRenderAction::NO_SORT );
  g_simplifiedShapeViewer->setSceneGraph( simplifiedRoot.ptr() );
  g_simplifiedShapeViewer->setTitle( "Interactive Simplification" );
  g_simplifiedShapeViewer->viewAll();
  g_simplifiedShapeViewer->show();
  ////////////////////////////////////////////


  // Original Shape Scene
  ////////////////////////////////////////////
  SoRef<SoSeparator> originalRoot = readFile( "$OIVHOME/examples/source/Inventor/Features/Simplification/InteractiveSimplification/InteractiveSimplification_original_scene.iv" );
  if ( originalRoot.ptr() == NULL )
    return 1;

  // Nodes
  g_originalScene      = dynamic_cast<SoSeparator*> (SoNode::getByName( SbName( "ORIGINAL_SCENE"       ) ));
  g_originalShapeHints = dynamic_cast<SoShapeHints*>(SoNode::getByName( SbName( "ORIGINAL_SHAPE_HINTS" ) ));

  // Set up original shape viewer:
  g_originalShapeViewer = new SoXtExaminerViewer( customOriginalViewer->getWidget() );
  g_originalShapeViewer->setBorder( FALSE );
  g_originalShapeViewer->setDecoration( FALSE );
  g_originalShapeViewer->setAnimationEnabled( FALSE );
  g_originalShapeViewer->setTransparencyType( SoGLRenderAction::NO_SORT );
  g_originalShapeViewer->setSceneGraph( originalRoot.ptr() );
  g_originalShapeViewer->setTitle( "Interactive Simplification" );

  // Connect cameras in order to always have the same view in both viewers
  SoCamera* simplifiedShapeCamera = g_simplifiedShapeViewer->getCamera();
  SoCamera* originalShapeCamera = g_originalShapeViewer->getCamera();
  SoFieldList originalCamFields;
  SoFieldList simplifiedCamFields;
  originalShapeCamera->getAllFields( originalCamFields );
  simplifiedShapeCamera->getAllFields( simplifiedCamFields );
  const int numFields = originalCamFields.getLength();
  for ( int i = 0; i < numFields; i++ )
  {
    originalCamFields[i]->connectFrom( simplifiedCamFields[i] );
    simplifiedCamFields[i]->connectFrom( originalCamFields[i] );
  }

  g_originalShapeViewer->show();
  ////////////////////////////////////////////


  SoXt::show( window );

  // Default model
  SbString modelFile( "$OIVHOME/data/models/vehicles/fiero.iv" );

  // Load other model if specified
  if ( argc > 1 )
    modelFile = SbString( argv[1] );

  loadModel( modelFile );

  SoXt::mainLoop();

  delete auditor;
  delete g_simplifiedShapeViewer;
  delete g_originalShapeViewer;

  topLevelDialog = NULL;
  simplifiedRoot = NULL;

  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}
