/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Pascal DOUX (Jan 1998)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/SoDB.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/actions/SoWriteAction.h>

#include <Inventor/actions/SoShapeSimplifyAction.h>
#include <Inventor/simplifier/SoDecimator.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoWWWInline.h>
#include <Inventor/events/SoKeyboardEvent.h>


SoXtExaminerViewer *mySimplifiedViewer;
int numLevels = 4;
static float levels[5] = { 1.0f, 0.6f, 0.3f, 0.1f };

typedef struct cbdata {
  SoXtExaminerViewer *vwr;
  SoNode *root;
  int maxLevels;
} callbackData;

void 
saveToFile(SoNode *root)
{
  SoOutput output;

  printf("storing the simplified scene graph in simplified.iv...\n");
  output.openFile("simplified.iv");
  SoWriteAction wa(&output);
  wa.apply(root);
  output.closeFile();
}

static void
processKeyEvents(void *data, SoEventCallback *cb)
{
  SoKeyboardEvent *keyEvent = (SoKeyboardEvent *) cb->getEvent();
  SoKeyboardEvent::Key key = keyEvent->getKey();

  if (SoKeyboardEvent::isKeyPressEvent(cb->getEvent(), key)) 
    return;

  callbackData *cbData = (callbackData *)data;

  if (key == SoKeyboardEvent::L) { 
    printf("Level Of Simplification used for switching levels\n");
    mySimplifiedViewer->setDecimationStrategy(SoXtViewer::NORMAL);
  } 
  else {
    int whichChild = (int)key - (int)SoKeyboardEvent::NUMBER_1;
    
    if (0 <= whichChild && whichChild < cbData->maxLevels) {
      printf("changing to level %d percentage=%f\n", whichChild+1, levels[whichChild]);
      mySimplifiedViewer->setDecimationStrategy(SoXtViewer::FIXED_PERCENTAGE);
      mySimplifiedViewer->setFixedPercentage(levels[whichChild]);
    } 
  }
  cb->setHandled(); 
}

void
myInlineCallback(const SbString &url, void *, SoWWWInline *node)
{
  // Open the input file
  SoInput mySceneInput;
  
  printf("opening url %s\n", url.getString());
  if (!mySceneInput.openFile(url.getString())) {
    fprintf(stderr, "Cannot open file %s\n", url.getString());
    return;
  }
  
  // Read the whole file into the database
  SoSeparator *myGraph = SoDB::readAll(&mySceneInput);
  if (myGraph == NULL) {
    fprintf(stderr, "Problem reading file\n");
    return ;
  }
  myGraph->renderCulling = SoSeparator::ON;
  node->setChildData(myGraph);
  mySceneInput.closeFile();
}

int 
main(int argc, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) 
    exit(1);

  if (argc != 2) {
    printf("Usage: %s file_name \n", argv[0]);
    exit(-1);
  }

  SoInput input;

  if (! input.openFile(argv[1])) {
    printf("Cannot open file %s \n", argv[1]);
    exit(-1);
  }
   
  SoWWWInline::setFetchURLCallBack(myInlineCallback, (void *)NULL);
  printf("reading %s...\n", argv[1]);
  SoSeparator *root = SoDB::readAll(&input);
  root->ref();

  // the Simplifier object: uses geometric simplification
  SoDecimator *decimer = new SoDecimator;

  // the simplification action: uses the global simplification.
  SoShapeSimplifyAction simplify(decimer);

  simplify.setSimplificationLevels(numLevels, levels);
  simplify.setSizeFactor(4.);
  simplify.setUrlName("new");
  simplify.setMinTriangles(4);

  printf("simplification started...\n");
  simplify.apply(root);
  printf("simplification done\n");
  saveToFile(root);

  SoEventCallback *eventCB = new SoEventCallback;

  root->addChild(eventCB);

  printf("displaying the simplified scene graph...\n");
  // Open a viewer for the simplified scene:
  mySimplifiedViewer = new SoXtExaminerViewer(myWindow);
  mySimplifiedViewer->setSceneGraph(root);
  mySimplifiedViewer->setTitle("Simplified Scene");
  mySimplifiedViewer->setViewing(FALSE);
  mySimplifiedViewer->show();

  printf("\n Enter Esc to switch between viewing and picking mode \n");
  printf("\n Enter a number between 0 and %d to see the corresponding \n"
         "level of simplification\n", numLevels);
  printf("\n Enter l to switch to the Level Of Detail switching method \n"
         "based on distance to the viewer\n\n");

  // Setup the event callback data and routine for switching levels
  callbackData *data = new callbackData;

  data->vwr = mySimplifiedViewer;
  data->root = root;
  data->maxLevels = numLevels;
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), 
                            processKeyEvents, data);
  SoXt::show(myWindow);
  SoXt::mainLoop();

  root->unref();
  delete mySimplifiedViewer;
  SoXt::finish();

  return 0;
}



