/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

#include <Inventor/SbDict.h>

class GraphIcon;
class SoCoordinate3;
class SoIndexedLineSet;
class SoNode;
class SoPath;
class SoSwitch;

////////////////////////////////////////////////////////////////////////
//
// Class: DisplayGraph
//
// Holds display graph. The graph is stored as an array of GraphIcon
// instances.
//
////////////////////////////////////////////////////////////////////////

class DisplayGraph {

  public:

    // Constructor - passed scene to view graph of, so it can count nodes
    DisplayGraph(SoNode *sceneGraph);

    // Destructor
    ~DisplayGraph();

    // Sets up horizontal and vertical spacing values
    void		setIconSpacing(const SbVec2f spacing)
	{ iconSpacing = spacing; }

    // Returns horizontal and vertical spacing values
    SbVec2f		getIconSpacing() const	{ return iconSpacing; }

    // Builds display graph. Returns root node. May be passed another
    // DisplayGraph from which to copy info about icons.
    SoNode *		build(DisplayGraph *oldDisplayGraph = NULL);

    // Returns root node of display graph after it has been built
    SoNode *		getRoot() const		{ return root; }

    // Updates display graph when something changed in scene graph
    void		update();

    // Returns icon corresponding to given path to icon geometry
    // within the display graph
    GraphIcon *		find(const SoPath *path);

    // Returns the icon that corresponds to the given path that starts
    // at the root of the scene graph this is an iconic version of
    GraphIcon *		findFromScenePath(const SoPath *path);

    // Shows/hides what node an instance icon is an instance of.
    // Returns TRUE if graph needs to be rebuilt
    SbBool		toggleInstance(GraphIcon *icon);

    // Swaps instance icon with icon it is an instance of. This allows
    // selection under an instance of a group
    void		swapInstance(GraphIcon *icon);

    // Returns a path from the given root of the graph to given graph
    // icon's root. The path is ref'ed.
    SoPath *		findPathToIcon(SoNode *root, GraphIcon *icon);

    // Used by selection pasting to determine where to draw the
    // feedback icon and where in the graph to insert the selection
    SbBool		getPasteLocation(float x, float y,
					 GraphIcon *&parentIcon,
					 int &childIndex,
					 float &xFeedback, float &yFeedback);

    // Returns icon subgraphs for closed groups and instances
    static SoNode *	getInstanceIcon()	{ return instanceIcon; }
    static SoNode *	getClosedIcon()		{ return closedIcon; }

  private:
    static SbBool	initialized;	// TRUE if class initialized
    GraphIcon		*icons;		// Array of icons
    int			numIcons;	// Number of icons in array
    SoNode		*root;		// Root of display graph
    SoIndexedLineSet	*lineSet;	// Lines connecting icons
    SoCoordinate3	*coords;	// Coordinates of lines
    SoSwitch		*instSwitch;	// Switch to turn on/off instance line
    SoCoordinate3	*instCoords;	// Coordinates of line to show instance
    GraphIcon		*instShown;	// Which instance has line shown
    SbVec2f		iconSpacing;	// Spacing between icons
    static SbDict	*iconDict;	// Holds icons for each node type

    // Scene graphs used for icons
    static SoNode	*instanceIcon;	// Graph for instances
    static SoNode	*closedIcon;	// Graph for closed groups
    static SoNode	*otherIcon;	// Graph for all other icons

    // Used to detect instances when counting or setting up node
    // correspondences
    SbDict		nodeDict;

    // This is used to find a GraphIcon from a picked shape in the
    // display graph
    SbDict		shapeDict;

    // Initializes class
    static void		init();

    // Recursively sets up correspondences between icons and nodes
    void		setNode(SoNode *node, GraphIcon *parentIcon,
				int iconIndex, int &firstChildIndex,
				int childIndex);

    // Counts nodes in subgraph rooted by node
    int			countNodes(const SoNode *root);
    
    // Makes sure state of icons rooted by given icon is consistent
    static void		checkIconState(GraphIcon *root,
				       SbBool shouldBeHidden);

    // Sets up the coordinate3 and lineset nodes to display connecting lines
    void		buildLines();

    // Makes sure the given icon is visible
    void		makeIconVisible(GraphIcon *icon);

    // Sets up coords to draw line connecting instance
    void		setInstanceCoords();

    // Finds the icon graph to use for the given node
    static SoNode *	findIconGraph(SoNode *node);
};


