/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/

#include <Inventor/SbTime.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

class DisplayGraph;
class GraphIcon;

class SbDict;
class SoBoxHighlightRenderAction;
class SoCallbackList;
class SoCoordinate3;
class SoEvent;
class SoEventCallback;
class SoOrthographicCamera;
class SoPath;
class SoSwitch;
class SoTransform;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: GraphViewer
//
//  Subclass of SoXtPlaneViewer that is used to view an iconic version
//  of an Inventor scene graph. This iconic version is referred to as
//  the display graph.
//
//////////////////////////////////////////////////////////////////////////////

class GraphViewer : public SoXtPlaneViewer {
 public:
    GraphViewer(Widget parent = NULL, const char *name = NULL);
    ~GraphViewer();

    // Redefine this function to set up the display graph rather than
    // the graph itself
    virtual void	setSceneGraph(SoNode *newScene);

    // Updates the display graph based on changes to the current scene
    // graph. Sets up the array of GraphIcons.
    void		update();

    // Selects the icon that corresponds to the given path from the
    // root of the scene graph to another node. The root may appear
    // anywhere in the path (i.e., there may be other nodes above it.
    void		select(SoPath *path);

    // Deselects the current selection
    void		deselectAll();

    // Adds/removes a selection callback, which is invoked immediately
    // after a graph icon is selected. The selection path passed to
    // the callback is from the root of the scene graph to the node
    // represented by the selected graph icon.
    void		addSelectionCallback(SoSelectionPathCB *f,
					     void *userData = NULL);
    void		removeSelectionCallback(SoSelectionPathCB *f,
						void *userData = NULL);

    // Adds/removes a deselection callback, which is invoked immediately
    // after a graph icon is deselected. The selection path passed to
    // the callback is from the root of the scene graph to the node
    // represented by the deselected graph icon.
    void		addDeselectionCallback(SoSelectionPathCB *f,
					       void *userData = NULL);
    void		removeDeselectionCallback(SoSelectionPathCB *f,
						  void *userData = NULL);
    // Redefine this from the base class
    virtual Widget	getWidget() const;

    // Returns the scene graph representing the display graph
    SoNode *		getDisplayGraph() const;

  private:
    SoNode		*userScene;	// Scene we are displaying
    SoGroup		*displayRoot;	// Root of display graph
    SoOrthographicCamera *camera;	// Camera used to view scene
    SoSelection		*selector;	// Selection node
    SoSwitch		*pasteSwitch;	// Root of paste feedback subgraph
    SoTransform		*pasteXform;	// Transform node in paste feedback
    SoCoordinate3	*pasteCoord;	// Coordinates for paste feedback line
    SoEventCallback	*ecb;		// Handles all mouse/kbd events
    DisplayGraph	*displayGraph;	// Display graph
    GraphIcon		*pickedIcon;	// Icon hit by last selection pick
    GraphIcon		*selectedIcon;	// Currently-selected icon
    SoNode		*bufferNode;	// Copied/cut node from scene graph
    SbBool		pasteByRef;	// TRUE if paste is by reference
					// as opposed to copying
    SbBool		isGrabbing;	// TRUE if currently grabbing events
    SbTime		prevTime;	// Time of prev mouse press
					// (to detect double-click)
    SoCallbackList  	*selCBs;	// Selection callbacks
    SoCallbackList  	*deselCBs;	// Deselection callbacks
    Widget		widget;		// Top level widget
    SoBoxHighlightRenderAction *rendAct;// Render action with box highlighting

    // Redefine this to add only full viewer buttons (not the plane
    // viewer stuff)
    virtual void	createViewerButtons(Widget parent);
    
    // Builds all widgets
    Widget		buildWidget(Widget parent);

    // Builds the top-bar menu widget
    Widget		buildMenu(Widget parent);

    // Constructs a display graph from the current array of
    // GraphIcons. Sets the viewer to display it. Takes old display
    // graph which it passes to DisplayGraph::build().
    void		buildGraph(DisplayGraph *oldDisplayGraph);

    // Counts nodes in subgraph rooted by node
    static int		countNodes(const SoNode *root, SbDict *nodeDict);

    // Callback to prune a selected path that extends to a shape in an
    // icon to end at the root of the icon
    static SoPath *	selectionFilterCB(void *data,
					  const SoPickedPoint *pick);

    // Callback to store currently-selected icon and invoke callbacks
    static void		selectionCB(void *data, SoPath *path);

    // Callback to clear currently-selected icon and invoke callbacks
    static void		deselectionCB(void *data, SoPath *path);

    // Callbacks for menu display and menu button press
    static void		menuDisplayCB(Widget, XtPointer, XtPointer);
    static void		menuButtonCB(Widget, XtPointer, XtPointer);

    // Callback for mouse button presses
    static void		mousePressCB(void *data, SoEventCallback *ecb);

    // Returns action to be initiated by mouse press
    int			getMousePressAction(SoEventCallback *ecb);

    // Processes action initiated by menu or key press
    void		processAction(int action, SoEventCallback *ecb);

    // Returns GraphIcon corresponding to current selected icon, or NULL
    GraphIcon *		getSelectedIcon() const
	{ return selectedIcon; }

    // Changes selection based on passed code: 0 = parent, 1 = first
    // child, 2 = sibling to left, 3 = sibling to right. Nothing is
    // done if the change is not valid. Returns update code.
    SbBool		changeSelection(int code);

    // Changes selection to be given icon
    void		select(GraphIcon *icon);

    // Invokes user-supplied selection or deselection callbacks
    void		invokeCallbacks(SbBool selecting);

    // Creates and returns subgraph used for selection pasting feedback
    SoNode *		createPasteFeedback();

    // Returns a path from the root of the user's scene graph down to
    // the node represented by the given icon. The path is ref'ed.
    SoPath *		buildPathToIconNode(GraphIcon *tailIcon);

    // Copies current selection
    void		copy();

    // Cuts/clears current selection
    SbBool		cut(SbBool saveSelection);

    // Begins selection pasting
    void		pasteBegin(SoEventCallback *ecb);

    // Callbacks for mouse motion and button presses during selection pasting
    static void		pasteMoveCB(void *data, SoEventCallback *ecb);
    static void		pasteEndCB(void *data, SoEventCallback *ecb);

    // Moves paste feedback to follow cursor, given event containing position
    void		movePasteFeedback(const SoEvent *event,
					  SbBool doPaste);

    // Resets paste feedback to initial state
    void		resetPasteFeedback();

    // Callback for NodeCreator
    static void		nodeCreationCB(void *data, SoNode *newNode);
};

