/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

// OIV-MT Test/Demo
//
//  The compiler directive TGS_TESt is only for VSG Internal use. 
//  THe user does not have the files to run parts of the code.
//
// Multiple Threads version with framerate display.
//
// usage : demo <number of windows> [-s n] [-t]
//            -s n : use single scene graph 1..N
//            -t   : use blended transparency
//            -a   : use anti aliasing
//            -ss n: use single scene graph 1..N in stereo
//
// The following options are for VSG Internal use only
//            -v   : use the small model
//            -d   : use the dragon model (VSG Internal use only)
//            -x   : render everything in one window (large models)

//        The options -v, -d and -s (-ss) are not compatible.
//        The last writen will be used.
//
// This demo uses only "core" Open Inventor, no SoXt or SoWin classes.
// By default demonstrates that using separate threads for viewers
// de-couples the viewer's respective frame rates.  In other words, 
// using a single thread (as you must with classic Inventor) the frame
// rate is limited by the slowest viewer because the viewers render
// sequentially.  Using multiple threads, the viewer with the small
// model should get a higher frame rate because the viewers render in
// parallel (depending somewhat on the number of processors).
//
// The "-s n" option makes demo load one copy of the specified model
// (from the predefined file names numbered 1..N), so all the threads
// are traversing a single copy of the scene graph.  In this case the
// frame rate should (obviously) be the same for each viewer.
//
// Original: Olivier Fedkiw
// Modified: Mike Heck, 25-Feb-01, added code for Windows platform
//           Mike Heck, 26-Feb-01, added cmd line options, code cleanup
//           Julien Chaplier, 26-Feb-01, added stereo option and 'all in one window' option. 
//
// Notes:
// 1) On Windows, build as a console app.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <memory.h>

#ifdef _WIN32
#  include <windows.h>
#else //UNIX
#  include <unistd.h>
#  include <pthread.h>
#  include <GL/glx.h>      
#  include <X11/Intrinsic.h>
#  include <X11/Xresource.h>
#  include <X11/StringDefs.h>
#  include <X11/keysym.h>
#endif

// For Win32 this must be included after windows.h
#include <Inventor/sys/SoGL.h>
#include <GL/glu.h>

#include <Inventor/SoDB.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoGetPrimitiveCountAction.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoWWWInline.h>

#include <Inventor/threads/SbThread.h>
#include <Inventor/threads/SbThreadMutex.h>
#include <Inventor/threads/SbThreadBarrier.h>

#define WINWIDTH    400    
#define WINHEIGHT    400    
#define PI 3.141592654
#define DEG_TO_RAD(DEG) ((DEG/180.0)*PI)
#define RAD_TO_DEG(RAD) ((RAD/PI)*180.0)

#ifdef _WIN32
#  define WINDOW  HWND
#  define GLCONTEXT HGLRC

#  pragma comment(lib, "OpenGL32.lib")
#  pragma comment(lib, "GLU32.lib")

#  define outputString0(_string_) { \
  OutputDebugString(_string_);  \
  fprintf(stderr, _string_); \
}
#  define outputString1(_string_, _arg1_) { \
  char buf[128]; sprintf(buf, _string_, _arg1_); \
  OutputDebugString(buf); \
  fputs(buf, stderr); \
}
#else //UNIX
#  define WINDOW  Window
#  define GLCONTEXT GLXContext

int 
dblBuf[] = { 
  GLX_RGBA, 
  GLX_DEPTH_SIZE, 
  16, 
  GLX_DOUBLEBUFFER, 
  None 
};

// Timer 
XtIntervalId XtTimer;
void TimerCallback(XtPointer, XtIntervalId *);

#  define outputString0(_string_) \
  fprintf(stderr, _string_);
#  define outputString1(_string_, _arg1_) \
  fprintf(stderr, _string_, _arg1_);
#endif //_WIN32 or UNIX

// windowInfo struct for drawing windows (to be allocated)
typedef struct _windowInfo {
  SbThread *thread;
  WINDOW window;
  GLCONTEXT context;
  SbBool sceneBuilt;
  SoSeparator *root;
  SoSeparator *model;
  SoPerspectiveCamera *camera;
  SoRotation *headlightRot;
  SoGLRenderAction *action;
  SbVec2s size;
  SbBool sizeChanged;
#ifdef _WIN32
  HDC hdc;
#else
  Display *dpy;
  XVisualInfo *vi;
#endif

  SoGLContext* windowSoContext;

} windowInfoStruct;

windowInfoStruct *windowInfo = NULL;

int canShareLists = 1;  // Assume this is true unless OpenGL call fails

int timeToQuit  =  0;  // Set this to tell render threads to return from threadRout
int threadCount = -1;  // Mutex protected count of running threads
SbBool clearBuffer = TRUE;  // Mutex protected flag

SbThreadMutex *pCountMutex = NULL;
SbThreadMutex *pClearMutex = NULL;
SbThreadBarrier *pRenderBarrier = NULL;
SbThreadBarrier *quitBarrier = NULL;
#ifndef _WIN32
SbThreadBarrier *initBarrier = NULL;
SbThreadBarrier *ctxBarrier = NULL;
#endif

// ----- Options ----- //
int nWin = 2;   // Default is two windows
int useBlendedTransp = 0; // Default is screen door transp
int useAntiAlias = 0;     // Default is no anti aliasing
int useSingleScene = 0;   // Default is different scene in each window
int useRenderBarrier = 0; // Default is no barrier (threads render independently)
int isStereo = 0;         // Default is monoscopic
int allInOne = 0;         // to render all the scenegraph in one window
int verySmallModel = 0;   // to use a small model (in 1 or 2 part, depending of the allInOne mode)
#ifdef TGS_TEST
int dragonModel = 0;      // to use a dragon model (in 1 or 2 part, depending of the allInOne mode)
#endif
int useMultipleDpy = 0;   // Default is one display (:0.0)

// !!!! WARNING : the position of the 3 very small models and the three
// dragon models is important in the list !!!!!
const char * InventorFiles[] = {
  "$OIVHOME/examples/source/Inventor/MultiThread/data/star.iv", 
  "$OIVHOME/examples/source/Inventor/MultiThread/data/eiffel.iv", 
#ifdef TGS_TEST
  "../data/harley.iv", 
  "../data/Barcelona.iv", 
  "../data/table.iv", 
  "../data/fiero.iv", 
  
  "../data/verySmall_melt.iv", // very small model
  "../data/verySmall_1.iv", // very small model
  "../data/verySmall_2.iv", // very small model
  
  "../data/dragon.iv", // dragon
  "../data/dragon_body.iv", // dragon
  "../data/dragon_fly.iv", // dragon
  "../data/queen.iv", 
  "../data/teapot.iv", 
  "../data/simpleEngine.iv", 
  "../data/bird.iv", 
  "../data/gunstar.iv", 
  "../data/x29.iv", 
  "../data/aircar.iv", 
  "../data/slotMachine.iv", 
  "../data/out.iv", 
  "../data/monitor.iv"
#endif
};

int NUMIVFILES = sizeof(InventorFiles)/sizeof(char*);

typedef struct {
    int id;
} ThreadFuncStruct;

ThreadFuncStruct *tfStruct;

//----------------------------------------------------------------------------
//    FrameRate
#define numRecord 20
typedef struct {
  SbTime startTime;    
  long timeDepart;
  unsigned long timeRecord[numRecord];
  int nbRecord;
} FrameRateStruct;

FrameRateStruct *frameRate;

//////////////////////////////////////////////////////////////////////////////
// Frame rate
//////////////////////////////////////////////////////////////////[JUMP05]////
void 
initFrameRate(int nWin)
{
  for (int j = 0; j<nWin; j++) {
    frameRate[j].startTime = SbTime::getTimeOfDay();
    frameRate[j].timeDepart = -1;
    frameRate[j].nbRecord = 0;
    for (int i = 0; i < numRecord; i++) {
      frameRate[j].timeRecord[i] = 0; 
    }
  }
}

void 
actualiseFrameRate(int id)
{
  for (int i = frameRate[id].nbRecord; i > 0; i--) {
    frameRate[id].timeRecord[i] = frameRate[id].timeRecord[i-1]; 
  }
  frameRate[id].timeRecord[0] = 
    (SbTime::getTimeOfDay()-frameRate[id].startTime).getMsecValue();
  if (frameRate[id].timeDepart < 0) {
    frameRate[id].timeDepart = frameRate[id].timeRecord[0];
  }
  if (frameRate[id].nbRecord<(numRecord-1)) {
    frameRate[id].nbRecord++;
  }
}

void 
afficheFrameRate(int id)
{
  float totalDuration = 
    (frameRate[id].timeRecord[0] - frameRate[id].timeDepart) / 1000.0F;
  float imageDuration = 
    (frameRate[id].timeRecord[0] - frameRate[id].timeRecord[1]) / 1000.0F;
  float durationAverage = 
    (frameRate[id].timeRecord[0] - frameRate[id].timeRecord[frameRate[id].nbRecord]) / 1000.0F;
  float frame1 = 1.0F / imageDuration;
  float frame = (frameRate[id].nbRecord) / durationAverage;
  char message[64];
  sprintf(message, "Window%d - Framerate avg: %.2f", id, frame);
  
  // set window name ...
#ifdef _WIN32
  SetWindowText(windowInfo[id].window, message);
#else //UNIX
  XSetStandardProperties(windowInfo[id].dpy, windowInfo[id].window, 
                         message, NULL, None, NULL, 0, NULL);
  //XFlush(windowInfo[id].dpy);
#endif
}

//////////////////////////////////////////////////////////////////////////////
SoNode *
buildScene(int ivFileIndex, int id)
{
  char buffer[64];
  windowInfoStruct *w = &(windowInfo[id]);
  
  // Build a simple scene graph
  //
  // Note there is no point in caching at the root node since we will
  // be constantly modifying the camera and headlight nodes under it.
  // Hopefully caching will occur at the model Separator node.
  //
  // *** NOTE ***
  // Each window, and therefore each thread, will have its own camera
  // and headlight nodes, even they are using exactly the same scene
  // graph for the model.  This allows us to freely modify the camera
  // and headlight nodes in each thread, without any synchronization.
  
  w->sceneBuilt = TRUE;
  w->root = new SoSeparator;
  w->root->ref();
  w->root->boundingBoxCaching = FALSE;
  
  sprintf(buffer, "ROOT_%d", id);
  w->root->setName(buffer); // for debug output
  
  // Create camera
  w->camera = new SoPerspectiveCamera;
  w->camera->position.setValue(0, 0, 10);
  SbVec3f axis;
  float angle;
  axis.setValue(0, 1, 0);
  angle = (float) (0.5*PI);
  //w->camera->orientation.setValue(axis, angle);
  
  w->root->addChild(w->camera);
  
  // Some light
  
  SoTransformSeparator *headlightGroup;
  SoDirectionalLight *headlightNode;
  
  headlightGroup = new SoTransformSeparator(2);
  w->headlightRot = new SoRotation;
  headlightNode = new SoDirectionalLight;
  headlightGroup->ref();
  headlightGroup->addChild(w->headlightRot);
  headlightGroup->addChild(headlightNode);
  headlightNode->direction.setValue(SbVec3f(0.2F, -0.2F, -0.9797958971F));
  w->root->addChild(headlightGroup);
  headlightGroup->unref();
  
  SoSeparator *model = NULL;
  int fileIndex = ivFileIndex;
  if (useSingleScene) {
    // Single scene graph requested
    // If this is not the first window, we already have the model
    //     scene graph.
    // Else load the specified model file.
    //
    // Note each window will have its own camera and headlight, since those
    // are updated by the window's render thread.
    if (id > 0) {
      model = windowInfo[0].model;
    }
    else {
      fileIndex = useSingleScene - 1;
    }
  }
  else if (verySmallModel) {
#ifdef TGS_TEST
    if (allInOne) {
      fileIndex = id + 7; // we want to use the files 
    }
    // 8 and 9 (verySmall_1.iv and 
    //verySmall_2.iv)
#else
    if (allInOne) {
      fileIndex = id + 3; // we want to use the files 
    }
    // 4 and 5 (verySmall_1.iv and 
    // verySmall_2.iv)
#endif
  }
#ifdef TGS_TEST
  else if (dragonModel) {
    if (allInOne) {
      fileIndex = id + 10; // we want to use the files
    }
    // 11 and 12 (dragon_body.iv 
    // and dragon_fly.iv)
    else {
      fileIndex = id + 9; // we want to use the file 10 (dragon.iv)
    }
  }
#endif
  if (model == NULL) {
    // Open the input file
    SoInput mySceneInput;
    if (!mySceneInput.openFile(InventorFiles[fileIndex])) {
      fprintf(stderr, "Cannot open file %s\n", InventorFiles[fileIndex]);
    }
    else {
      SoWWWInline::setReadAsSoFile(TRUE);

      // Read the whole file into the database
      model = SoDB::readAll(&mySceneInput);
      if (model == NULL) {
        fprintf(stderr, "Problem reading file %s\n", InventorFiles[fileIndex]);
        return w->root;
      } 
      mySceneInput.closeFile();
    }
    // Substitute our favorite Inventor geometry :-)
    if (model == NULL) {
      model = new SoSeparator;
      SoMaterial *pMat = new SoMaterial;
      pMat->diffuseColor.setValue(SbColor(1, 0, 0));
      model->addChild(pMat);
      model->addChild(new SoCone);
    }
    sprintf(buffer, "MODEL_%d", id);
    model->setName(buffer); // for debug output
  }

  w->root->addChild(model);
  
  // In the allInOne case, the first camera does a viewAll. Then the others
  // keep the same position and orientation to preserve the relative positions
  // between the different objects
  if (!allInOne || id == 0) {
    w->camera->viewAll(w->root, SbViewportRegion(WINWIDTH, WINHEIGHT));
  }
  else {
    w->camera->viewAll(windowInfo[0].root, SbViewportRegion(WINWIDTH, WINHEIGHT));
  }
  w->headlightRot->rotation.setValue(w->camera->orientation.getValue());
  
  // This must be set after the viewAll method is called
  if (isStereo) {
    if (id == 0) w->camera->setStereoMode(SoCamera::LEFT_VIEW);
    else if (id == 1) w->camera->setStereoMode(                                                         SoCamera::RIGHT_VIEW);
  }
  
  w->model = model;
  
  // ?????
  // It may make sense to disable notification on the camera
  // and headlighRot nodes...
  // Since we will be continuously traversing anyway, we probably
  // don't care about notification.  This saves cpu cycles and, 
  // more importantly, mutex locking.
  w->camera->enableNotify(FALSE);
  w->headlightRot->enableNotify(FALSE);
  
  // Could be useful to know how big this model actually is...
  if (id == 0 || !useSingleScene) {
    SoGetPrimitiveCountAction count;
    count.apply(model);
    int tris = count.getTriangleCount();
    int lines = count.getLineCount();
    int pts = count.getPointCount();
    char buf[128];
    sprintf(buf, "   Model contains %d triangles, %d lines, %d points\n", 
            tris, lines, pts);
    outputString0(buf);
  }
  return w->root;
}


//////////////////////////////////////////////////////////////////////////////
// Rotate camera
//////////////////////////////////////////////////////////////////////////////
void 
rotateCamera(SoCamera * camera, const SbRotation &Rot)
{
  if (camera == NULL) {
    return;
  }
  
  SbRotation camRot = camera->orientation.getValue();
  SbVec3f position = camera->position.getValue();
  
  SbMatrix pos, matrice;
  
  pos[0][3] = position[0];
  pos[1][3] = position[1];
  pos[2][3] = position[2];
  
  matrice = Rot.inverse();
  pos = matrice * pos;
  
  camRot = camRot * Rot;
  
  position[0] = pos[0][3];
  position[1] = pos[1][3];
  position[2] = pos[2][3];
  
  camera->position = position;
  camera->orientation = camRot;
}

///////////////////////////////////////////////////////////////////////
// Swap OpenGL buffers
///////////////////////////////////////////////////////////////////////
void 
swapBuffers(int id)
{
  windowInfo[id].windowSoContext->swapBuffers();
}

///////////////////////////////////////////////////////////////////////
// Bind current OpenGL context
///////////////////////////////////////////////////////////////////////
void 
bindCurrent(int id)
{
  int windId = id;
#ifdef _WIN32
  windId = (allInOne ? 0 : id);
#endif
  windowInfo[windId].windowSoContext->bind();
}

///////////////////////////////////////////////////////////////////////
// UnBind current OpenGL context
///////////////////////////////////////////////////////////////////////
void 
unbindCurrent(int id)
{
  int windId = id;
#ifdef _WIN32
  windId = (allInOne ? 0 : id);
#endif
  windowInfo[windId].windowSoContext->unbind();
}


void 
antiAliasCB(void *)
{
  // This is what was done by default in open inventor :
  // glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  // But this can't be done for the allInOne case.
  // We'll see this later.
}

//////////////////////////////////////////////////////////////////////////////
// Redraw the IV scene
//////////////////////////////////////////////////////////////////////////////
void 
redrawScene(int id)
{
  windowInfoStruct *w = &(windowInfo[id]);
  if (w->root == NULL) {
    return;
  }
  
  if (w->action == NULL) {
    SbViewportRegion myViewport(WINWIDTH, WINHEIGHT);
    w->action = new SoGLRenderAction(myViewport);
    // Enable transparency if requested
    if (useBlendedTransp) {
      w->action->setTransparencyType(SoGLRenderAction::NO_SORT);
    }
    // Display lists should be shareable since all the render windows in
    // this demo should be using the same visual/pixelformat, therefore
    // all actions will have the same cache context.
    // If *not* shareable, set each action's cache context to a different
    // value (for example, the "id" passed in to this function).
    if (canShareLists) {
      w->action->setCacheContext(1);
    }
    else {
      w->action->setCacheContext(id+1);
    }
    if (useAntiAlias) {
      w->action->setNumPasses(2);
      if (allInOne) {
        w->action->setPassCallback(antiAliasCB, NULL);
      }
    }
  }
  
  bindCurrent(id);
  
  if (w->sizeChanged) {
    // Make sure Open Inventor knows what size our window is.
    // This is important for some nodes (e.g. SoText2).
    // The renderAction will update OpenGL's viewport.
    w->sizeChanged = FALSE;
    int width = w->size[0];
    int height = w->size[1];
    w->action->setViewportRegion(SbViewportRegion(width, height));
  }
  
  // For the allInOne case, the buffer need to be clear only one time
  // before all the threads render in it.
  if (allInOne) {
    pClearMutex->lock();
    if (clearBuffer) {
      glClearColor(0.5f, 0.5f, 0.5f, 1.0f);
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      clearBuffer = false;
    }
    pClearMutex->unlock();
  }
  else {
    glClearColor(0.5f, 0.5f, 0.5f, 1.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  }
  
  // Update camera and headlight
  SbRotation rot(SbVec3f(0, 1, 0), (float) DEG_TO_RAD(10));
  
  // Switch to monoscopic mode before the rotation to use the right 
  // orientation and position off the camera
  if (isStereo) {
    w->camera->setStereoMode(SoCamera::MONOSCOPIC);
  }
  rotateCamera(w->camera, rot);
  
  // Switch back to stereoscopic mode
  if (isStereo) {
    if (id == 0) {
      w->camera->setStereoMode(SoCamera::LEFT_VIEW);
    }
    else if (id == 1) {
      w->camera->setStereoMode(SoCamera::RIGHT_VIEW);
    }
  }
  w->headlightRot->rotation.setValue(w->camera->orientation.getValue());
  
  // Render the scene
  w->action->apply(w->root);

  if (useRenderBarrier) {
    pRenderBarrier->enter();
  }
  
  // In the allInOne case, there's just one swapBuffer that's need to be made.
  if (!allInOne || id == 0) {
    swapBuffers(id);
  }

  // Unbind the context
  unbindCurrent(id);

  if (allInOne) {
    pClearMutex->lock();
    if (!clearBuffer) {
      clearBuffer = true;
    }
    pClearMutex->unlock();
    // A barrier is needed here to avoid that one thread render again
    // before all test the clearBuffer flag.
    if (useRenderBarrier) {
      pRenderBarrier->enter();
    }
  }
  
  if (!timeToQuit) {
    actualiseFrameRate(id);
    afficheFrameRate(id);
  }
  
  return;
}

//////////////////////////////////////////////////////////////////////////////
// Create and initialize OpenGL window.
//////////////////////////////////////////////////////////////////////////////
#ifdef _WIN32
LRESULT CALLBACK 
windowProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  HDC hdc;
  PAINTSTRUCT ps;
  static char globalLockState = 'U';
  
  // Window "id" is stored with window
  int id = (int)GetWindowLongPtr(hwnd, 0);
  
  switch (message) {
  // Render threads are continuously updating, so need to redraw.
  // Still need to make these calls to validate window.
  case WM_PAINT:
    hdc = BeginPaint(hwnd, &ps);
    EndPaint(hwnd, &ps);
    break;
    
  // Handle this so Windows knows we're painting our own background
  case WM_ERASEBKGND:
    return TRUE;
    
  // TODO: Handle window resize
  case WM_SIZE: {
    int width = LOWORD(lParam);  // width of client area               
    int height = HIWORD(lParam); // height of client area              
    windowInfo[id].size.setValue(width, height);
    windowInfo[id].sizeChanged = TRUE;
    // all the scenes need to be resized
    if (allInOne) {
      for (int i = 1; i<nWin; i++) {
        windowInfo[i].size.setValue(width, height);
        windowInfo[i].sizeChanged = TRUE;
      }
    }
                }
    break;
    
  // TODO: Handle ESC to exit
  case WM_KEYDOWN:
    if (wParam == VK_ESCAPE || wParam == 'Q') {
      timeToQuit = 1;
      outputString0("Quit key detected, requesting thread exit\n");
    }
    
    // Experimental -- Should stop rendering while locked
    if (wParam == 'W') {
      if (globalLockState == 'U') {
        outputString0("Global write lock requested...\n");
        globalLockState = 'W';
        SoDB::writelock();
      }
    }
    if (wParam == 'R') {
      if (globalLockState == 'U') {
        outputString0("Global read lock requested...\n");
        globalLockState = 'R';
        SoDB::readlock();
      }
    }
    if (wParam == 'U') {
      if (globalLockState == 'W') {
        outputString0("Releasing global write lock...\n");
        SoDB::writeunlock();
      }
      if (globalLockState == 'R') {
        outputString0("Releasing global read lock...\n");
        SoDB::readunlock();
      }
      globalLockState = 'U';
    }
    break;
    
  case WM_CLOSE:
  case WM_DESTROY:
    timeToQuit = 1;
    break;
  }

  return DefWindowProc(hwnd, message, wParam, lParam);
}

void 
openWindow(int id, char *windowName)
{
  HWND hwnd;
  WNDCLASS wndclass;
  const char *className = "renderWindow";
  const int windowOffset = 25;
  
  static int classRegistered = 0;
  
  if (!classRegistered) {
    classRegistered = 1;
    wndclass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    wndclass.lpfnWndProc = windowProc;
    wndclass.cbClsExtra = 0;
    wndclass.cbWndExtra = sizeof(void*);   // Used to store window "id"
    wndclass.hInstance = GetModuleHandle(NULL);
    wndclass.hIcon = NULL;
    wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = NULL;
    wndclass.lpszMenuName = NULL;
    wndclass.lpszClassName = className;
    
    ATOM result = RegisterClass(&wndclass);
  }
  
  int xPos = id * windowOffset, yPos = id * windowOffset;

  if (nWin <= 3) {
    xPos = id * (WINWIDTH + 1);
    yPos = windowOffset;
  }
  
  hwnd = CreateWindow(className, windowName, 
    WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS, // Style
    xPos, // X position
    yPos, // Y position
    WINWIDTH, // Width
    WINHEIGHT, // Height
    NULL, // Parent
    NULL, // Menu
    GetModuleHandle(NULL), // App instance handle
    NULL); // Window creation data
  if (hwnd == NULL) {
    outputString1("CreateWindow failed for id %d\n", id);
    return;
  }
  else {
    outputString1("Created window for id %d\n", id);
  }
  
  // Store ptr to "id" for use in window proc
  SetWindowLongPtr(hwnd, 0, id);
  ShowWindow(hwnd, SW_SHOW);
  
  windowInfo[id].window = hwnd;
  windowInfo[id].hdc = GetDC(hwnd);
  windowInfo[id].size.setValue(WINWIDTH, WINHEIGHT);
  windowInfo[id].sizeChanged = TRUE;
}

#else //UNIX

void 
openWindow(int id, char *windowName)
{
  Display *dplay = windowInfo[id].dpy;
  XVisualInfo *vi = windowInfo[id].vi;
  
  // create a window
  Colormap cmap;
  XSetWindowAttributes swa;
  cmap = XCreateColormap(dplay, RootWindow(dplay, vi->screen), 
                         vi->visual, AllocNone);
  swa.colormap = cmap;
  swa.border_pixel = 0;
  swa.event_mask = 
    StructureNotifyMask | KeyPressMask | ButtonPressMask | ExposureMask;
  Window w = XCreateWindow(dplay, RootWindow(dplay, vi->screen), 
                           0, 0, WINWIDTH, WINHEIGHT, 
                           0, vi->depth, InputOutput, vi->visual, 
                           CWBorderPixel|CWColormap|CWEventMask, &swa);
  // set window name ...
  XSetStandardProperties(dplay, w, windowName, windowName, 
                         None, NULL, 0, NULL);
  XMapWindow(dplay, w);
  XMoveWindow(dplay, w, (id * (WINWIDTH + 20)) + 20, 100);
  printf("(openWindow) window %s\n", (w ? "created" : "= null"));
  windowInfo[id].window = w;
  windowInfo[id].size.setValue(WINWIDTH, WINHEIGHT);
  windowInfo[id].sizeChanged = TRUE;
}
#endif //_WIN32


///////////////////////////////////////////////////////////////////////
void 
createContext(int id)
{
#ifdef _WIN32
  static int pixelformat = 0;
  static PIXELFORMATDESCRIPTOR pfd;
  
  HDC hdc;
  if (!allInOne) hdc = windowInfo[id].hdc;
  else hdc = windowInfo[0].hdc;
  
  if (pixelformat == 0) {
    int numpf = DescribePixelFormat(hdc, 1, 0, NULL);
    for (int i = 1; i < numpf; i++) {
      DescribePixelFormat(hdc, i, sizeof(pfd), &pfd);
      if (pfd.iPixelType == PFD_TYPE_RGBA &&
          pfd.dwFlags & PFD_SUPPORT_OPENGL &&
          pfd.dwFlags & PFD_DRAW_TO_WINDOW &&
          pfd.dwFlags & PFD_DOUBLEBUFFER &&
          pfd.cDepthBits > 0) {
        pixelformat = i;
        break;
      }
    }
  }
  if (!SetPixelFormat(hdc, pixelformat, &pfd)) {
    outputString1("SetPixelFormat failed for id %d\n", id);
  }
  windowInfo[id].context = wglCreateContext(hdc);
  if (windowInfo[id].context == NULL) {
    outputString1("CreateContext failed for id %d\n", id);
    return;
  }
  
  windowInfo[id].windowSoContext = new SoGLContext(hdc, &pfd, NULL, windowInfo[id].context);
  windowInfo[id].windowSoContext->ref();

#else //UNIX
  Display * dplay = windowInfo[id].dpy;
  XVisualInfo *vi = windowInfo[id].vi;
  
  if ((dplay==NULL) || (vi==NULL))
    printf("(CreateContext) Houston we have a problem\n");
  
  if (id > 0) {
    
    ctxBarrier->enter();
    windowInfo[id].context =
      glXCreateContext(dplay, vi, windowInfo[0].context /* share display lists */, 
      /* direct rendering if possible */ GL_TRUE);
  }
  else {
    
    windowInfo[id].context =
      glXCreateContext(dplay, vi, /* no sharing of display lists */ None, 
      /* direct rendering if possible */ GL_TRUE);
      ctxBarrier->enter();
  }
  if (!windowInfo[id].context) {
    outputString1("(CreateContext) failed for id %d\n", id);
  }
  else {
    outputString1("(CreateContext) succeed for id %d\n", id);
  }

  windowInfo[id].windowSoContext = new SoGLContext(dplay, vi, windowInfo[id].window, windowInfo[id].context);
  windowInfo[id].windowSoContext->ref();
#endif
}

//////////////////////////////////////////////////////////////////////////////
// Initialize OpenGL context
//////////////////////////////////////////////////////////////////////////////
void 
initContext(int id)
{
  glEnable(GL_DEPTH_TEST);
  glClearColor(0.8f, 0.8f, 0.8f, 1.0f);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(90.0, 1.0, 2.0, 12.0);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  glTranslatef(0.0, 0.0, -5.0);
  return;
}

//////////////////////////////////////////////////////////////////////////////
// Cleanup stuff created by thread
//////////////////////////////////////////////////////////////////////////////
void 
cleanup(int id)
{
  windowInfoStruct *w = &(windowInfo[id]);
  
  if (w->action) {
    delete w->action;
    w->action = NULL;
  }
  if (w->root) {
    w->root->unref();
    w->root = NULL;
  }
  if (w->context) {
#ifdef _WIN32
    if (w->window && IsWindow(w->window)) {
      wglDeleteContext(w->context);
    }
#else
    glXDestroyContext(w->dpy, w->context);
#endif
    w->context = NULL;
  }
  if (w->window) {
#ifdef _WIN32
    if (IsWindow(w->window)) {
      DestroyWindow(w->window);
    }
    w->window = NULL;
#endif
  }
}


//////////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////////
void *
threadRout (void * _userData)
{
  pCountMutex->lock();
  if (threadCount == -1) {
    threadCount = 1;  // I am the first thread
  }
  else {
    threadCount++;    // I am some other thread
  }
  pCountMutex->unlock();
  
  // Every thread that makes OpenInventor calls must call SoDB::init
  // first.  This is required to setup thread local storage.
  
  SoDB::init();
  SbThreadId_t tid = SbThread::getCurrentThreadId();
  
  if (_userData == NULL) {
    outputString1("(threadRout) thread 0x%x No user data in threadRout\n", tid);
  }
  else {
    outputString1("(threadRout) thread 0x%x userdata OK\n", tid);
  }
  ThreadFuncStruct *tfStruct = (ThreadFuncStruct *)_userData;
  
  // Create and initialize windows
  char name[32];
  int i = tfStruct->id;
  
  // Note: For Win32, must create window before context
  sprintf(name, "Window%d", i);
#ifndef _WIN32
  XVisualInfo *vi;
  Display *dplay;
  char *display_name;
  
  pClearMutex->lock();
  if (useMultipleDpy) {
    printf("(threadRout) thread 0x%x connecting to Xserver\n", tid);
    display_name = (i == 0 ? strdup(":0.0") : strdup(":0.1"));
  } 
  else {
    display_name = NULL;
  }
  
  dplay = XOpenDisplay(display_name);
  pClearMutex->unlock();
  if (dplay == NULL) {
    printf("(threadRout) thread 0x%x No display\n", tid);
    exit(0);
  }
  else {
    printf("(threadRout) thread 0x%x Display OK\n", tid);
  }
  
  vi = glXChooseVisual(dplay, DefaultScreen(dplay), dblBuf);
  printf("(threadRout) thread 0x%x %s\n", 
         tid, (vi == NULL ? "No visual": "Visual OK"));
  windowInfo[i].dpy = dplay;
  windowInfo[i].vi = vi;
  
  // signal to the event loop that the initialization is complete
  initBarrier->enter();
  
  // For Win32, we need to create the render windows in the main
  // thread so GetMessage can retrieve messages for all of them.
  // We also need to create the contexts and call wglShareLists
  // before we start using the contexts, else wglShareLists will
  // fail with error 170 (Resource is busy).
  openWindow (i, name);
  createContext(i);
#endif
  bindCurrent(i);
  initContext(i);
  swapBuffers(i);
  unbindCurrent(i);
  
  while (!timeToQuit) {
    redrawScene(i);        
    //Sleep(1000); // Experiment
  }
  
  cleanup(i);
  
  pCountMutex->lock();
  threadCount--;
  pCountMutex->unlock();
  
  outputString1("(threadRout) thread 0x%x exiting...\n", tid);
  SoDB::finish();
  quitBarrier->enter();
  
  return NULL;
}


//////////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////////
void *
eventLoop(void * arg)
{
#ifdef _WIN32
  // Note: GetMessage with a zero window handle will retrieve messages for
  // all windows created by the current thread (which should be the main
  // thread in this case), so it should handle all our rendering windows.
  MSG msg;
  int result;
  while (!timeToQuit) {
    result = GetMessage(&msg, 0 /*parentWindow*/, 0, 0);
    if (result == 0 || msg.message == WM_QUIT) {
      exit(0);
    }
    else if (result > 0) {
      DispatchMessage(&msg);
    }
    pCountMutex->lock();
    if (threadCount == 0) {
      pCountMutex->unlock();
      break;
    }
    pCountMutex->unlock();
  }
#else //UNIX
  // wait here until the other threads initialize their window 
  initBarrier->enter(); 
  delete initBarrier;
  
  XEvent event;
  
  int i;    
  while (1) {
    for (i = 0; i < nWin; i++){
      windowInfoStruct *w = &(windowInfo[i]);
      do {
        XNextEvent(w->dpy, &event);
        switch (event.type) {
        case ConfigureNotify:
          //fprintf(stdout, "ConfigureNotify type event\n");
          //glXMakeCurrent(dplay, window, context)
          //glViewport(0, 0, event.xconfigure.width, event.xconfigure.height);
          break;
        case Expose:
          //fprintf(stdout, "Expose type event\n");
          break;
        case KeyPress:
          fprintf(stdout, "KeyPress type event\n");
          KeySym keysym;
          XKeyEvent *kevent;
          char buffer[1];
          /* It is necessary to convert the keycode to a 
          * keysym before checking if it is an escape */
          kevent = (XKeyEvent *) &event;
          if (XLookupString((XKeyEvent *)&event, buffer, 1, &keysym, NULL) == 1 &&
              keysym == (KeySym)XK_Escape) {
            timeToQuit = 1;               
          }
          break;
        case ButtonPress:
          Window win;
          win = event.xbutton.window;
          fprintf(stdout, "ButtonPress type event in window %x\n", win);
          break;
        }
        
        pCountMutex->lock();
        if (threadCount == 0) {
          pCountMutex->unlock();
          break;
        }
        pCountMutex->unlock();
      } while (XPending(w->dpy));
    }
  }
#endif
  return NULL;
}

//////////////////////////////////////////////////////////////////////////////
int 
main(int argc, char **argv)
{
  if (argc > 1) {
    for (int i = 1; i < argc; i++) {
      char *cp = argv[i];
      if (*cp == '-') {   // an option
        cp++;
        if (*cp == 'h') {
          // help
          nWin = 0;
        }
        if (*cp == 'b') {
          useRenderBarrier = TRUE;
        }
        if (*cp == 't') {
          useBlendedTransp = TRUE;
        }
        if (*cp == 'a') {
          useAntiAlias = 1;
        }
        // There are two versions of the very small model : 
        // one in one piece in verySmall_melt.iv and the other, 
        // the same one but split in two files verySmall_1.iv 
        // and verySmall_2.iv, so it can be used for the allInOne
        // demo.
        if (*cp == 'v') {
          verySmallModel = 1;
#ifdef TGS_TEST
          dragonModel = 0;
#endif
          useSingleScene = 0;
        }
#ifdef TGS_TEST
        if (*cp == 'd') {
          // There are also two versions of the dragon model : 
          // one in one piece (dragon.iv) and the other, 
          // the same one but split in two files (dragon_body.iv 
          // and dragon_fly.iv), so it can be used for the allInOne
          // demo.
          dragonModel = 1;
          verySmallModel = 0;
          useSingleScene = 0;
        }
#endif
        if (*cp == 'x') {
          allInOne = TRUE;
          useRenderBarrier = TRUE;
        }
        if (*cp == 'X') {
          useMultipleDpy = TRUE;
        }
        if (*cp == 's') {
          cp++;
          if (*cp == 's') { // ss : stereo mode
            outputString0("Stereo mode\n");
            isStereo = 1;
            // barrier are needed to synchronise the two views
            useRenderBarrier = TRUE;
          }
          i++;
          if (i < argc) {
            useSingleScene = atoi(argv[i]);
#ifdef TGS_TEST
            dragonModel = 0;
#endif
            verySmallModel = 0;
          }
        }
      }
      else {
        nWin = atoi(argv[i]);
      }
    }
  }
  if (nWin > 0) {
    outputString1("demo with %i windows\n", nWin);
  }
  else { 
    outputString0("usage : demo <number of windows> [-s n] [-t]\n");
    outputString0("        -s n : use single scene graph 1..N\n");
    outputString0("        -ss n: use single scene graph 1..N in stereo\n");
    outputString0("        -v   : use the small model\n");
    outputString0("        -d   : use the dragon model\n");
    outputString0("        -x   : render everything in one window (large models)\n");
    outputString0("        -t   : use blended transparency\n");
    outputString0("        -a   : use anti aliasing\n");
    outputString0("The options -v, -d and -s (-ss) are not compatible.\n");
    outputString0("The last writen will be used.\n");
    exit(0);
  }
  
  windowInfo = (windowInfoStruct*)malloc(sizeof(windowInfoStruct) * nWin);
  if (windowInfo == NULL) {
    outputString0("error allocating memory for window info\n");
    exit(0);
  }
  memset(windowInfo, 0, sizeof(windowInfoStruct)*nWin);
  
  tfStruct = (ThreadFuncStruct *)malloc(sizeof(ThreadFuncStruct) * nWin);
  if (tfStruct == NULL) {
    outputString0("error allocating memory for tfStruct\n");
    exit(0);
  }
  
  outputString0("(main) Init OK !\n");
  
#ifndef _WIN32
  XInitThreads();
#endif
  
  // Disable render caching if problems occur
  //_putenv("IV_SEPARATOR_MAX_CACHES=3");
  
  // Debug use of render caches if needed
  //_putenv("IV_DEBUG_CACHELIST=1");
  
  SoDB::init();
  SoInteraction::init();
  outputString0("(main) SoDB Init OK !\n");
  
  //---------------------------------------------------------------------------
  // Create and initialize the windows
  
  outputString0("(main) DO_THREAD mode____________________________\n");
  outputString1("nWin = %i\n", nWin);
  
  int i;
  // build all scene in the main thread
  for (i=0; i < nWin; i++) {
    outputString1("Building scene %i ...\n", i);
    buildScene(i, i);
    outputString1(" ... %i done\n", i);
  }
  
  frameRate = (FrameRateStruct*)malloc(sizeof(FrameRateStruct)*nWin);
  if (frameRate == NULL) {
    outputString0("error: framerate memory allocation\n");
    exit(0);
  }
  initFrameRate(nWin);
  
  /////////////////////////////////
  // creation of rendering threads
  /////////////////////////////////
  
  pCountMutex = new SbThreadMutex;
  pClearMutex = new SbThreadMutex;
  quitBarrier = new SbThreadBarrier(nWin+1);
  if (useRenderBarrier) {
    pRenderBarrier = new SbThreadBarrier(nWin);
  }
#ifndef _WIN32    
  // This barrier is used to assure that the eventLoop will
  // not attempt to read windowInfo->dpy before it is set 
  // by the other threads.
  initBarrier = new SbThreadBarrier(nWin+1);
  ctxBarrier = new SbThreadBarrier(nWin);
#endif
  
#ifdef _WIN32
  // Note: For Win32, we need to create the windows in the main thread
  // so our event loop can process messages for all the windows.
  // We also need to create the contexts and call wglShareLists
  // before we start using the contexts, else wglShareLists will
  // fail with error 170 (Resource is busy).  Don't know why. :-(
  for (i = 0; i < nWin; i++) {
    char name[80];
    sprintf(name, "Window%d", i);
    // There's only one window that's built for the allInOne case
    if (!allInOne || i==0) openWindow(i, name);
    createContext(i);
  }
  
  // NULL in the main thread to avoid problems with others threads
  wglMakeCurrent(NULL, NULL);
  
#endif //_WIN32
  
  // Actually create the rendering threads
  for (i = 0; i < nWin; i++) {
    tfStruct[i].id = i;
    windowInfo[i].thread = SbThread::create(threadRout, &tfStruct[i]);
    outputString1("\n(main) Created thread %d\n", i);
  }    
  
  /////////////////////////////////
  // creation of event thread
  /////////////////////////////////
  outputString0("\n(main) Start event loop\n");
#ifdef _WIN32
  // Note: For Win32, we need to run the event loop in the main thread
  // so it can process messages for all our rendering windows
  // (GetMessage retrieves messages for all windows owned by the
  // current thread.)
  eventLoop(NULL);
  
#else //UNIX
  SbThread * eventThread;
  eventThread= SbThread::create(eventLoop, NULL);
  if (eventThread == NULL){
    outputString0("Event thread not created\n");
  }   
  while (!timeToQuit) {
    sleep(1);
  }; 
  
#endif
  
  quitBarrier->enter();
  
  outputString0("\n(main) Exiting...\n");
  for (i = 0; i < nWin; i++) {
    if (windowInfo[i].thread) {
#ifdef _WIN32
      SbThread::destroy(windowInfo[i].thread);
#endif // on Unix pthread_exit called implicitely when threadrout return.
      outputString1("(main) Destroyed thread %d\n", i);
    }
  }    
  delete pCountMutex;
  if (pClearMutex) {
    delete pClearMutex;
  }
  if (pRenderBarrier) {
    delete pRenderBarrier;
  }
  if (quitBarrier) {
    delete quitBarrier;
  }
  SoInteraction::finish();
  SoDB::finish();
  return 0;
}


