/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

// MT_Modify1.cxx  (Multi-thread modify scene graph example 1)
//
// This example is based on Inventor Mentor example 13.1.GlobalFlds
//
// In that example a digital clock is implemented by connecting the
// RealTime global field to an SoText3 string field.
//
// In this example a digital clock is implemented by creating a
// separate thread which wakes up once every second and updates the
// string field of an SoText3 node.
//
// Equally pointless :-) but suggests some interesting possibilities.
//
// Important points:
//
// 1) Each additional thread that will use Open Inventor objects
//    must do per-thread initialization (for thread local storage).
//    Call SoDB::init in each addition thread.
//
// 2) This example uses Open Inventor's portable SbThread class to
//    create the thread, but this is only for convenience.
//
// 3) If more than one thread may be accessing the scene graph (for
//    example the viewer in the main thread may be rendering), in
//    order to safely _modify_ the scene graph the second thread
//    must gain exclusive access by calling SoDB::writelock.

////////////////////////////////////////////////////////////////////////

#include <Inventor/SoDB.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText3.h>

// Use Open Inventor's portable thread classes
#include <Inventor/threads/SbThread.h>

// Forward decls
void *threadRout(void *_userData);
void updateText(SoText3 *pTextNode);

////////////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
  // Initialize Open Inventor
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) {
    exit(1);     
  }
  
  // Create scene graph ------------------------------------------------
  SoSeparator *root = new SoSeparator;
  
  SoMaterial *myMaterial = new SoMaterial;
  myMaterial->diffuseColor.setValue(1.0, 0.0, 0.0);   
  root->addChild(myMaterial);
  
  SoText3 *myText = new SoText3;
  myText->string.setValue(SbTime::getTimeOfDay().formatDate());
  root->addChild(myText);
  
  // Create viewer -----------------------------------------------------
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Date & Time");
  myViewer->show();
  
  // Create thread to modify scene graph
  //
  // threadRout is the function the thread will execute
  // myText is passed to threadRout as userData
  SbThread *pModifyThread = SbThread::create(threadRout, (void*)myText);
  
  SoXt::show(myWindow);
  SoXt::mainLoop();

  SbThread::destroy(pModifyThread);

  SoXt::finish();

  return 0;
}

////////////////////////////////////////////////////////////////////////
// Modify the text string

void 
updateText(SoText3 *pTextNode)
{
  // Get date/time string
  SbString str = SbTime::getTimeOfDay().formatDate();
  
  // Request exclusive access to scene graph
  SoDB::writelock();
  
  // Modify text node
  pTextNode->string.setValue(str);
  
  // Release exclusive access as soon as possible
  SoDB::writeunlock();
}

////////////////////////////////////////////////////////////////////////
// Thread action function

void *
threadRout(void *userData)
{
  // Init of thread classes for asynchronous loading
  SoDB::init();

  // Get pointer to text node
  SoRef<SoText3> pTextNode = static_cast<SoText3 *>(userData);
  
  // Wakeup interval
  const int seconds = 1;
  
#if !defined(_WIN32)
  // On some UNIX systems, Open Inventor's automatic redraw on scene
  // graph change does not work reliably when the scene graph is
  // changed in a non-UI thread (no event loop).  This is an issue
  // with Xt timers that will be corrected in the next release.
  // For now, enabling the RealTimeSensor will ensure redraws happen.
  SoDB::enableRealTimeSensor(TRUE);
#endif
  
  // In this simple example the thread just runs continuously.
  // It will be cleaned up by the operating system when the program exits.
  while ( !SbThread::isStopRequested() )
  {
    SbTime::sleep(seconds*1000);
    updateText(pTextNode.ptr());
  }
  
  SoDB::finish();

  return NULL;
}
