/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM yyyy)
**=======================================================================*/

#include <Inventor/Qt/SoQt.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/touch/events/SoTouchEvent.h>
#include <Inventor/touch/devices/SoQtTouchScreen.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/sensors/SoIdleSensor.h>

#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/SoPickedPoint.h>

SoOrthographicCamera *myCamera;
SbViewVolume myViewVolume;
SoQtRenderArea *myRenderArea;

SoTransform *sphereTransform1, *sphereTransform2;

SoNode *node1, *node2;
SoPickedPoint *myPickedPoint;
unsigned int idSphere1, idSphere2;

SbVec2f sphereSpeed1;
SbVec2f sphereSpeed2;

SoIdleSensor *sensor;

SbName sphere1("Sphere1");
SbName sphere2("Sphere2");

SbTime currentTime;

void sensorCallback(void* /*data*/,  SoSensor *sensor)
{
  SbTime t = SbTime::getTimeOfDay();
  SbTime dt = t - currentTime;
  currentTime = t;
  SbVec2s size = myRenderArea->getSceneManager()->getSize();

  const SbViewportRegion &region = myRenderArea->getViewportRegion();
  float aspect = region.getViewportAspectRatio();

  myViewVolume = myCamera->getViewVolume( aspect );
  if ( aspect < 1 )
    myViewVolume.scale( 1 / aspect );


  //Window borders
  SbVec3f b1,b2;
  SbVec2f bordersInit(1,1);
  
  myViewVolume.projectPointToLine(bordersInit,b1,b2);

  SbVec3f borders;
  borders = (b1+b2)/2;

  sphereSpeed1 = sphereSpeed1 - 10.0*dt.getValue()*sphereSpeed1; 
  sphereSpeed2 = sphereSpeed2 - 10.0*dt.getValue()*sphereSpeed2; 

  SbVec2f d1 = sphereSpeed1 * dt.getValue();
  SbVec2f d2 = sphereSpeed2 * dt.getValue();

  SbVec2f normDisplacement1;
  normDisplacement1[0] = d1[0]/size[0];
  normDisplacement1[1] = d1[1]/size[1];

  SbVec2f normDisplacement2;
  normDisplacement2[0] = d2[0]/size[0];
  normDisplacement2[1] = d2[1]/size[1];

  SbVec3f vector1 = myViewVolume.getPlanePoint(0, normDisplacement1);
  SbVec3f vector2 = myViewVolume.getPlanePoint(0, normDisplacement2);
  SbVec3f vectorOrigin = myViewVolume.getPlanePoint(0, SbVec2f(0.0,0.0));

  SbVec3f pos1 = sphereTransform1->translation.getValue();
  sphereTransform1->translation.setValue(pos1 + vector1 - vectorOrigin);
  SbVec3f pos2 = sphereTransform2->translation.getValue();
  sphereTransform2->translation.setValue(pos2 + vector2 - vectorOrigin);

  float trans10 = sphereTransform1->translation.getValue()[0];
  float trans11 = sphereTransform1->translation.getValue()[1];
  float trans20 = sphereTransform2->translation.getValue()[0];
  float trans21 = sphereTransform2->translation.getValue()[1];

  // right and left rebound
  if( (trans10+1 >= borders[0] && sphereSpeed1[0]>0) || (trans10-1 <= -borders[0] && sphereSpeed1[0]<0)) 
  {
    sphereSpeed1[0] = -sphereSpeed1[0]; 
  }
  // top and down rebound
  if( (trans11+1 >= borders[1] && sphereSpeed1[1]>0) || (trans11-1 <= -borders[1] && sphereSpeed1[1]<0)) 
  {
    sphereSpeed1[1] = -sphereSpeed1[1]; 
  }
  if( (trans20+1 >= borders[0] && sphereSpeed2[0]>0) || (trans20-1 <= -borders[0] && sphereSpeed2[0]<0)) 
  {
    sphereSpeed2[0] = -sphereSpeed2[0]; 
  }
  if( (trans21+1 >= borders[1] && sphereSpeed2[1]>0) || (trans21-1 <= -borders[1] && sphereSpeed2[1]<0)) 
  {
    sphereSpeed2[1] = -sphereSpeed2[1]; 
  }

  if(sphereSpeed1.lengthSquared() > 2.0 || sphereSpeed2.lengthSquared() > 2.0)
  {
    sensor->schedule();
  }
  else if (sphereSpeed1.lengthSquared() < 2.0)
  {
    sphereSpeed1 = SbVec2f(0.0,0.0);
  }
  else
  {
    sphereSpeed2 = SbVec2f(0.0,0.0);
  }
}

// This routine is called for every mouse button event.
void
myTouchPressCB(void *userData, SoEventCallback *eventCB)
{
  SoSeparator *root = (SoSeparator *) userData;
  SoTouchEvent *event = (SoTouchEvent*)eventCB->getEvent();
  
  const SbViewportRegion &myRegion = eventCB->getAction()->getViewportRegion();
  
  SoRayPickAction myPickAction(myRegion);
  // Set an 8-pixel wide region around the pixel
  myPickAction.setPoint(event->getPosition(myRegion));
  myPickAction.setRadius(8.0);
  
  // Start a pick traversal
  myPickAction.apply(root);
  
  float aspect = myRegion.getViewportAspectRatio();

  SbVec2f pos = event->getPositionFloat();
  SbVec2s size = myRenderArea->getSceneManager()->getSize();
  pos[0] = pos[0]/size[0];
  pos[1] = pos[1]/size[1];

  myViewVolume = myCamera->getViewVolume( aspect );
  if ( aspect < 1 )
    myViewVolume.scale( 1 / aspect );

  SbVec3f vector = myViewVolume.getPlanePoint(0, pos);

  if(event->getState() == SoTouchEvent::DOWN) 
  {
    myPickedPoint = myPickAction.getPickedPoint();
    if(myPickedPoint != NULL)
    {
      if(node1 == NULL) {
        node1 = myPickedPoint->getPath()->getTail();
        idSphere1 = event->getFingerId();
      }
      else if (node2 == NULL) 
      {
        node2 = myPickedPoint->getPath()->getTail();
        idSphere2 = event->getFingerId();
      }
    }
  }

  else if(event->getState() == SoTouchEvent::UP)
  {
    if(node1 != NULL)
    {
      if(node1->getName() == sphere1) 
      {
        sphereSpeed1 = event->getSpeed();
        currentTime = SbTime::getTimeOfDay();
        sensor->schedule();
        idSphere1 = std::numeric_limits<unsigned int>::max();
        node1 = NULL;
        myPickedPoint = NULL;
      }
      else if(node1->getName() == sphere2) 
      {
        sphereSpeed2 = event->getSpeed();
        currentTime = SbTime::getTimeOfDay();
        sensor->schedule();
        idSphere1 = std::numeric_limits<unsigned int>::max();
        node1 = NULL;
        myPickedPoint = NULL;
      }
    }
    if(node2 != NULL)
    {
      if(node2->getName() == sphere1) 
      {
        sphereSpeed1 = event->getSpeed();
        currentTime = SbTime::getTimeOfDay();
        sensor->schedule();
        idSphere2 = std::numeric_limits<unsigned int>::max();
        node2 = NULL;
        myPickedPoint = NULL;
      }
      else if(node2->getName() == sphere2) 
      {
        sphereSpeed2 = event->getSpeed();
        currentTime = SbTime::getTimeOfDay();
        sensor->schedule();
        idSphere2 = std::numeric_limits<unsigned int>::max();
        node2 = NULL;
        myPickedPoint = NULL;
      }
    }
  }

  if((event->getState() != SoTouchEvent::UP ) && event->getFingerId() == idSphere1 ) 
  {
    if (node1->getName() == sphere1 )
      sphereTransform1->translation.setValue(vector);
    else
		  sphereTransform2->translation.setValue(vector);
	}
	else if((event->getState() != SoTouchEvent::UP ) && event->getFingerId() == idSphere2 ) 
  {
		if (node2->getName() == sphere2 )
			sphereTransform2->translation.setValue(vector);
		else
			sphereTransform1->translation.setValue(vector);
	}
  eventCB->setHandled();
}


int
	main(int, char **argv)
{
  idSphere1 = std::numeric_limits<unsigned int>::max();
  idSphere2 = std::numeric_limits<unsigned int>::max();
  // Print out usage instructions
  printf("\tTouch spheres to move them \n");
                                                            
  // Initialize Inventor and Xt
  QWidget *appWindow = SoQt::init(argv[0]);
  if (appWindow == NULL) exit(1);

  // Create and set up the root node
  SoSeparator *root = new SoSeparator;
  root->ref();

  // Add a camera
  myCamera = new SoOrthographicCamera;
  myCamera->position.setValue(0.0f, 0.0f, 4.0f);
  root->addChild(myCamera);
  root->addChild(new SoDirectionalLight);

  SoSeparator* separator = new SoSeparator;
  root->addChild(separator);

  // Sphere 1
  SoSeparator *sphereRoot1 = new SoSeparator; 
  SoMaterial *sphereMaterial1 = new SoMaterial;
  sphereTransform1 = new SoTransform;
  sphereRoot1->addChild(sphereTransform1);
  sphereRoot1->addChild(sphereMaterial1);

  sensor = new SoIdleSensor(sensorCallback, sphereTransform1);

  SoSphere* sp1 = new SoSphere;
  sp1->setName("Sphere1");
  sphereRoot1->addChild(sp1);
  sphereTransform1->translation.setValue(-4.0f, -4.0f, 0.0f);
  sphereMaterial1->diffuseColor.setValue(0.0f, 0.0f, 0.8f);
  separator->addChild(sphereRoot1);
	
  // Sphere 2
  SoSeparator *sphereRoot2 = new SoSeparator; 
  SoMaterial *sphereMaterial2 = new SoMaterial;
  sphereTransform2 = new SoTransform;
  sphereRoot2->addChild(sphereTransform2);
  sphereRoot2->addChild(sphereMaterial2);

  SoSphere* sp2 = new SoSphere;
  sp2->setName("Sphere2");
  sphereRoot2->addChild(sp2);
  sphereTransform2->translation.setValue(4.0f, 4.0f, 0.0f);
  sphereMaterial2->diffuseColor.setValue(0.0f, 0.8f, 0.0f);
  separator->addChild(sphereRoot2);
	
  myRenderArea = new SoQtRenderArea(appWindow);

  // Register the touch screen device.
  SoQtTouchScreen touchScreenDevice(appWindow);
  myRenderArea->registerDevice(&touchScreenDevice);

  myRenderArea->setSceneGraph(root);
  myRenderArea->setTitle("TouchInertia");

  myRenderArea->getSceneManager()->setRedrawPriority(0);

  SoEventCallback *myEventCB = new SoEventCallback;
  myEventCB->addEventCallback(SoTouchEvent::getClassTypeId(), myTouchPressCB, myRenderArea->getSceneManager()->getSceneGraph());
  root->addChild(myEventCB);

  // Make the camera see the whole scene
  SbViewportRegion viewportRegion = myRenderArea->getViewportRegion();
  myCamera->viewAll(root, viewportRegion, 2.0f);

  // Show our application window, and loop forever...
  myRenderArea->show();
  SoQt::show(appWindow);
  SoQt::mainLoop();

  root->unref();
  delete myRenderArea;
  SoQt::finish();
  return 0;
}




