/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM yyyy)
**=======================================================================*/

#include <Inventor/Win/SoWin.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/gestures/recognizers/SoScaleGestureRecognizer.h>
#include <Inventor/gestures/recognizers/SoRotateGestureRecognizer.h>
#include <Inventor/gestures/recognizers/SoDoubleTapGestureRecognizer.h>
#include <Inventor/gestures/recognizers/SoLongTapGestureRecognizer.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/touch/devices/SoWinTouchScreen.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoDrawStyle.h>

SoMaterial* coneMaterial;
SoWinRenderArea *myRenderArea;
SoOrthographicCamera *myCamera;
SoDrawStyle* drawStyle;
SoTransform *sphereTransform1;

void
myScaleCB(void *userData, SoEventCallback *eventCB)
{
  SoScaleGestureEvent *scaleEvent = (SoScaleGestureEvent*)eventCB->getEvent();
  if(scaleEvent != NULL)
  {
    const SbViewportRegion &region = myRenderArea->getViewportRegion();
    float aspect = region.getViewportAspectRatio();

    SbViewVolume myViewVolume = myCamera->getViewVolume( aspect );
    if ( aspect < 1 )
      myViewVolume.scale( 1 / aspect );
        
    SbVec2s size = myRenderArea->getSceneManager()->getSize();

    SbVec2f pos =  scaleEvent->getPositionFloat();
    SbVec2f normPosition;
    normPosition[0] = pos[0]/size[0];
    normPosition[1] = pos[1]/size[1];

    // realPosition is the zoom center
    SbVec3f realPosition = myViewVolume.getPlanePoint(0, normPosition);
    SbVec3f vector = myCamera->position.getValue() - realPosition;
    vector[2] = 0;

    // Camera is moved to the new position according to the scale factor
    myCamera->position.setValue(realPosition + vector/scaleEvent->getDeltaScaleFactor());
    myCamera->height.setValue(myCamera->height.getValue() / scaleEvent->getDeltaScaleFactor());

    eventCB->setHandled();
  }
}

void
myRotateCB(void *userData, SoEventCallback *eventCB)
{
  const SoRotateGestureEvent *rotateEvent = dynamic_cast<const SoRotateGestureEvent*>(eventCB->getEvent());
  float angle;
  SbVec3f axis;

  // Rotation of an object around some point : translation - rotation - reverse translation.
  if(rotateEvent != NULL) {

    if(rotateEvent->getGestureState() == SoGestureEvent::BEGIN)
    {
      drawStyle->style = SoDrawStyle::FILLED;
    }
    else if(rotateEvent->getGestureState() == SoGestureEvent::END)
    {
      drawStyle->style = SoDrawStyle::INVISIBLE;
    }

    myCamera->orientation.getValue().getValue(axis,angle);

    SbVec3f previousPositionCamera = myCamera->position.getValue();

    SbVec2s size = myRenderArea->getSceneManager()->getSize();
    const SbViewportRegion &region = myRenderArea->getViewportRegion();
    float aspect = region.getViewportAspectRatio();
    SbViewVolume myViewVolume = myCamera->getViewVolume( aspect );
    if ( aspect < 1 )
      myViewVolume.scale( 1 / aspect );

    SbVec2f pos = rotateEvent->getPositionFloat();
    pos[0] = pos[0] / size[0];
    pos[1] = pos[1] / size[1];

    // rotation center
    SbVec3f pointInvariant = myViewVolume.getPlanePoint(0, pos);

    // the vector from rotation center to camera position before rotation
    SbVec3f centreRotationToCentreCamera = previousPositionCamera - pointInvariant;
    centreRotationToCentreCamera[2] = 0;

    // To visualize the center
    sphereTransform1->translation.setValue(pointInvariant);

    // Camera rotation
    SbRotation rotation;
    if(axis[2] > 0) 
    {
      myCamera->orientation.setValue(axis, angle - rotateEvent->getDeltaRotation());
      rotation.setValue(axis, -rotateEvent->getDeltaRotation());
    }
    else 
    {
      myCamera->orientation.setValue(axis, angle + rotateEvent->getDeltaRotation());
      rotation.setValue(axis, rotateEvent->getDeltaRotation());
    }

    // Rotation of the vector according to the rotation angle.
    SbVec3f resultVector;
    rotation.multVec(centreRotationToCentreCamera,resultVector);
    // New camera poition
    myCamera->position.setValue(pointInvariant + resultVector);

    eventCB->setHandled();
  }
}

void
myDoubleTapCB(void *userData, SoEventCallback *eventCB)
{
  const SoDoubleTapGestureEvent *TapEvent = dynamic_cast<const SoDoubleTapGestureEvent*>(eventCB->getEvent());
  SbColor color(0.2f,0.2f,0.2f);
  SbColor a = coneMaterial->diffuseColor[0];
  if(TapEvent != NULL) 
  {
    if(coneMaterial->diffuseColor[0] != color )
    {
      coneMaterial->diffuseColor.setValue(color);
    }
    else
    {
      coneMaterial->diffuseColor.setValue(0.0f, 0.0f, 0.8f);
    }
    eventCB->setHandled();
  }
}

void
myLongTapCB(void *userData, SoEventCallback *eventCB)
{
  const SoLongTapGestureEvent *TapEvent = dynamic_cast<const SoLongTapGestureEvent*>(eventCB->getEvent());
  SbColor color(0.5f,0.2f,0.8f);
  SbColor a = coneMaterial->diffuseColor[0];
  if(TapEvent != NULL) 
  {
    if(coneMaterial->diffuseColor[0] != color )
    {
      coneMaterial->diffuseColor.setValue(color);
    }
    else
    {
      coneMaterial->diffuseColor.setValue(0.0f, 0.0f, 0.8f);
    }
    eventCB->setHandled();
  }
}

void myTouchCB(void *userData, SoEventCallback *eventCB) {
  SbViewVolume myViewVolume;
  const SoTouchEvent *touchEvent = dynamic_cast<const SoTouchEvent*>(eventCB->getEvent());

  if((touchEvent != NULL)
    && (touchEvent->getTouchManager()->getFingerNumber() < 2 ))
  {
    SbVec2s size = myRenderArea->getSceneManager()->getSize();

    const SbViewportRegion &region = myRenderArea->getViewportRegion();
    float aspect = region.getViewportAspectRatio();

    myViewVolume = myCamera->getViewVolume( aspect );
    if ( aspect < 1 )
      myViewVolume.scale( 1 / aspect );

    SbVec2f displacement =  touchEvent->getDisplacement();
    SbVec2f normDisplacement;
    normDisplacement[0] = displacement[0]/size[0];
    normDisplacement[1] = displacement[1]/size[1];

    SbVec3f vector = myViewVolume.getPlanePoint(0, normDisplacement);
    SbVec3f vectorOrigin = myViewVolume.getPlanePoint(0, SbVec2f(0.0,0.0));

    SbVec3f pos = myCamera->position.getValue();
    myCamera->position.setValue(pos + vectorOrigin - vector);

    eventCB->setHandled();
  }
}

int main(int, char **argv)
{
  SoScaleGestureRecognizer scaleRecognizer;
  SoRotateGestureRecognizer rotateRecognizer;
  SoDoubleTapGestureRecognizer doubleTapRecognizer;
  SoLongTapGestureRecognizer longTapRecognizer;

  // Print out usage instructions
  printf("\tTouch to manipulate\n");
  printf("Long tap / Double Tap \t\t Color changing \n");
  printf("Pinch \t\t\t\t Scale\n");
  printf("Two fingers \t\t\t Rotation around the z axis \n");

  // Initialize Inventor and Xt
  Widget appWindow = SoWin::init(argv[0]);
  if (appWindow == NULL) exit(1);

  // Create and set up the root node
  SoSeparator *root = new SoSeparator;
  root->ref();

  // Add a camera
  myCamera = new SoOrthographicCamera;
  myCamera->position.setValue(0.0f, 0.0f, 4.0f);
  root->addChild(myCamera);
  root->addChild(new SoDirectionalLight);

  // sphere showing the center of rotation 
  SoSeparator * sphere = new SoSeparator;
  root->addChild(sphere);

  drawStyle = new SoDrawStyle;
  drawStyle->style = SoDrawStyle::INVISIBLE;
  sphere->addChild(drawStyle);

  SoSphere* sp = new SoSphere;
  SoMaterial* sphereColor = new SoMaterial;
  sphereColor->diffuseColor.setValue(0.8f, 0.0f, 0.8f);
  sp->radius.setValue(0.1f);
  sphereTransform1 = new SoTransform;
  sphere->addChild(sphereTransform1);
  sphere->addChild(sphereColor);
  sphere->addChild(sp);

  // Cone
  coneMaterial = new SoMaterial;
  root->addChild(coneMaterial);
  
  SoCone* cone = new SoCone;
  root->addChild(cone);
  coneMaterial->diffuseColor.setValue(0.0f, 0.0f, 0.8f);

  myRenderArea = new SoWinRenderArea(appWindow);

  SoEventCallback *myEventCB = new SoEventCallback;
  myEventCB->addEventCallback(SoTouchEvent::getClassTypeId(), myTouchCB, myRenderArea->getSceneManager()->getSceneGraph());
  myEventCB->addEventCallback(SoScaleGestureEvent::getClassTypeId(), myScaleCB, myRenderArea->getSceneManager()->getSceneGraph());
  myEventCB->addEventCallback(SoRotateGestureEvent::getClassTypeId(), myRotateCB, myRenderArea->getSceneManager()->getSceneGraph());
  myEventCB->addEventCallback(SoDoubleTapGestureEvent::getClassTypeId(), myDoubleTapCB, myRenderArea->getSceneManager()->getSceneGraph());
  myEventCB->addEventCallback(SoLongTapGestureEvent::getClassTypeId(), myLongTapCB, myRenderArea->getSceneManager()->getSceneGraph());
  root->addChild(myEventCB);

  // Register the touch screen device.
  SoWinTouchScreen touchScreenDevice(appWindow);
  myRenderArea->registerDevice(&touchScreenDevice);

  touchScreenDevice.getTouchManager()->addRecognizer(&scaleRecognizer);
  touchScreenDevice.getTouchManager()->addRecognizer(&rotateRecognizer);
  touchScreenDevice.getTouchManager()->addRecognizer(&doubleTapRecognizer);
  touchScreenDevice.getTouchManager()->addRecognizer(&longTapRecognizer);
  myRenderArea->setSceneGraph(root);
  myRenderArea->setTitle("WinTouchEvent");

  // Make the camera see the whole scene
  SbViewportRegion viewportRegion = myRenderArea->getViewportRegion();
  myCamera->viewAll(root, viewportRegion, 2.0f);
  
  // Show our application window, and loop forever...
  myRenderArea->show();
  SoWin::show(appWindow);
  SoWin::mainLoop();

  root->unref();
  delete myRenderArea;
  SoWin::finish();

  return 0;
}
