/*=================================================================================
 ***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/elements/SoDepthBufferElement.h>
#include <Inventor/sys/SoGL.h>

#include "utils.h"

// This file provides an example of how to convert code that uses direct OpenGL calls
// to a code that only uses the Inventor API, especially Inventor Elements.
// Inventor Elements provide a set of classes with equivalents to most OpenGL features.
// In this example, the highlighted feature is about handling the depth test.

// Pointer to callback node
SoRef<SoCallback> g_callbackNode;

// This is the "OpenGL way" of setting the depth test parameters.
// As Open Inventor provides a way of setting the depth test function with SoDepthBufferElement,
// these GL calls should not be used anymore.
// The preffered implementation is proposed below.
// This function is meant to be be linked to the Callback node and called each time the node is traversed.
void
callbackGL( void*, SoAction* /*action*/ )
{
  glEnable( GL_DEPTH_TEST );
  glDepthMask( GL_TRUE );
  glDepthFunc( GL_ALWAYS );
  glDepthRange( 0.0, 1.0 );
}

// This is the implementation that should be used when the depth test parameters have to be changed.
// Generally, every direct OpenGL call should be avoided and replaced by its SoElement equivalent when available.
// This function will be linked to the Callback node and called each time the node is traversed.
void
callback( void*, SoAction* action )
{
  SoDepthBufferElement::set( action->getState(), TRUE, TRUE, SoDepthBufferElement::ALWAYS, SbVec2f(0.0f, 1.0f) );
}

int
main ( int /*argc*/, char** argv )
{
  Widget myWindow = SoXt::init( argv[0] );
  if ( myWindow == NULL )
    exit( 1 );

  SbString filename( "$OIVHOME/examples/source/Inventor/OpenGL/Callback/Callback.iv" );

  SoRef<SoSeparator> root = readFile( filename );

  // Find the callback node named "CALLBACK" in the scenegraph
  g_callbackNode = static_cast<SoCallback*>( SoNode::getByName("CALLBACK") );
  if ( g_callbackNode.ptr() == NULL )
  {
    std::cerr << "Callback node not not found in " << filename << std::endl;
    return EXIT_FAILURE;
  }

  // Here we link the callback node with the callback function containing our custom code.
  g_callbackNode->setCallback( callback );

  // Set up viewer:
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer( myWindow );
  viewer->setSceneGraph( root.ptr() );
  viewer->viewAll();
  viewer->setTitle( "Callback with Elements" );
  viewer->show();

  SoXt::show( myWindow );

  SoXt::mainLoop();

  delete viewer;

  root = NULL;

  SoXt::finish();

  return EXIT_SUCCESS;
}
