/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
//
// Some code for setting up and dealing with the profile window.
//

#include <stdio.h>
#include <getopt.h>
#include <math.h>
#include <malloc.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/sensors/SoSensor.h>

#include <Xm/Text.h>

#include "LineManip.h"
#include "Useful.h"
#include "RevClass.h"

//
// The line manipulator.
//
static LineManip2 *TheLineManip;

//
// The default coordinates for the line manipulator.
//
static float defaultCoords[2][3] =
{
    { 0.5F,  0.2F, 0.0F },
    { 0.5F, -0.2F, 0.0F },
};

//
// Grid for window:
//
static const char *GridString =
"#Inventor V2.0 ascii\n"
"Separator {"
"	PickStyle { style UNPICKABLE }"
"	LightModel { model BASE_COLOR }"
"	BaseColor { rgb 0.2 0.2 0.2 }"
"	Array {"
"		numElements1 19"
"		separation1 .1 0 0"
"		origin CENTER"
"		Coordinate3 { point [ 0 -1 0, 0 1 0 ] }"
"		LineSet { numVertices [ 2 ] }"
"	}"
"	Array {"
"		numElements1 19"
"		separation1 0 .1 0"
"		origin CENTER"
"		Coordinate3 { point [ -1 0 0, 1 0 0 ] }"
"		LineSet { numVertices [ 2 ] }"
"	}"
"	BaseColor { rgb 0.4 0.0 0.0 }"
"	Coordinate3 { point [ -1.0 0.0 0.0, 1.0 0.0 0.0,"
"			0.0 -1.0 0.0, 0.0 1.0 0.0 ]"
"	}"
"	LineSet { numVertices [ 2, 2 ] }"
"}";

//
// This is called when the mouse leaves the profile window.  It turns
// off the line manip's hilights.
//
static void
leaveProfileWindow(Widget, XtPointer mydata, XEvent *, Boolean *cont)
{
    ((LineManip2 *)mydata)->removeHilights();

    *cont = TRUE;	// Continue dispatching this event
}

//
// Called to get rid of all the points added so far.
//
void
clearPoints()
{
    SoCoordinate3 *coord = TheLineManip->getCoordinate3();
    coord->point.deleteValues(0);
}

//
// Called whenever the profile is edited.  It tells the surface to
// recalculate itself.  Note that this could be more efficient by
// noticing exactly what changed and just recomputing what is
// necessary (e.g. when adding a point to the end, only two sets of
// normals needs to be generated, etc).
//
static void
profileChangedCallback(void *data, SoSensor *)
{
    RevolutionSurface *s = (RevolutionSurface *)data;

    SoCoordinate3 *coord = TheLineManip->getCoordinate3();
    s->createSurface(&coord->point);
}

//
// This creates the profile's scene graph and sets up the callback so
// the line manipulator turns off its hilights when the mouse leaves
// the profile window.
//
SoNode *
createProfileGraph(Widget profileWidget, RevolutionSurface *surface)
{
    SoSeparator *result = new SoSeparator;
    result->ref();

    SoOrthographicCamera *c = new SoOrthographicCamera;
    result->addChild(c);
    c->position.setValue(0.5, 0.0, 1.0);
    c->nearDistance = 0.5;
    c->farDistance = 1.5;
    c->focalDistance = 1.0;	
    c->height = 1.0;

    // Axes and labels (appear underneath the line manipulator)
    SoInput in;
    in.setBuffer((void *)GridString, strlen(GridString));
    SoNode *node;	
    SoDB::read(&in, node);
    SoSeparator *g1 = (SoSeparator *)node;
    if (g1 != NULL)
    {
	result->addChild(g1);
   	g1->boundingBoxCaching = SoSeparator::ON;
    }

    SoSeparator *manipGroup = new SoSeparator;
    result->addChild(manipGroup);

    SoLightModel *lm = new SoLightModel;
    manipGroup->addChild(lm);
    lm->model = SoLightModel::BASE_COLOR;
 
    SoMaterial *m = new SoMaterial;
    m->diffuseColor.setValue(0.0F, 0.8F, 0.8F);
    manipGroup->addChild(m);

    TheLineManip = new LineManip2;
    TheLineManip->setHilightSize(0.02F);
    manipGroup->addChild(TheLineManip);
    SoCoordinate3 *coord = TheLineManip->getCoordinate3();
    SoNodeSensor *d = new SoNodeSensor(profileChangedCallback, surface);
    d->attach(coord);
    coord->point.setValues(0, 2, defaultCoords);

    //
    // Make sure surface of revolution and line manip both agree on
    // the default coordinates
    //
    surface->createSurface(&coord->point);

    XtAddEventHandler(profileWidget, LeaveWindowMask,
		      FALSE, leaveProfileWindow, (XtPointer)TheLineManip);

    result->unrefNoDelete();

    return result;
}

