//!oiv_include <Inventor/oivShaderState.h>
//!oiv_include <Inventor/oivShaderVariables.h>
//!oiv_include <Inventor/oivDepthPeeling_frag.h>

uniform float  bumpOn;    // true if bumpmapping is activated
uniform float bumpDepth; // control the normal transformation scale

uniform mat4 ModelToWC;  // modeling matrix (object to world space)

//uniform float     dimTexture;        // length of the texture (to calculate a 1-pixel offset)
uniform float     texCoordFrequency; // frequency of repetition of the texture
uniform sampler2D textureMap;
uniform sampler2D heightMap;

uniform float specularIntensity;

uniform samplerCube envMap;             // environment cube map

in vec3 eNormal;       // Normal in eye space
in vec3 eLightv;       // Vertex to light vector in eye space
in vec3 wReflectance;  // Reflected ray direction in world space
in vec3 wIncident;     // Incident ray direction in world space
in vec4 constColor;    // Non-bumpmapping dependent color (emissive and ambient)

void main()
{
  if (OivDepthPeel(gl_FragCoord.xyz))
  {
    float       dimTexture = 512.0;

    // writing to in/out variables into fragment shaders is illegal,
    // so we create a (f-prefixed) copy of those we write to.
    vec3 feNormal       = eNormal;
    vec3 feLightv       = eLightv;
    vec3 fwReflectance  = wReflectance;

    if(dot(feNormal, feLightv) <= 0.0) {
      // The fragment is not lighted before normal pertubations, we paint it black.
      // No need to normalize feNormal or feLightv first.
      vec4 color   = vec4(0.0, 0.0, 0.0, 1.0);
      color.a = OivFrontMaterialDiffuse().a;
      OivDepthPeelingOutputColor(color);
    }

    // gl_FrontFacing is not available with nVidia GLSL drivers yet
    // else if(!gl_FrontFacing) {
    //  OivDepthPeelingOutputColor(vec4(0.0));
    // }

    else {
      vec2 newTexCoord = OivFragmentTexCoord(0).st * vec2(texCoordFrequency);

      feLightv = normalize(feLightv);
      feNormal = normalize(feNormal);
  
      float diffuseCoeff;
      float specularCoeff;

      if(bumpOn!=0) {
        float Hg, Ha, Hr;
        float decalOnePixel = 1.0/dimTexture;
        Hg = texture(heightMap, newTexCoord).r;                            // height of the given texel
        Ha = texture(heightMap, newTexCoord + vec2(0.0, decalOnePixel)).r; // height of the texel directly above the given texel
        Hr = texture(heightMap, newTexCoord + vec2(decalOnePixel, 0.0)).r; // height of the texel directly to the right of the given texel

        vec3 tNormal; // normal in texture space
        tNormal = vec3(bumpDepth * (Hg-Hr), bumpDepth * (Hg-Ha), 1.0);
        tNormal = normalize(tNormal);

        // the following code are far-fetched computations to get an estimation of the direction of 
        // the s direction (evolution of the s texture coordinate) in eye space, to get a consistent tangent system
        vec2 deTangent    = vec2(dFdx(newTexCoord.s),  dFdy(newTexCoord.s));
        vec2 deZ          = vec2(dFdx(gl_FragCoord.z), dFdy(gl_FragCoord.z));
        vec3 tempeTangent = vec3(deTangent, deTangent.x * -deZ.x + deTangent.y * -deZ.y);
        vec3 temp         = cross(tempeTangent, feNormal);
        vec3 eTangent     = cross(feNormal, temp);

        eTangent  = normalize(eTangent);
        vec3 eBinormal = cross(feNormal, eTangent);

        vec3 tLightv; // vertex to light vector in texture space
        tLightv.x = dot(feLightv, eTangent);
        tLightv.y = dot(feLightv, eBinormal);
        tLightv.z = dot(feLightv, feNormal);
        tLightv   = normalize(tLightv);

        diffuseCoeff  = max(dot(tNormal, tLightv), 0.0);
        specularCoeff = diffuseCoeff; // cause light and eye are the same

        // Readjust normal in eye coordinate
        feNormal = vec3(tNormal.x) * eTangent 
               + vec3(tNormal.y) * eBinormal 
               + vec3(tNormal.z) * feNormal;

        // Compute reflected ray
        mat3 normalMatrixInverse = 
        mat3(OivModelViewMatrix()[0][0], OivModelViewMatrix()[1][0], OivModelViewMatrix()[2][0], 
            OivModelViewMatrix()[0][1], OivModelViewMatrix()[1][1], OivModelViewMatrix()[2][1],
            OivModelViewMatrix()[0][2], OivModelViewMatrix()[1][2], OivModelViewMatrix()[2][2]);
        vec3 oNormal  = normalize(normalMatrixInverse * feNormal);
        vec3 wNormal  = normalize(mat3(ModelToWC) * oNormal);
        fwReflectance = reflect(wIncident, wNormal);
      }
      else {
        diffuseCoeff  = max(dot(feNormal, feLightv), 0.0);
        specularCoeff = diffuseCoeff;
      }
    
      // Computation of final colors
      vec4 diffuseColor, reflectedColor, specularColor;
      diffuseColor  = texture(textureMap, newTexCoord)
                    * OivFrontMaterialDiffuse()
                    * OivLightSourceDiffuse(0)
                    * vec4(diffuseCoeff);

      specularColor = vec4(specularIntensity)
                    * vec4(pow(specularCoeff, OivFrontMaterialShininess()));


      reflectedColor = texture(envMap, fwReflectance);

      vec4 color = constColor + mix(diffuseColor, reflectedColor, OivFrontMaterialSpecular()) + specularColor;;
      color.a = OivFrontMaterialDiffuse().a;
      OivDepthPeelingOutputColor(color);
    }
  }
}
