/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM 2004)
**=======================================================================*/

#include "dialog.h"

#include <Inventor/actions/SoWriteAction.h> 
#include "misc.h"
#include "auditors.h"
#include "materialEditor.h"
#include "main.h"

SoTopLevelDialog * TopLevelDialog;
SoColumnDialog   * DialogComponents;
SoDialogCustom   * WidgetCustomDialog;
SoTabDialog      * ShaderTabDialog;
SoRowDialog      * ShaderParamDialog;
SoRowDialog      * ShaderIVFile;
SoMenuBar        * MenuBar;
SoMenuCheckBox   * ShowMaterialEditorButton;

/****************************************************************************/
// Initialize the tab dialog (all the dialog components shader-dependant).
void
initShaderTabDialog()
{
  // Content of the shader .iv file
  ShaderIVFile = new SoRowDialog;
  ShaderIVFile->label = "-";
  
  // Parameters of the current shader
  ShaderParamDialog = new SoRowDialog;
  ShaderParamDialog->label = "Parameters";
  ShaderParamDialog->fixedHeight = FALSE;
  ShaderParamDialog->maxWidth = DIALOG_COMPONENTS_MAX_WIDTH;
  
  ShaderTabDialog = new SoTabDialog;
  ShaderTabDialog->setName(DIALOGPART_NAME);
  ShaderTabDialog->addChild(ShaderIVFile);
  ShaderTabDialog->addChild(ShaderParamDialog);
}//---------------------------------------------------------------------------

/****************************************************************************/
// Empty all shader-dependent data of the Dialog tabs.
void
resetShaderTabDialog()
{
  ShaderTabDialog->selectedPage = 0;
  int indChild;
  while((indChild = ShaderTabDialog->getNumChildren()) > 2)
    ShaderTabDialog->removeChild(indChild-1);
  ShaderParamDialog->removeAllChildren();
  ShaderParamDialog->fixedHeight = FALSE;
  
  ShaderIVFile->removeAllChildren();
  ShaderIVFile->label = "-";
}//---------------------------------------------------------------------------

/****************************************************************************/
// Initialize the menu bar.
void
initMenuBar()
{
  // Load shader
  SoMenuFileSelection * loadShaderButton = new SoMenuFileSelection;
  loadShaderButton->label = "Load shader...";
  loadShaderButton->filter.set1Value(0, "*.iv");
  loadShaderButton->filter.set1Value(1, "IV shader description");
  loadShaderButton->fileDirectory = "$OIVHOME/examples/source/Inventor/ShadersBrowser/Shaders";
  loadShaderButton->addAuditor(new ChangeShaderButtonAuditor());

  // Load model
  SoMenuFileSelection * loadModelButton = new SoMenuFileSelection;
  loadModelButton->label = "Load model...";
  loadModelButton->filter.set1Value(0, "*.iv");
  loadModelButton->filter.set1Value(1, "IV model description");
  loadModelButton->fileDirectory = "$OIVHOME/examples/source/Inventor/ShadersBrowser/Models";
  loadModelButton->addAuditor(new ChangeModelButtonAuditor());

  // Write scene
  SoMenuPushButton * writeSceneButton = new SoMenuPushButton;
  writeSceneButton->label = "Write Scene";
  writeSceneButton->addAuditor(new WriteButtonAuditor());
  
  // Exit
  SoMenuPushButton * exitButton = new SoMenuPushButton;
  exitButton->label = "Exit";
  exitButton->addAuditor(new ExitButtonAuditor());
  
  // FILE Popup
  SoMenuPopup * fileMenuPopup = new SoMenuPopup;
  fileMenuPopup->label = "File";
  fileMenuPopup->addChild(loadShaderButton);
  fileMenuPopup->addChild(loadModelButton);
  fileMenuPopup->addChild(writeSceneButton);
  fileMenuPopup->addChild(exitButton);
  
  // Shader source
  SoMenuCheckBox * showDialogButton = new SoMenuCheckBox;
  showDialogButton->label = "Shader source";
  showDialogButton->state = FALSE;
  showDialogButton->addAuditor(new ShowDialogButtonAuditor());
  
  // Show Material Editor
  ShowMaterialEditorButton = new SoMenuCheckBox;
  ShowMaterialEditorButton->label = "Material editor";
  ShowMaterialEditorButton->state = FALSE;
  ShowMaterialEditorButton->addAuditor(new ShowMaterialEditorButtonAuditor());

  // OPTIONS Popup
  SoMenuPopup * optionsMenuPopup = new SoMenuPopup;
  optionsMenuPopup->label = "Options";
  optionsMenuPopup->addChild(showDialogButton);
  optionsMenuPopup->addChild(ShowMaterialEditorButton);
  
  // Returned menu bar
  MenuBar = new SoMenuBar;
  MenuBar->addChild(fileMenuPopup);
  MenuBar->addChild(optionsMenuPopup);
}//---------------------------------------------------------------------------

/****************************************************************************/
// The dialog part of the application lies on two different parts :
// - the tab dialog in which we get the parameters control components of the
//   shader, the source code of the shader and the content of its .iv file.
// - the menu bar, which allows to load shaders, models and to control if 
//   the tab dialog and/or the material editor shall be shown...
SoTopLevelDialog * 
initDialog()
{
  initShaderTabDialog();
  initMenuBar();
  
  // Widget for the examiner
  // (in order to have the viewer and the tab dialog in the same window, we
  // have to create the following DialogCustom to have a zone where to set
  // the viewer)
  WidgetCustomDialog = new SoDialogCustom;
  WidgetCustomDialog->width  = 800;
  WidgetCustomDialog->height = 600;
  
  DialogComponents = new SoColumnDialog;
  // By default, we only add the viewer part in this column dialog, but il 
  // will also contain the tab dialog when it is shown.
  DialogComponents->addChild(WidgetCustomDialog);
  
  TopLevelDialog = new SoTopLevelDialog;
  TopLevelDialog->addChild(DialogComponents);
  TopLevelDialog->addChild(MenuBar);
  
  return TopLevelDialog;
}//---------------------------------------------------------------------------

/****************************************************************************/
// Indicates if the dialog component implies that a MaterialEditor is queried
// or not.
SbBool
needMaterialEditor(SoDialogComponent * comp)
{
  if(!comp) {
#if defined(_DEBUG)
    SoDebugError::post("needMaterialEditor", "The given parameter is NULL.");
#endif
    return FALSE;
  }
  return (comp->auditorID.getValue() == MATERIAL_AUDITOR_ID);
}//---------------------------------------------------------------------------

/****************************************************************************/
// Add a slider in the Parameter tab.
void
addParamControl(SoDialogComponent * paramControl)
{
  if(!paramControl) {
#if defined(_DEBUG)
    SoDebugError::post("addParamControl", "The given parameter is NULL.");
#endif
    return;
  }

  ShaderParamDialog->addChild(paramControl);

  if(needMaterialEditor(paramControl)) {
    if(dynamic_cast<SoDialogRadioButtons *>(paramControl)) {
      ((SoDialogRadioButtons *)paramControl)->addAuditor(new MaterialEditorChoiceAuditor());
    }
    setMaterialToAttach(getLastMaterialNode());
    attachMaterialEditor();
    showMaterialEditor();
  }
}//---------------------------------------------------------------------------

/****************************************************************************/
// Add a tab whcih contains the shader's glsl source code.
// The title of the tab is set to the filename of this source code.
void
addShaderObjectSourceFile(SbString filename, SbString sourceProgram)
{
  SoDialogEditText * source = new SoDialogEditText;
  source->editText = sourceProgram;
  source->editable = FALSE;
  source->multiLine = TRUE;
  source->wrapLine = FALSE;
  source->minWidth = DIALOG_COMPONENTS_MIN_WIDTH;
  source->maxWidth = DIALOG_COMPONENTS_MAX_WIDTH;
  
  SoRowDialog * newTab = new SoRowDialog;
  newTab->label = filename;
  newTab->addChild(source);
  
  ShaderTabDialog->addChild(newTab);
}//---------------------------------------------------------------------------

size_t g_currentBufferSize =0;

/****************************************************************************/
// Reallocation function used by setShaderIVFileContent.
void *
ouputReallocCB(void * ptr, size_t newSize)
{
  char * prevBuffer = (char *)ptr;
  char * newBuffer = new char[newSize];
  memcpy(newBuffer, prevBuffer,g_currentBufferSize);
  g_currentBufferSize = newSize;
  delete [] prevBuffer;
  return newBuffer;
}//---------------------------------------------------------------------------

/****************************************************************************/
// Write the iv file format of the given subgraph.
// The filename is just indicated in the tab label as an indication cause what
// is displayed may be slighty different of the effective content of the 
// shader iv file.
void
setShaderIVFileContent(SoSeparator * shader, SbString filename)
{
  if(!shader) {
#if defined(_DEBUG)
    SoDebugError::post("setShaderIVFileContent", "The given shader parameter is NULL.");
#endif
    return;
  }

  // Write the shader in a buffer
  SoWriteAction myAction;
  char * buffer = new char[4096];
  g_currentBufferSize = 4096;
  myAction.getOutput()->setBuffer(buffer, g_currentBufferSize, ouputReallocCB);
  myAction.getOutput()->setBinary(FALSE);
  myAction.apply(shader);
  
  // Get the number of written char
  size_t nBytes = 0;
  myAction.getOutput()->getBuffer((void *&)buffer, nBytes);
  
  // Create the edit text and add it to the dialog part
  SoDialogEditText * writtenShader = new SoDialogEditText;
  writtenShader->editText = buffer;
  writtenShader->editable = FALSE;
  writtenShader->multiLine = TRUE;
  writtenShader->minWidth = DIALOG_COMPONENTS_MIN_WIDTH;
  writtenShader->maxWidth = DIALOG_COMPONENTS_MAX_WIDTH;
  writtenShader->wrapLine = FALSE;
  
  delete [] buffer;
  
  ShaderIVFile->removeAllChildren();
  ShaderIVFile->addChild(writtenShader);
  ShaderIVFile->label = (!filename ? SbString("-") : removeDirectories(filename));
}//---------------------------------------------------------------------------

/****************************************************************************/
void
showShaderTabDialog()
{
  DialogComponents->addChild(ShaderTabDialog);
}//---------------------------------------------------------------------------

/****************************************************************************/
void
hideShaderTabDialog()
{
  ShaderTabDialog->ref();
  DialogComponents->removeChild(ShaderTabDialog);
  ShaderTabDialog->unrefNoDelete();
}//---------------------------------------------------------------------------

/****************************************************************************/
// Return the widget that must be used by a viewer to render the scene.
Widget
getSceneWidget()
{
  return WidgetCustomDialog->getWidget();
}//---------------------------------------------------------------------------

/****************************************************************************/
void
setTitleBar(SbString newTitle)
{
  if(!newTitle)
    TopLevelDialog->label = TITLE;
  else         
    TopLevelDialog->label = newTitle.getString();
}//---------------------------------------------------------------------------

/****************************************************************************/
void
setMaterialEditorCheckboxState(SbBool val)
{
  ShowMaterialEditorButton->state = val;
}//---------------------------------------------------------------------------

/****************************************************************************/
void
setMaterialEditorCheckboxEnablement()
{
  if(isMaterialEditorAttached()) {
    ShowMaterialEditorButton->enable = TRUE;
  }
  else {
    ShowMaterialEditorButton->enable = FALSE;
  }
}//---------------------------------------------------------------------------

/****************************************************************************/
void
errorMessageDialog(SbString message)
{
  SoMessageDialog * errorDialog = new SoMessageDialog;
  errorDialog->title = ERROR_TITLE;
  errorDialog->type  = SoMessageDialog::MD_WARNING;
  errorDialog->label = message;
  errorDialog->show();
}//---------------------------------------------------------------------------

