/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM 2004)
**=======================================================================*/

//
// The shaders nodes are stored in an iv file.
// There are predefined parameters which can be used by 
// setting their name field to one of the following:
// 
// 			available parameters:
// 
// 				"ModelToWC"     : modeling Matrix
// 				"EyePosition"   : eye position
// 				"Time"	         : current time
//
// To be considered as valid, a shader must contain two key nodes:
// - a SHADER group node, which can contain all necessary nodes to
//   make the shader work correctly (textures, etc...) and must
//   itself contain:
// - a SHADERPROGRAM node which is the effective SoShaderProgram node
//   of the shader file.
// 
// To insert DialogViz elements in the GUI (to control your parameters' 
// values for example), the .iv file must also contain a 
// DIALOG_COMPONENTS group node which can only contain SoDialogComponent
// or derived nodes.
//
// All connections between DialogViz and ShaderParameter components must be
// made directly in the iv file.
//
// The following example shows how to control an SoShaderParameter1{if} 
// node's value using the .iv file.
//
// Example:
//
//    (...)
//    DEF MySlider DialogRealSlider {
//      label "Value:"
//      value 0.5
//    }
//    (...)
//    ShaderParameter1f {
//      name "myParamName"
//      value = USE MySlider.value
//    }
//    (...)
//
// This mechanism can also be set for SoShaderParameter{234}{if} nodes.
//
// Example:
//
//    (...)
//    DEF MySliderX DialogRealSlider{
//      label  "val1"
//      value 1.0
//    }
//    DEF MySliderY DialogRealSlider{
//      label  "val2"
//      value 1.0
//    }
//    (...)
//    ShaderParameter2f {
//      name "paramName"
//      value = ComposeVec2f { x = USE MySliderX.value
//                             y = USE MySliderY.value }.vector
//    }
//    (...)
//
// You can also use radio buttons, check boxes, or other DialogViz components
// depending on the ability you have to connect one of its fields to your
// ShaderParameter node's value field.
//
// BE CAREFUL :
// As all the children of the DIALOG_COMPONENTS are inserted in the  
// 'Parameters' tab, you need to take special care to prevent any useless
// component (i.e. a component connected to nothing) or any other node than an 
// SoDialogComponent-derived node from appearing in the DIALOG_COMPONENTS part
// of the .iv file.
//
// If a MaterialEditor is needed, the dialogComponent which will be 
// associated with it must have its auditorID set to "materialEditor".
// The MaterialEditor will control the value of the last SoMaterial node
// in the SHADER group node.
//
//////////////////////////////////////////////////////////////////////////////

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/SoPath.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/antialiasing/SoAccumulationAntialiasingParameters.h>

#include "dialog.h"
#include "environment.h"
#include "misc.h"
#include "model.h"
#include "shaders.h"
#include "materialEditor.h"

SbString ModelIv = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/ShadersBrowser/Models/teapot.iv");
SbString ShaderInit = SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/ShadersBrowser/Shaders/GenericBumpMapShader.iv");

/****************************************************************************/
// Global variables
SoPerspectiveCamera			* Camera;
SoSeparator					    * SceneSep, * ShaderSep, * root;
//----------------------------------------------------------------------------

/****************************************************************************/
// Provide a way to access this file's global variables into the other files
// without having to use the 'extern' keyword...
SoSeparator * getRoot()      { return root;      }
SoSeparator * getShaderSep() { return ShaderSep; }
SoSeparator * getSceneSep()  { return SceneSep;  }
/****************************************************************************/

/****************************************************************************/
void
connectToEyePosition(SoShaderParameter3f * param)
{
  param->value.connectFrom(&Camera->position);
}//---------------------------------------------------------------------------

/****************************************************************************/
SoMaterial *
getLastMaterialNode()
{
  SoSearchAction searchAction;
  searchAction.setType(SoMaterial::getClassTypeId());
  searchAction.setInterest(SoSearchAction::LAST);
  searchAction.apply(ShaderSep);
  SoPath * pathFound = searchAction.getPath();
  if(pathFound) {
    return ((SoMaterial *)pathFound->getNodeFromTail(0));
  }
  return NULL;
}//---------------------------------------------------------------------------

/****************************************************************************/
void
updateShaderInGraph(SoSeparator * toInsert)
{
  toInsert->ref();
  ShaderSep->removeAllChildren();
  int numChildren = toInsert->getNumChildren();
  for(int i=0; i<numChildren; i++)
    ShaderSep->addChild(toInsert->getChild(i));
  ShaderSep->addChild(SceneSep);
  toInsert->unref();
}//---------------------------------------------------------------------------

/****************************************************************************/
// Get the fps rate
void
fpsCB(float fps, void *, SoXtViewer* /*viewer*/)
{
  SbString titleStr(TITLE) ;
  SbString fpsStr((int)fps) ;
  titleStr += " : " ;
  titleStr += fpsStr ;
  titleStr += " fps" ;	
  setTitleBar(titleStr);
}//---------------------------------------------------------------------------

/****************************************************************************/
// Main function
int
main(int, char **argv)
{
  // Initialize Inventor, Xt and DialogViz
  Widget myWindow = SoXt::init(argv[0]);
  SoDialogViz::init();
  /*
  if(!isGLSLSupported()) {
    errorMessageDialog("GLSL is not supported by your graphics board!");
  }
  */

  root = new SoSeparator;
  root->ref() ;
  
  Camera = new SoPerspectiveCamera;
  
  SoComplexity * comp = new SoComplexity ;
  comp->textureQuality = 1.0f ;
  
  // Shader Parameters
  initGlobalShaderParameters();
  // Build DialogViz interface
  SoTopLevelDialog * topLevelDialog = initDialog();
  // Material Editor
  initMaterialEditor();
  
  SoSeparator * environment = loadEnvironment();
  SceneSep = loadModel(ModelIv);
    
  // The separator where the shader program (and the others 
  // necessary nodes) and  the model have to be set.
  ShaderSep = new SoSeparator;
  
  // Build the scene graph
  SoSeparator * scene3D = new SoSeparator ;
  scene3D->addChild(Camera) ;
  scene3D->addChild(comp);
  scene3D->addChild(environment);
  scene3D->addChild(ShaderSep);
  root->addChild(scene3D) ;
  
  shaderInit(readFile(ShaderInit), ShaderInit);
  topLevelDialog->buildDialog(myWindow, TRUE);
  topLevelDialog->show();
  
  // Attach and show viewer
  SoXtExaminerViewer * eViewer = new SoXtExaminerViewer(getSceneWidget(), "", TRUE);
  eViewer->setTransparencyType(SoGLRenderAction::NO_SORT);
  eViewer->setSceneGraph(root);
  eViewer->setHeadlight(TRUE);
  //eViewer->setFramesPerSecondCallback(fpsCB);

  SoAccumulationAntialiasingParameters params(TRUE, 1);
  eViewer->setAntialiasing(&params);

  eViewer->bindNormalContext();
  bool isSupported = isGLSLSupported();
  eViewer->unbindNormalContext();

  if(!isSupported)
  {
    errorMessageDialog("GLSL is not supported by your graphics board!");
  }
  else
  {
    eViewer->show();

    //Set initial camera position
    Camera->position.setValue( -346.201f, 31.7967f, -190.293f) ;
    Camera->orientation.setValue(SbVec3f(-0.00644153f, -0.999976f, -0.00242554f),  2.0691f) ;
    Camera->nearDistance = 0.940594f;
    Camera->farDistance  = 942.477f;
    Camera->focalDistance = 390.934f;

    // Loop forever
    SoXt::show(myWindow);
    SoXt::mainLoop();
  }
  // cleanup allocated resources
  root->unref();
  delete eViewer;

  SoDialogViz::finish();
  SoXt::finish();

  return EXIT_SUCCESS;
}//---------------------------------------------------------------------------


