/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM 2004)
**=======================================================================*/

#include "shaders.h"

#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoGetMatrixAction.h>
#include <Inventor/sensors/SoIdleSensor.h>
#include <Inventor/nodes/SoShaderProgram.h>

#include "dialog.h"
#include "model.h"
#include "misc.h"
#include "materialEditor.h"
#include "main.h"

/****************************************************************************/
// Global shader parameters
SoShaderParameterMatrix	* ModelToWC, * ModelToWCInv;
SoShaderParameter1f			* TimeValueParameter;
SoIdleSensor            * TimeValueSensor;

/****************************************************************************/
// Time sensor
void 
mysensorCB(void* /*data*/, SoSensor * sensor)
{
  static int tps = 0;
  TimeValueParameter->value = static_cast<float>(tps++);
  sensor->schedule();
}//---------------------------------------------------------------------------

/****************************************************************************/
// Initialize global shader parameters
void
initGlobalShaderParameters()
{
  ModelToWC    = new SoShaderParameterMatrix;
  ModelToWCInv = new SoShaderParameterMatrix;
  TimeValueParameter    = new SoShaderParameter1f;
  TimeValueSensor = new SoIdleSensor(mysensorCB, NULL);
  TimeValueSensor->schedule();
}//---------------------------------------------------------------------------

/****************************************************************************/
// Check if the given group is a valid shader i.e. contains two nodes named 
// with special keywords TOINSERT_NAME and SHADERPROG_NAME
SbBool 
isValidShader(SoSeparator * sep, SbBool test) 
{
  sep->ref();
  SbBool toInsertFound = FALSE;

  {
    SoSearchAction searchAction1;
    searchAction1.setName(TOINSERT_NAME);
    searchAction1.apply(sep);
    toInsertFound = searchAction1.getPath() != NULL;
  }

  if(toInsertFound) {
    SbBool shaderProgName = FALSE;
    {
      SoSearchAction searchAction2;
      searchAction2.setName(SHADERPROG_NAME);
      searchAction2.apply(sep);
      shaderProgName = searchAction2.getPath() != NULL;
    }

    if(shaderProgName) {
      sep->unrefNoDelete();
      return TRUE;
    }
#if defined(_DEBUG)
    else {
      if(test)
        SoDebugError::postInfo("isValidShader", "Can't find a SHADERPROG node.");
    }
#else
    (void)test; // avoid unused parameter warning
#endif
  }

#if defined(_DEBUG)
  else {
    if(test)
      SoDebugError::postInfo("isValidShader", "Can't find a SHADER node.");
  }
#endif

  sep->unrefNoDelete();
  return FALSE;
}//---------------------------------------------------------------------------

/****************************************************************************/
// Update the shader's parameters
void 
updateShaderParameters(SoSeparator* /*shaderSep*/, SoShaderObject * shaderObject)
{
  int numParameters = shaderObject->parameter.getNumShaderParameters();
  for(int i=0; i<numParameters; i++) {
    SoShaderParameter * sparam = shaderObject->parameter.getShaderParameter(i);
    if(!sparam) {
#if defined(_DEBUG)
      SoDebugError::post("updateShaderParameters", 
        "No %dth parameter is set for the current shader.", (i+1));
#endif
      continue;
    }
    
    // ModelToWC
    if(sparam->name.getValue() == "ModelToWC") {
      ((SoShaderParameterMatrix *)sparam)->value.connectFrom(&ModelToWC->value);
    }
    
    // ModelToWCInv
    if(sparam->name.getValue() == "ModelToWCInv") {
      ((SoShaderParameterMatrix *)sparam)->value.connectFrom(&ModelToWCInv->value);
    }
    
    // EyePosition
    else if(sparam->name.getValue() == "EyePosition") {
      connectToEyePosition((SoShaderParameter3f *)sparam);
    }
    
    // Time
    else if(sparam->name.getValue() == "Time"){
      TimeValueParameter = (SoShaderParameter1f *)sparam;
      TimeValueSensor->schedule();
    }
  }
}//---------------------------------------------------------------------------

/****************************************************************************/
// 
void 
updateParametersControl(SoSeparator * shaderSep)
{
  SoGroup * dialogComponents = ((SoGroup *)shaderSep->getByName(DIALOG_COMPONENTS_NAME));
  if(dialogComponents) {
    int numDialogComponents = dialogComponents->getNumChildren();
    for(int j=0; j<numDialogComponents; j++) {
      // Check that the element inherits from SoDialogComponent
      SoNode * temp = (SoNode *)dialogComponents->getChild(j);
      if(temp->getTypeId().isDerivedFrom(SoDialogComponent::getClassTypeId())) {
        addParamControl((SoDialogComponent *)dialogComponents->getChild(j));
      }
#if defined(_DEBUG)
      else {
        SoDebugError::post("updateParametersControl", 
          "A non-SoDialogComponent node was set in the %s group of the shader file.", 
          DIALOG_COMPONENTS_NAME);
      }
#endif
    }
  }
}//---------------------------------------------------------------------------

/****************************************************************************/
// Load the shader
void 
shaderInit(SoSeparator * shaderSep, SbString filename)
{
  if(!shaderSep) {
#if defined(_DEBUG)
    SoDebugError::post("shaderInit", "The shader can't be initialized.");
#endif
    return;
  }

  shaderSep->ref();

  TimeValueSensor->unschedule();

  if(!isValidShader(shaderSep)) {
#if defined(_DEBUG)
    SoDebugError::post("shaderInit", "The shader is not valid.");
#endif
    return;
  }

  // Retrieve all special nodes of the iv file
  SoShaderProgram * shaderProgram = (SoShaderProgram *)shaderSep->getByName(SHADERPROG_NAME);
  SoSeparator     * toInsert      = (SoSeparator *)shaderSep->getByName(TOINSERT_NAME);
  if(!shaderProgram || !toInsert) {
#if defined(_DEBUG)
    SoDebugError::post("shaderInit", "Failure when searching the named nodes of the shader.");
#endif
    return;
  }
  
  // Reset Material Editor part
  resetMaterialEditor();

  // Add all the nodes necessary to make the shader work correctly (textures, ...)
  updateShaderInGraph(toInsert);

  // Rebuild dialog part
  resetShaderTabDialog();
  setShaderIVFileContent(shaderSep, (!filename) ? SbString("unknown") : filename);
  updateParametersControl(shaderSep);
  // Update parameters on all the shader objects
  int numShaders = shaderProgram->shaderObject.getNum();
  for(int j=0; j<numShaders; j++) {
    addShaderObjectSourceFile(removeDirectories(((SoShaderObject *)shaderProgram->shaderObject[j])->sourceProgram.getValue()),
      ((SoShaderObject *)shaderProgram->shaderObject[j])->getSourceProgram());
    updateShaderParameters(shaderSep, (SoShaderObject *)shaderProgram->shaderObject[j]);
  }
    
  // Disable the MaterialEditor menuPushButton if necessary
  setMaterialEditorCheckboxEnablement();

  setTitleBar();
  shaderSep->unref();
}//---------------------------------------------------------------------------

/****************************************************************************/
// Update the value of the ModelToWC shader parameter.
void
updateModelingMatrixParam(SbViewportRegion vp, SoTransform * modelTransform)
{
  SoGetMatrixAction matAction(vp);
  matAction.apply(modelTransform);
  ModelToWC->value = matAction.getMatrix();
  ModelToWCInv->value = matAction.getInverse();
}//---------------------------------------------------------------------------

/****************************************************************************/
// Update the value of the ModelToWC shader parameter.
SbBool
isGLSLSupported()
{
  return (SoVertexShader::isSupported(SoShaderObject::GLSL_PROGRAM) && SoFragmentShader::isSupported(SoShaderObject::GLSL_PROGRAM));
}//---------------------------------------------------------------------------


