#version 410 core

//!oiv_include <Inventor/oivShaderState.h>


// tessellation config
layout(quads, equal_spacing, ccw) in;

// pre-processor param
#define PI 3.14159265

// uniform parameter
uniform float freqFactor;

// normals from tessellation control shader invocations
in vec3 tcNormal[];

// output position to fragment shader
out vec3 tePos;

// output eye direction to fragment shader
out vec3 teEyeDir;

// output normal to fragment shader
out vec3 teNormal;

// output texture coordinate
out vec2 teTexCoord;


// displacement function with a read-only parameter
vec3
displace ( in vec3 p )
{
  // compute delta
  float delta = cos(p.x*freqFactor*PI)*sin(p.y*freqFactor*PI)*cos(p.z*freqFactor*PI);

  // apply transformation along the normal
  return 0.5*(p + smoothstep(0.0, 1.0, abs(delta)) * normalize(p));
}

// compute normal for current position
vec3
computeNormal ( in vec3 pos )
{
  // epsilon value
  vec2 eps = vec2(0.001, 0.0);
  
  // directional gradient
  vec3 gx = normalize(displace(normalize(pos + eps.xyy)) - displace(normalize(pos - eps.xyy)));
  vec3 gy = normalize(displace(normalize(pos + eps.yxy)) - displace(normalize(pos - eps.yxy)));
  vec3 gz = normalize(displace(normalize(pos + eps.yyx)) - displace(normalize(pos - eps.yyx)));

  // find the normal
  if ( abs(pos.z) > eps.x )
    return sign(pos.z)*normalize(cross(gx,gy));
  else if ( abs(pos.x) > eps.x )
    return sign(pos.x)*normalize(cross(gy,gz));
  else
    return -sign(pos.y)*normalize(cross(gx,gz));
}

// interpolate position according to tessellation coordinates
vec3
interpolatePosition ( in vec2 tessCoord )
{
  // interpolate the current position
  vec3 p0 = mix(gl_in[0].gl_Position.xyz, gl_in[1].gl_Position.xyz, tessCoord.x);
  vec3 p1 = mix(gl_in[3].gl_Position.xyz, gl_in[2].gl_Position.xyz, tessCoord.x);
  return mix(p0, p1, tessCoord.y);
}

void
main()
{
  // get position
  vec3 pos = interpolatePosition(gl_TessCoord.xy);

  // push to sphere
  pos = normalize(pos);
    
  // assign the texcoord
  teTexCoord = gl_TessCoord.xy;
  
  // normal in object space
  teNormal = computeNormal(pos);

  // displace the position
  pos = displace(pos);

  // normal in camera space
  teNormal = OivNormalMatrix()*teNormal;

  // position in camera space
  vec4 cPos = OivModelViewMatrix() * vec4(pos, 1.0);
  
  // eye pos/direction in camera space
  tePos = cPos.xyz;
  teEyeDir = -normalize(tePos);
  
  // position in NDC
  gl_Position = OivProjectionMatrix() * cPos;
}
