#version 410 core

//!oiv_include <Inventor/oivShaderState.h>
//!oiv_include <Inventor/oivShaderVariables.h>

// uniform parameters
uniform sampler2D colorMap;
uniform sampler2D normalMap;

uniform float Ka;
uniform float Kd;
uniform float Ks;
uniform float shininess;

// input parameters from Tessellation evaluation shader
in vec3 tePos;
in vec3 teEyeDir;
in vec3 teNormal;
in vec2 teTexCoord;


// shade it with 'n', the normal, 'v', the view direction, 'l', the light direction
// and c, the surface color. All directions must be normalized
vec3
phongShading ( in vec3 n, in vec3 v, in vec3 l, in vec3 c )
{ 
    // n dot l
  float NdL = max(dot(n, l), 0.0);
      
  // specular reflection
  vec3 r = reflect(-l, n);
  float specular = pow( max(dot(r, v), 0.0), shininess );
  
  // final composition
  return min((Ka + Kd*NdL)*c + Ks*specular, 1.0);
}

// compute tangent frame
mat3
cotangent_frame ( in vec3 N, in vec3 p, in vec2 uv )
{
  // get edge vectors of the pixel triangle
  vec3 dp1 = dFdx(p);
  vec3 dp2 = dFdy(p);
  vec2 duv1 = dFdx(uv);
  vec2 duv2 = dFdy(uv);

  // solve the linear system
  vec3 dp2perp = normalize(cross(dp2, N));
  vec3 dp1perp = normalize(cross(N, dp1));
  vec3 T = dp2perp * duv1.x + dp1perp * duv2.x;
  vec3 B = dp2perp * duv1.y + dp1perp * duv2.y;

  // construct a scale-invariant frame 
  float invmax = inversesqrt( max( dot(T,T), dot(B,B) ) );
  return mat3(normalize(T*invmax), normalize(B*invmax), N);
}

// main func
void
main ()
{
  // the directions
  vec3 n = normalize(teNormal);
  vec3 v = normalize(teEyeDir);
  vec3 l = normalize(OivLightSourcePosition(0).xyz);

  // compute tangent space
  mat3 TBN = cotangent_frame(n, tePos, teTexCoord);

  // project into the tangent plane
  l = normalize(l*TBN);
  v = normalize(v*TBN);

  // get parameter from maps
  n = normalize(2.0*texture(normalMap, teTexCoord).xyz - 1.0);
  vec3 c = texture(colorMap, teTexCoord).rgb;

  // shade
  vec3 color = phongShading(n, v, l, c);
  
  // final color
  OivFragmentOutput(vec4(color, 1.0));
}


