#version 410 core

//!oiv_include <Inventor/oivShaderState.h>
//!oiv_include <Inventor/oivShaderVariables.h>

// uniform parameters
uniform sampler2D colorMap;
uniform sampler2D normalMap;
uniform sampler2D heightMap;

uniform float heightScale;
uniform float Ka;
uniform float Kd;
uniform float Ks;
uniform float shininess;

// input parameters from Tessellation evaluation shader
in vec3 tePos;
in vec3 teEyeDir;
in vec3 teNormal;
in vec2 teTexCoord;


// shade it with 'n', the normal, 'v', the view direction, 'l', the light direction
// and c, the surface color. All directions must be normalized
vec3
phongShading ( in vec3 n, in vec3 v, in vec3 l, in vec3 c )
{ 
    // n dot l
  float NdL = max(dot(n, l), 0.0);
      
  // specular reflection
  vec3 r = reflect(-l, n);
  float specular = pow( max(dot(r, v), 0.0), shininess );
  
  // final composition
  return min((Ka + Kd*NdL)*c + Ks*specular, 1.0);
}

// compute tangent frame
mat3
cotangent_frame ( in vec3 N, in vec3 p, in vec2 uv )
{
  // get edge vectors of the pixel triangle
  vec3 dp1 = dFdx(p);
  vec3 dp2 = dFdy(p);
  vec2 duv1 = dFdx(uv);
  vec2 duv2 = dFdy(uv);

  // solve the linear system
  vec3 dp2perp = normalize(cross(dp2, N));
  vec3 dp1perp = normalize(cross(N, dp1));
  vec3 T = dp2perp * duv1.x + dp1perp * duv2.x;
  vec3 B = dp2perp * duv1.y + dp1perp * duv2.y;

  // construct a scale-invariant frame 
  float invmax = inversesqrt( max( dot(T,T), dot(B,B) ) );
  return mat3(normalize(T*invmax), normalize(B*invmax), N);
}

// ray marching from dp in the direction ds
float
ray_intersect_rm ( in vec2 dp, in vec2 ds )
{
  const int linear_search_steps = 10;
  const int binary_search_steps = 5;
  
  float depth_step = 1.0/linear_search_steps;
  float size = depth_step; // current size of search window
  float depth = 0.0; // current depth position
  // best match found (starts with last position 1.0)
  float best_depth = 1.0;
  // search from front to back for ﬁrst point inside the object
  for ( int i = 0; i < linear_search_steps-1; i++ ) { 
    depth += size;
    float t = texture(heightMap, dp + ds*depth).r;
    if (best_depth > 0.996 ) // if no depth found yet
      if (depth >= t )
        best_depth=depth; // store best depth
  }
  depth = best_depth;
  
  // search around ﬁrst point (depth) for closest match
  for ( int i = 0; i < binary_search_steps; i++ ) {
    size *= 0.5;
    float t = texture(heightMap, dp + ds*depth).r;
    if ( depth >= t ) {
      best_depth = depth;
      depth -= 2*size;
    }
    depth+=size;
  }
  return best_depth;
}

// main func
void
main ()
{
  // the directions
  vec3 n = normalize(teNormal);
  vec3 v = normalize(teEyeDir);
  vec3 l = normalize(OivLightSourcePosition(0).xyz);

  // get tangent plane  
  mat3 TBN = cotangent_frame(n, tePos, teTexCoord);

  // project into the tangent plane
  l = normalize(l*TBN);
  v = normalize(v*TBN);

  // texture coordinate
  vec2 texCoord = teTexCoord;

  // prepare relief mapping
  vec2 ds = heightScale*v.xy;
  vec2 dp = teTexCoord;
  float d = ray_intersect_rm(dp, ds);
  texCoord = dp + ds*d;

  n = normalize(2.0*texture(normalMap, texCoord).xyz - 1.0);
  vec3 c = texture(colorMap, texCoord).rgb;

  vec3 color = phongShading(n, v, l, c);
  OivFragmentOutput(vec4(color, 1.0));
}


