/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/sensors/SoSensor.h>

#include "RaceCar.h"
#include "State.h"

#ifdef _WIN32
#  include <windows.h>
#  include <Inventor/Win/SoWin.h>
#  include "RobotCar.h"

extern "C" double drand48(void);
extern void doOptionDialog( GameState* gameState );
#endif

#include <assert.h>

void
mouseInputCB(void *_gameState, SoEventCallback *eventCBnode)
{
    const SoEvent *event = eventCBnode->getEvent();
    
    GameState *gameState = (GameState *)_gameState;

    Car *car = gameState->getControlledCar();

    // No matter what, keep track of how fast the user wants the car
    // to go:
    if (event->isOfType(SoLocation2Event::getClassTypeId())) {

        // If still in the startScreen, car hasn't been created yet.
        // Just return.
        if (car == NULL) {
	    eventCBnode->setHandled();
            return;
        }

	// Get normalized location:
	const SbViewportRegion &vpReg =
	    eventCBnode->getAction()->getViewportRegion();

	const SbVec2f &position = 
	    event->getNormalizedPosition(vpReg);
	
	// Velocity ranges from 0.0 to 5.0:
	float newSpeed = position[1]*6.0f - 1.0f;
	if (newSpeed < 0.0)
	    newSpeed = 0.0;
        if (newSpeed > 5.0)
            newSpeed = 5.0;
	car->setSpeed(newSpeed);
    }

    // If the 'A' key is pressed, toggle on/off the annotation above each
    // car.
    else if (SO_KEY_PRESS_EVENT(event, A)) {

        int i;
        if (gameState->showAnnotation) {
            gameState->showAnnotation = FALSE;
            for (i = 0; i < gameState->getNumCars(); i++) {
                Car *car = gameState->getCar(i);
                if (!car->isSimulated())
                    car->removeAnnotation();
            }
        }
        else {
            gameState->showAnnotation = TRUE;
            for (i = 0; i < gameState->getNumCars(); i++) {
                Car *car = gameState->getCar(i);
                if (!car->isSimulated())
                    car->addAnnotation();
            }
        }
    }
    else if (SO_KEY_PRESS_EVENT(event, UP_ARROW)) {
        if (gameState->cameraUpDown < 20) {
	    ++gameState->cameraUpDown;
	}
    }
    else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW)) {
        if (gameState->cameraUpDown > 0) {
	    --gameState->cameraUpDown;
	}
    }
    else if (SO_KEY_PRESS_EVENT(event, RIGHT_ARROW)) {
        if (gameState->cameraBackForth < 20) {
	    ++gameState->cameraBackForth;
	}
    }
    else if (SO_KEY_PRESS_EVENT(event, LEFT_ARROW)) {
        if (gameState->cameraBackForth > 0) {
	    --gameState->cameraBackForth;
	}
    }
#ifdef _WIN32
    else if (SO_KEY_PRESS_EVENT(event, ESCAPE)) {
        int rc = MessageBox( SoWin::getTopLevelWidget(),
						 "Are you sure?",
						 "Leave the Race",
	                     MB_ICONQUESTION | MB_YESNO );
        if (rc == IDYES) {
            exit(0);
	}
        eventCBnode->setHandled();
    }
#endif

    switch (gameState->mode->getValue()) {
      case GameState::START_SCREEN:
	if (SO_MOUSE_PRESS_EVENT(event, BUTTON1)) {
	    const SoPickedPoint *pp = eventCBnode->getPickedPoint();
	    if (pp == NULL) break;
	    const SoPath *pickPath = pp->getPath();
	    if (pickPath == NULL) break;
	    const SoNode *pickNode = pickPath->getTail();

#ifdef _WIN32
            // WINxx programs are rarely started from the command line!
	    // Therefore we've added some buttons to the startup screen.
	    if (pickNode->getName() == "CancelButton") {
                int rc = MessageBox( NULL, "Are you sure?", "Leave the Race",
	                             MB_ICONQUESTION | MB_YESNO );
                if (rc == IDYES)
                    exit(0);
                eventCBnode->setHandled();
            }

	    if (pickNode->getName() == "OptionButton") {
 	        eventCBnode->setHandled();
	        doOptionDialog( gameState );
		return;
	    }
#endif
	    
	    if (pickNode->getName() == "StartButton") {

		int i;
#ifdef _WIN32
                // Create robot cars, and add them to the track:
		// (For WINxx, moved this here to allow runtime option setting)
                for (i = 0; i < gameState->numRobotCars; i++) {
	            RobotCar *car = new RobotCar;
	            // old: trackGeom->addChild(car->getCarRoot());
	            gameState->track->addCar(car);
	            gameState->addCar(car, FALSE);
                }
#endif

                if (gameState->myCarNumber == -1) {

                    // Choose the car for this player.  Pick a unique (if
                    // possible) car number by checking to see which car numbers
                    // have already been chosen.  If 8 or more cars are playing
                    // then cycle through the numbers again.
                    int minEncounters = gameState->getNumCars() / 8;
                    int proposedNumber = (int)(drand48() * 8);
                    for (int i=0; i<8; i++, proposedNumber++) {
                        proposedNumber = proposedNumber%8;

                        // Check the proposed number against the list of cars
                        int numEncounters = 0, j;
                        for (j=0; j<gameState->getNumCars(); j++) {
                            if (gameState->getCar(j)->getCarNum() ==
                                    (proposedNumber+1)) {
                                ++numEncounters;
                                if (numEncounters > minEncounters)
                                    break;
                            }
                         }
                        if (j == gameState->getNumCars())
                            break;
                    }
                    gameState->myCarNumber = proposedNumber + 1;
                }

                // Create the car for this player and add it to the scene.
                // Set its initial position on the track using the
                // Car::packetUpdate() method and placing the car somewhere
                // after the finish line.  On the shoulder?
#if defined(_DEBUG)
fprintf(stderr, "DEBUG:  picking car #%d\n", gameState->myCarNumber);
#endif
                RaceCar *rcar = new RaceCar(CarPlayer(0,
                        gameState->myCarNumber), gameState->playerName, TRUE);
                gameState->addCar(rcar, TRUE);

                // If playing quietly, don't initialize the sounds
                if (!gameState->playQuietly)
		    rcar->initializeSounds();
                rcar->packetUpdate(SbTime::zero(),
                        gameState->track->getLength()*(VICTORY_LAP
                        + 1.1f + (float)drand48()*0.3f),
                        0, 0.0f, 0.0f, -1, SCREECH_CUTOFF);
                // old: SoSeparator *carRoot = rcar->getCarRoot();    
                // old: SoSeparator *trackGeom = gameState->track->getTrack();
                // old: trackGeom->addChild(carRoot);
                gameState->track->addCar( rcar );

                // Set the game state to PRACTICING and start playing!
		gameState->mode->setValue(GameState::PRACTICING);
		gameState->simulationSensor->schedule();

                // Update the positions of any cars that might have joined
                // the race already.
                SbTime newTime;
                SbTime tod = SbTime::getTimeOfDay();
                newTime = tod - gameState->startTime;
                for (i = 0; i < gameState->getNumCars(); i++) {
                    Car *car = gameState->getCar(i);
                    car->updatePosition(newTime, gameState->track);
                }

                rcar->setRevving(TRUE);
	    }
	}
	break;
      case GameState::PRACTICING:
      case GameState::RACING:
      case GameState::AFTER_RACE:
      case GameState::RACE_OVER:
	{
            // If still in the startScreen, car hasn't been created yet.
            // Just return.
            if (car == NULL) {
	        eventCBnode->setHandled();
                return;
            }

	    if (SO_MOUSE_PRESS_EVENT(event, BUTTON1)) {
		int currentLane = car->getTargetLane();
		if (currentLane > 0) 
		    car->setTargetLane(currentLane-1);
	    }
	    else if (SO_MOUSE_PRESS_EVENT(event, BUTTON3)) {
		int currentLane = car->getTargetLane();
		if (currentLane < (NUM_LANES-1)) 
		    car->setTargetLane(currentLane+1);
	    }
	    eventCBnode->setHandled();
	}
	break;
      case GameState::RACE_STARTING:
	{
            // If still in the startScreen, car hasn't been created yet.
            // Just return.
            if (car == NULL) {
	        eventCBnode->setHandled();
                return;
            }

            // If the race is starting, and the car is already at the
            // start/finish line, don't allow it to change lanes.
            if (gameState->track->atFinishLine(car->getCurrentPosition()))
                break;

	    if (SO_MOUSE_PRESS_EVENT(event, BUTTON1)) {
		int currentLane = car->getTargetLane();
		if (currentLane > 0) 
		    car->setTargetLane(currentLane-1);
	    }
	    else if (SO_MOUSE_PRESS_EVENT(event, BUTTON3)) {
		int currentLane = car->getTargetLane();
		if (currentLane < (NUM_LANES-1)) 
		    car->setTargetLane(currentLane+1);
	    }
	    eventCBnode->setHandled();
	}
	break;
    }
}


