/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
** Modified by : Andrew Cherenson (MMM yyyy)
**=======================================================================*/

#include <stdio.h>
#include <string.h>

#ifndef _WIN32
#  include <unistd.h>
#endif

#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>

#ifndef _WIN32
#  include <sys/socket.h>
#endif

#ifndef _WIN32
#  include <sys/fcntl.h>
#  include <netinet/in.h>
#  include <sys/stat.h>
#  include <netdb.h>
#  include <arpa/inet.h>
#endif

#include "Network.h"

/* 
 * Note: This port is chosen not to collide with any other known port.
 * Use the same multicast address that dog uses, because we're lazy to
 * get an official address from the Internet authorities.
 */

#define SLOTCAR_PORT    5140
#define SLOTCAR_GROUP   "224.0.1.2"

#define SLOTCAR_TTL     8       /* Time-to-live */

static unsigned char packetBuf[256];

////////////////////////////////////////////////////////////////////////
//
// Description:
//    This routine initializes a datagram socket for multicasting.
//    Information is loaded into the game state for use
//    in sending and receiving packets of data from the network.
//

Network::Network(unsigned char _gameId)

//
////////////////////////////////////////////////////////////////////////
{
#if defined(_WIN32)
    gameId = _gameId;
    fileDesc = -1;
    return;
#else
    struct sockaddr_in  addr;
    struct in_addr      grpaddr;

    gameId = _gameId;
    if (gameId == DISABLE_NETWORK_GAMEID) {
	fileDesc = -1;
	return;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;


    // Let the administrator define different values for the
    // mcast address and port used by this program.

    struct hostent* hp = gethostbyname("sgi-slotcar.mcast.net");
    if (hp != NULL) {
	grpaddr = *((struct in_addr *)hp->h_addr);
    } else {
	grpaddr.s_addr = inet_addr(SLOTCAR_GROUP);
    }

    struct servent* sp = getservbyname("sgi-slotcar", "udp");
    if (sp != NULL) {
	addr.sin_port = sp->s_port;
    } else {
	addr.sin_port = htons(SLOTCAR_PORT);
    }

    fileDesc = socket(AF_INET, SOCK_DGRAM, 0);
    if (fileDesc < 0) {
        perror("Could not open socket");
        exit(1);
    }

    // Make the socket non-blocking so it can read from the net faster
    if (fcntl(fileDesc, F_SETFL, FNDELAY) < 0) {
        perror("Could not set socket for nonblocking reads");
    }

    // Assign a name to the socket.
    if (bind(fileDesc, (sockaddr *)&addr, sizeof(addr)) < 0) {
        perror("Could not bind socket name");
        exit(1);
    }

    // Indicate our interest in receiving packets sent to the mcast address.
    struct ip_mreq  mreq;
    mreq.imr_multiaddr = grpaddr;
    mreq.imr_interface.s_addr = INADDR_ANY;
    if (setsockopt(fileDesc, IPPROTO_IP, IP_ADD_MEMBERSHIP,
                (const char *)&mreq, sizeof(mreq)) < 0) {
        perror("Could not add multicast address to socket");
        exit(1);
    }

    // Put a limit on how far the packets can travel.
    u_char ttl = SLOTCAR_TTL;
    if (setsockopt(fileDesc, IPPROTO_IP, IP_MULTICAST_TTL,
		(const char *)&ttl, sizeof(ttl)) < 0) {
        perror("Could not initialize ttl on socket");
        exit(1);
    }

    // Turn looping off so packets will not go back to the same host.
    // This means that multiple instances of this program will not receive
    // packets from each other.
    u_char loop = 0;
    if (setsockopt(fileDesc, IPPROTO_IP, IP_MULTICAST_LOOP, 
		(const char *)&loop, sizeof(loop)) < 0) {
        perror("Could not initialize socket");
        exit(1);
    }

    // Set up the destination address for packets we send out.
    destAddr = addr;
    destAddr.sin_addr = grpaddr;
#endif
}

//////////////////////////////////////////////////////////////////////////////
//
//  Destructor
//
Network::~Network()
//
//////////////////////////////////////////////////////////////////////////////
{
#ifndef _WIN32
    if (fileDesc != -1) {
        close(fileDesc);
	fileDesc = -1;
    }
#endif
}

//////////////////////////////////////////////////////////////////////////////
//
//  Send data on the socket.
//
SbBool
Network::send(NetworkMsg &msg, int numBytes)
//
//////////////////////////////////////////////////////////////////////////////
{
  if (gameId == DISABLE_NETWORK_GAMEID) {
    return TRUE;
  }

#ifndef _WIN32
  msg.gameId = gameId;

  // Don't send the network address:
  const void *bytes = &msg.gameId;

  if (sendto(fileDesc, (char *)bytes, numBytes - sizeof(msg.fromAddress), 0,
             (sockaddr *)&destAddr, sizeof(struct sockaddr_in)) < 0) {
#  if defined(_DEBUG)
    perror("ERROR: send failed");
#  endif
    return FALSE;
  }
#endif

  return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
//
//  Receive data on the socket.  The first four bytes of buf are
//  filled in with the internet address of the host that sent the
//  packet.
//
SbBool
Network::receive(unsigned char *buf, int size)
//
//////////////////////////////////////////////////////////////////////////////
{
#ifndef _WIN32
  NetworkMsg* msg = (NetworkMsg*) buf;
  struct sockaddr_in  whereFrom;    // sender's network address
  socklen_t fromlen = sizeof(whereFrom);

  if (recvfrom(fileDesc, (char *)(&msg->gameId),
               size-sizeof(msg->fromAddress), 0,
               (sockaddr *)&whereFrom, &fromlen) < 0) {
    return FALSE;
  }
  msg->fromAddress = (u_int32_t)whereFrom.sin_addr.s_addr;
#endif

  return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
//
//  Receive data on the socket
//
const NetworkMsg *
Network::get()
//
//////////////////////////////////////////////////////////////////////////////
{
#ifndef _WIN32
    NetworkMsg* msg = (NetworkMsg *) packetBuf;
    if (!receive(packetBuf, sizeof(packetBuf)) || (msg->gameId != gameId))
	return NULL;
    return msg;
#else
    return NULL;
#endif
}


