/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/

#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/elements/SoLightModelElement.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/elements/SoDepthBufferElement.h>

#include "Sky.h"

SO_NODE_SOURCE(Sky);

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Init Sky
//

void
Sky::initClass()
//
////////////////////////////////////////////////////////////////////////
{
    getClassRenderEngineMode().setRenderMode(SbRenderEngineMode::OIV_OPENGL_COMPATIBILITY_RENDERING);
    SO_NODE_INIT_CLASS(Sky, SoNode, "Node");
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor; setup fields, etc.
//

Sky::Sky()
//
////////////////////////////////////////////////////////////////////////
{
  SO_NODE_CONSTRUCTOR(Sky);

  SO_NODE_ADD_FIELD(horizonColor, (0.8f, 0.8f, 1.0f));
  SO_NODE_ADD_FIELD(horizonHeight, (100.0f));
  SO_NODE_ADD_FIELD(skyBottomColor, (0.0f, 0.4f, 0.7f));
  SO_NODE_ADD_FIELD(skyTopColor, (0.0f, 0.0f, 0.44f));

  m_skyFaceSet = new SoFaceSet;

  m_skyVertices = new SoVertexProperty;
  m_skyFaceSet->vertexProperty = m_skyVertices.ptr();

  m_skyVertices->materialBinding = SoVertexProperty::PER_VERTEX;
  m_skyVertices->orderedRGBA.setNum( 8 );
  m_skyVertices->vertex.setNum( 8 );
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//

Sky::~Sky()
//
////////////////////////////////////////////////////////////////////////
{
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Render
//

void
Sky::GLRender(SoGLRenderAction *action)
//
////////////////////////////////////////////////////////////////////////
{
    SoState *state = action->getState();
    state->push();

    // Turn off lighting
    SoLightModelElement::set(state, SoLightModelElement::BASE_COLOR);

    // Don't z-buffer
    SoDepthBufferElement::setTest( state, FALSE );

    // Get the view volume
    const SbViewVolume &vv = SoViewVolumeElement::get(state);
    
    // Figure out stuff:
    const SbVec3f &eye = vv.getProjectionPoint();
    const SbVec3f &eyeDir = vv.getProjectionDirection();

    // Point in the middle of the view volume:
    SbVec3f midVolume = eye + eyeDir*(vv.getNearDist() +
				      vv.getDepth()*0.98f);
    SbVec3f midVolume0(midVolume[0], 0.0, midVolume[2]);
    
    // We'll construct the sky polygons in world space, located at
    // mid-view volume, based at y==0, oriented appropriately in x/z:

    // ??? This will not work for arbitrary eye orientations!  In
    // particular, looking straight up or down the Y axis will cause
    // problems.
    SbVec3f perpendicular(-eyeDir[2], 0.0, eyeDir[0]);
    perpendicular.normalize();
    
    const float bigWidth = 10000.0;

    // Just draw big polygons a good distance along perpendicular,
    // from the midVolume point:
    const SbVec3f leftHorizon = midVolume0 + perpendicular * bigWidth;
    const SbVec3f rightHorizon = midVolume0 - perpendicular * bigWidth;
    const SbVec3f leftBottomSky = leftHorizon + SbVec3f( 0.0, horizonHeight.getValue(), 0.0 );
    const SbVec3f rightBottomSky = rightHorizon + SbVec3f( 0.0, horizonHeight.getValue(), 0.0 );
    const SbVec3f leftTopSky = leftHorizon + SbVec3f( 0.0, bigWidth, 500.0 );
    const SbVec3f rightTopSky = rightHorizon + SbVec3f( 0.0, bigWidth, 500.0 );

    m_skyVertices->vertex.set1Value( 0, leftHorizon );
    m_skyVertices->vertex.set1Value( 1, rightHorizon );
    m_skyVertices->vertex.set1Value( 2, rightBottomSky );
    m_skyVertices->vertex.set1Value( 3, leftBottomSky );

    m_skyVertices->orderedRGBA.set1Value( 0, horizonColor.getValue().getPackedValue() );
    m_skyVertices->orderedRGBA.set1Value( 1, horizonColor.getValue().getPackedValue() );
    m_skyVertices->orderedRGBA.set1Value( 2, skyBottomColor.getValue().getPackedValue() );
    m_skyVertices->orderedRGBA.set1Value( 3, skyBottomColor.getValue().getPackedValue() );

    m_skyVertices->vertex.set1Value( 4, leftBottomSky );
    m_skyVertices->vertex.set1Value( 5, rightBottomSky );
    m_skyVertices->vertex.set1Value( 6, leftTopSky );
    m_skyVertices->vertex.set1Value( 7, rightTopSky );

    m_skyVertices->orderedRGBA.set1Value( 4, skyBottomColor.getValue().getPackedValue() );
    m_skyVertices->orderedRGBA.set1Value( 5, skyBottomColor.getValue().getPackedValue() );
    m_skyVertices->orderedRGBA.set1Value( 6, skyTopColor.getValue().getPackedValue() );
    m_skyVertices->orderedRGBA.set1Value( 7, skyTopColor.getValue().getPackedValue() );

    action->forwardTraversal( m_skyFaceSet.ptr() );

    state->pop();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    This node does not affect state.
//

SbBool
Sky::affectsState() const
//
////////////////////////////////////////////////////////////////////////
{
    return FALSE;
}

