/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/errors/SoDebugError.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/misc/SoChildList.h>

#include "Car.h"
#include "Track.h"

#include <assert.h>

#define CAR_OFFSET 0.01f

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Constructor.
//
Track::Track()

//
////////////////////////////////////////////////////////////////////////
{
    trackData = NULL;
    clearStartingPositions();
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Destructor
//
Track::~Track()

//
////////////////////////////////////////////////////////////////////////
{
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Read in the track, and return a scene graph representing it
//

SoSeparator *
Track::getTrack(int whichTrack)
//
////////////////////////////////////////////////////////////////////////
{
    // If we've already read in, just return geometry:
    if (trackData != NULL) return trackData;

    // Read the track scenery geometry from a file and add to the scene
    SoInput myInput;
    if (whichTrack) {
        if (!myInput.openFile("$OIVHOME/examples/source/Inventor/SlotCar/data/ShortTrack.iv"))
                exit (1);
    }
    else {
        if (!myInput.openFile("$OIVHOME/examples/source/Inventor/SlotCar/data/LongTrack.iv"))   // System long
                exit (1);
    }
    trackData = SoDB::readAll(&myInput);
    if (trackData == NULL) 
        exit (1);

    trackData->ref();

    // Find the pieces of track in the scene read in:

    // WARNING!  There is an incredibly subtle bug that happens here
    // if we declare the SearchAction on the stack instead of creating
    // it with new.  If declared on the stack, the
    // trackData->unrefNoDelete() will happen before the search action
    // is destroyed; when the search action is destroyed (at the end
    // of this routine), trackData's reference count will go to zero
    // and it will be deleted.
    // Very insidious, indeed.
    SoSearchAction *sa = new SoSearchAction;
    sa->setInterest(SoSearchAction::ALL);
    sa->setType(TrackSegment::getClassTypeId());
    sa->apply(trackData);
    
    const SoPathList &paths = sa->getPaths();
    int i;
    for (i = 0; i < paths.getLength(); i++) {
	segments.append(paths[i]->getTail());
    }

    // Link up the track:
    int numSegs = segments.getLength();
    SbVec3f segPosition(0,0,0);
    SbRotation segOrientation(SbVec3f(0,1,0),0);
    for (i = 0; i < numSegs; i++) {
	TrackSegment *seg = (TrackSegment *)segments[i];
	seg->link(segPosition, segOrientation,
		  (TrackSegment *)segments[(i+1)%numSegs], 
		  (TrackSegment *)segments[(i-1+numSegs)%numSegs],
		  i);
    }

    delete sa;


    // Create root of all cars and properties for all cars
    carData = new SoSeparator;
    trackData->addChild( carData );
    SoLightModel *carLm  = new SoLightModel;
    carLm->model.setValue( SoLightModel::PER_VERTEX_PHONG );
    carData->addChild( carLm );

    SoSFInt32 *typeField = (SoSFInt32 *) SoDB::getGlobalField("trackType");
    if ( typeField->getValue() == 2 ) {
        SoTexture2 *offTex = new SoTexture2;
        carData->addChild( offTex );
    }

    trackData->unrefNoDelete();

    return trackData;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Move something down the track:
//

void
Track::getNewPosition(const TrackPosition &start,
		      const TrackPosition &velocity,
		      const SbTime &deltaTime,
		      TrackPosition &result,
                      float &grip)
//
////////////////////////////////////////////////////////////////////////
{
    int numSegments = segments.getLength();
    int startID = ((int)start.along)%numSegments;
    
    TrackSegment *startSegment = (TrackSegment *)segments[startID];
    
    result.along = start.along;
    result.across = start.across;

    // Initialize grip to a large value; the segments that we travel
    // across will reset it if their grip is smaller:
    grip = MAX_GRIP;
    startSegment->getNewPosition(start, velocity, deltaTime, result, grip);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Given a position on the track, figure out the world space
//    orientation/translation.
//

void
Track::getWorldPosition(const TrackPosition &position,
			  WorldPosition &result)
//
////////////////////////////////////////////////////////////////////////
{
    int numSegments = segments.getLength();
    int startID = ((int)position.along)%numSegments;
    
    TrackSegment *startSegment = (TrackSegment *)segments[startID];
    
    startSegment->getWorldPosition(position, result);
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Given a position on the track, figure out the curvature of the 
//    track at that position.
//

float
Track::getCurvature(const TrackPosition &position)
//
////////////////////////////////////////////////////////////////////////
{
    int numSegments = segments.getLength();
    int startID = ((int)position.along)%numSegments;
    
    TrackSegment *startSegment = (TrackSegment *)segments[startID];
    
    return startSegment->getCurvature(position);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Given a position on the track, figure out the length of the 
//    segment of track at that position.
//

float
Track::getSegmentLength(const TrackPosition &position)
//
////////////////////////////////////////////////////////////////////////
{
    int numSegments = segments.getLength();
    int startID = ((int)position.along)%numSegments;
    
    TrackSegment *startSegment = (TrackSegment *)segments[startID];
    
    return startSegment->getLength(position);
}

/////////////////////////////////////////////////////////////////////////
//
// Description:
//    Figure out if a car is sitting at/near the start/finish line.
//

SbBool
Track::atFinishLine(const TrackPosition &position)
//
////////////////////////////////////////////////////////////////////////
{
    if (position.along < getLength()*STARTING_LINE_LAP)
        return FALSE;

    int lane = (int)position.across;
    assert(startingPositions.getLength() == NUM_LANES);

    // Just in case we're crashing...
    if (lane < 0) lane = 0;
    if (lane >= NUM_LANES) lane = NUM_LANES-1;

    float p = fmod(position.along, getLength());

    // Figure out if we are within 1 unit of possible starting
    // position:
    float sp = (float)(getLength()) - 
	(float)(startingPositions[lane]*START_CAR_SEPARATION);
    
    if ((p < sp) && (p > (sp-1.0))) return TRUE;

    return FALSE;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Assign a starting position.  Depends on positions already
//    assigned.
//

void
Track::assignStartingPosition(const TrackPosition &current,
			      TrackPosition &result)
//
////////////////////////////////////////////////////////////////////////
{
    int lane = (int)current.across;

    // Just in case we're crashing...
    if (lane < 0) lane = 0;
    if (lane >= NUM_LANES) lane = NUM_LANES-1;
    
    result.across = (float)lane;
    result.along = (float)(segments.getLength()*(STARTING_LINE_LAP+1))
        - CAR_OFFSET - (float)(startingPositions[lane]*START_CAR_SEPARATION);

    ++startingPositions[lane];
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Clear out the starting positions.
//

void
Track::clearStartingPositions()
//
////////////////////////////////////////////////////////////////////////
{
    for (int i = 0; i < NUM_LANES; i++) {
	startingPositions[i] = 0;
    }
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Add a car to the track.
//

void
Track::addCar( Car *car )
//
////////////////////////////////////////////////////////////////////////
{
    carData->addChild( car->getCarRoot() );
}


////////////////////////////////////////////////////////////////////////
//
// Description:
//    Remove a car from the track.
//

void
Track::removeCar( Car *car )
//
////////////////////////////////////////////////////////////////////////
{
    carData->removeChild( car->getCarRoot() );
}




//------------------------- TrackSegment ---------------------------

SO_NODE_ABSTRACT_SOURCE(TrackSegment);

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Init TrackSegment class
//

void
TrackSegment::initClass()

//
////////////////////////////////////////////////////////////////////////
{
    getClassRenderEngineMode().setRenderMode(SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING);
    SO_NODE_INIT_ABSTRACT_CLASS(TrackSegment, SoSeparator, "Separator");
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    TrackSegment constructor
//

TrackSegment::TrackSegment()

//
////////////////////////////////////////////////////////////////////////
{
    SO_NODE_CONSTRUCTOR(TrackSegment);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    TrackSegment destructor
//

TrackSegment::~TrackSegment()

//
////////////////////////////////////////////////////////////////////////
{
}


