/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/


#ifndef  _TRACK_
#define  _TRACK_

#include <Inventor/SbLinear.h>
#include <Inventor/SbPList.h>
#include <Inventor/SbTime.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoSeparator.h>


class SoTransform;
class TrackPosition;
class WorldPosition;
class Car;

#define MAX_GRIP 1000.0f
#define NUM_LANES 6
#define START_CAR_SEPARATION 0.3f


// Constants for track segements
#define CURB_WIDTH      	4.0f
#define GUARD_RAIL_OFFSET	8.0f
#define NUM_SEGS_CIRCLE 	24.0f
#define SHOULDER_WIDTH  	18.0f

#define SHOULDER_COLOR          SbVec3f(0.3f,  0.1f,  0.0f)
#define SHOULDER_COLOR_TEXTURED SbVec3f(0.8f,  0.8f,  0.8f)
#define SLOT_LINE_COLOR         SbVec3f(0.0f,  0.0f,  0.0f)
#define SLOT_LINE_COLOR_LO_RES  SbVec3f(0.2f,  0.2f,  0.2f)
#define TRACK_COLOR             SbVec3f(0.6f,  0.6f,  0.6f)
#define TRACK_COLOR_TEXTURED    SbVec3f(0.9f,  0.9f,  0.9f)
#define TRACK_COLOR_LO_RES      SbVec3f(0.45f, 0.45f, 0.42f)

// Number of laps in each race.  The first lap of the track is slop to
// avoid nasty cases where the track position might go negative.
#define NUM_LAPS 5
#define STARTING_LINE_LAP 1
#define VICTORY_LAP (NUM_LAPS + STARTING_LINE_LAP + 1)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: Track
//
//  This class stores a list of all track segments, and has routines
//  for dealing with the track as a whole.
//
//  A word on track space:
//  Each track segment goes from a paramaterized 0.0 to 1.0.  So, a
//  length of one lap is paramaterized from 0.0 to the number of track
//  segments+1.  Notice that a track-space distance of 1.0 might be
//  very long or very short in world space; internally, the track
//  segments compensate for this; velocities in track space are stated
//  in 'unit-length' track space; a unit-length track is 100
//  world-space units long.
//
//////////////////////////////////////////////////////////////////////////////

class Track {
 public:

    // Constructor and Destructor
    Track();
    virtual ~Track();

    // Returns the scene graph of the whole track
    virtual SoSeparator *getTrack(int whichTrack);

    // Given a starting position, a velocity (expressed in
    // track-length units), and a time delta, calculate a new position
    // on the track and the grip of the track at that position.
    void getNewPosition(const TrackPosition &start,
			const TrackPosition &velocity,
			const SbTime &deltaTime,
			TrackPosition &result,
                        float &grip);

    // Given a track position, return a world-space position:
    void getWorldPosition(const TrackPosition &position,
			  WorldPosition &result);

    // Given a track position, return the curvature at that position
    float getCurvature(const TrackPosition &position);

    // Returns TRUE if a car is near the start/finish line.
    SbBool atFinishLine(const TrackPosition &position);

    // Figures out starting positions of cars.  You MUST sort the cars
    // by their distance along the track to get accurate starting
    // positions (this routine uses the order in which it is called to
    // hand out starting positions).
    void assignStartingPosition(const TrackPosition &current,
				TrackPosition &result);

    // Clear the starting position info.  Must be called before
    // first call to assignStartingPosition.
    void clearStartingPositions();
    
    // Returns the length of the track, or of a segment located at the
    // given track position.
    float getLength() { return (float)segments.getLength(); }
    float getSegmentLength(const TrackPosition &pos);

    // Add and remove car methods
    void addCar(Car *car);
    void removeCar(Car *car);

  private:
    // List of track segments...
    SoNodeList	segments;

    SbIntList startingPositions;

    // Scene graph for track
    SoSeparator *trackData;

    // Root node of cars
    SoSeparator *carData;
};

//////////////////////////////////////////////////////////////////////////////
//
//  Class: TrackSegment
//
//  Pure virtual class that defines the interface to different pieces
//  of track (straight, curved, etc).
//
//////////////////////////////////////////////////////////////////////////////

class TrackSegment : public SoSeparator
{
    SO_NODE_HEADER(TrackSegment);

  public:
    static void		initClass();

    TrackSegment();

    // This is used to link pieces together.  Called by Track.
    // This modifies lastPosition/lastDirection.
    virtual void link(SbVec3f &lastPosition, SbRotation &lastOrientation,
		      TrackSegment *nextSeg, TrackSegment *prevSeg,
		      int trackID) = 0;
    
    // Given a starting position, a velocity (expressed in
    // track-length units), and a time delta, calculate a new position
    // on the track.
    virtual void getNewPosition(const TrackPosition &start,
				const TrackPosition &velocity,
				const SbTime &deltaTime,
				TrackPosition &result,
				float &grip) = 0;

    // Given a track position, return a world-space position:
    virtual void getWorldPosition(const TrackPosition &position,
				  WorldPosition &result) = 0;

    // Given a track position, return the curvature or length at that position
    virtual float getCurvature(const TrackPosition &position) = 0;
    virtual float getLength(const TrackPosition &position) = 0;

  protected:
    virtual ~TrackSegment();

};

//////////////////////////////////////////////////////////////////////////////
//
//  Class: TrackPosition
//
//  Really simple structure containing a position on the track.
//
//////////////////////////////////////////////////////////////////////////////

class TrackPosition 
{
  public:
    TrackPosition()	{ along = 0.0f; across = 0.0f; }
    float		along;
    float		across;
};

//////////////////////////////////////////////////////////////////////////////
//
//  Class: WorldPosition
//
//  Another simple class, this time containing a world-space
//  translation/rotation.  It is possible to convert from a
//  CarPosition to a WorldPosition, but not vice/versa.
//
//////////////////////////////////////////////////////////////////////////////

class WorldPosition 
{
  public:
    WorldPosition()	{ translation.setValue(0,0,0);
			  rotation.setValue(SbVec3f(0,1,0), 0);
		      }
    SbVec3f		translation;
    SbRotation		rotation;
};

#endif /* _TRACK_ */


