/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/

#ifdef _WIN32
#  include "windows/getopt.h"
#else
# include <getopt.h>
#endif

#include <math.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoOutput.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoEnvironment.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/sensors/SoIdleSensor.h>
#include <Inventor/sensors/SoOneShotSensor.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/sys/SoGL.h>

#include <Inventor/elements/SoDepthBufferElement.h>

// This define turns on the performance meter.  Comment this line out
// to make it go away
#define PERF_METER
#ifdef PERF_METER
#  include <Inventor/actions/SoCallbackAction.h>
#  include <Inventor/elements/SoCacheElement.h>
#  include <Inventor/nodes/SoColorIndex.h>
#  include <Inventor/nodes/SoFont.h>
#  include <Inventor/nodes/SoOrthographicCamera.h>
#  include <Inventor/nodes/SoText2.h>
#endif

#include "InventorLogo.h"
#include "RaceCar.h"
#include "RobotCar.h"
#include "State.h"
#include "Packet.h"
#include "Network.h"
#include "Track.h"
#include "Curve.h"
#include "Straight.h"
#include "StartFinish.h"
#include "Sky.h"
#include "LODD.h"

#ifdef _WIN32
#  include <process.h>  // for _getpid() used to init random numbers
#  include "windows/print.h"  // to make printf/fprintf work under Win32
#  define XtInputId UINT
#endif

#define SHORT_START_DELAY 15.0
#define LONG_START_DELAY  45.0  // System long

#if !defined(_WIN32) && !defined(__APPLE__)
// Xt Input callback for receiving packets from the network
extern void packetCB( XtPointer, int *, XtInputId * );
#endif

// Idle Sensor callback for running the simulation
extern void runSimulation( void *, SoSensor * );

// Callback to handle mouse input
extern void mouseInputCB(void *car, SoEventCallback *eventCBnode);

static void setupGame(void);

static SoOneShotSensor *polySensor = NULL;
static SoXtRenderArea  *myRenderArea = NULL;
static GameState *gameState = NULL;
static int numRobotCars = 0;
// uncomment to turn on texture mapping
// static int gameNum      = 1;
// uncomment for no texture mapping
static int gameNum      = 0;
static SbBool useTexturedTrack = FALSE;
static SbBool disableTexturedTrack = FALSE;
static SbBool disableOverlays = FALSE;

#ifdef PERF_METER
static int numFramesRendered = 0;
static void initPerfMeter(SoGroup *);
static void countUpdatesCB(void *, SoAction *);
static void perfSensorCB(void *, SoSensor *);
static SoSeparator *addUpdateCounter(SoNode *root);
#endif

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Print out a usage message.
//

void
usage()

//
////////////////////////////////////////////////////////////////////////
{
    fprintf(stderr, "\nUsage:  slotcar [-c 1-8] [-g 1-100] [-h] "
	    	    "[-n name] [-q] [-s] [-r 1-8] \n");
    fprintf(stderr, "    [-r 1-8]  :  Number of robot cars (default 0).\n");
    fprintf(stderr, "    [-n name] :  Your name (maximum 15 characters, "
		    "default is your login).\n");
    fprintf(stderr, "    [-c 1-8]  :  Desired car number "
	    	    "(chosen randomly by default)\n");
    fprintf(stderr, "    [-g 2-100]:  Private game from 2 to 100.  (0 is default.)\n");
    fprintf(stderr, "    [-s]      :  Single-player mode "
	    	    "(no network stuff).\n");
    fprintf(stderr, "    [-q]      :  Play the game quietly (no sound).\n");
    fprintf(stderr, "    [-h]      :  Print out this message.\n");
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Exit the program.
//

void
exitCB(void *userData, SoXtComponent *)

//
////////////////////////////////////////////////////////////////////////
{
    GameState *state = (GameState *)userData;

    // Send exit packets for all cars we're simulating:
    for (int i = 0; i < state->getNumCars(); i++) {
	Car *car = state->getCar(i);
	CarExit exitPacket(car->getCarNum());
	state->network->send(exitPacket, sizeof(CarExit));
    }

#ifdef _WIN32
    // Make sure we don't leave zombie sound files playing!
    PlaySound( NULL, NULL, 0 );
#endif

    exit(0);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Enable the z-buffer.
//

void
zEnableCB(void *, SoAction *action)

//
////////////////////////////////////////////////////////////////////////
{
    SoDepthBufferElement::setTest(action->getState(), TRUE);
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Disable the z-buffer the program.
//

void
zDisableCB(void *, SoAction *action)

//
////////////////////////////////////////////////////////////////////////
{
    SoDepthBufferElement::setTest(action->getState(), FALSE);
}


#define ABBREV_THRESH   0.9
#define TEXTURE_THRESH  0.9

////////////////////////////////////////////////////////////////////////////
//
// This callback is called after the window is displayed and the first
// rendering has finished. It times polygon rendering to see if it's
// fast enough for real-time play.
//
////////////////////////////////////////////////////////////////////////////

static void
timePolygonCB(void *, SoSensor *)
{
    int i;
    SoInput reader;
    reader.openFile("$OIVHOME/examples/source/Inventor/SlotCar/data/PolygonTime.iv");
    SoSeparator *testRoot = SoDB::readAll(&reader);
    reader.closeFile();
#ifdef _WIN32
    // Note: This is actually a bug in slotcar, this test should be here!
    if (testRoot == NULL) {
        MessageBox( NULL, "Unable to open \"$OIVHOME/examples/source/Inventor/SlotCar/data/PolygonTime.iv\"",
                    "SlotCar Error", MB_TASKMODAL | MB_ICONSTOP | MB_OK );
        exit(1);
    }
#endif
    testRoot->ref();
    myRenderArea->setSceneGraph(testRoot);

    // Scale the square so that it takes up about 200 x 200 pixels.
    // (The square by default takes up the entire window.)
    float scale = 200.0f / (float) myRenderArea->getSize()[1];
    SoScale *polyScale = (SoScale *) SoNode::getByName("PolygonTimeScale");
    polyScale->scaleFactor.setValue(scale, scale, scale);

    // Render the scene twice to make sure everything is loaded
    myRenderArea->render();
    myRenderArea->render();

    // Draw the scene three times and time the results
    SbTime startTime, endTime;
    startTime.setToTimeOfDay();
    for ( i=0; i<3; i++) 
        myRenderArea->render();
    endTime.setToTimeOfDay();
    testRoot->unref();
    myRenderArea->setSceneGraph(NULL);

    // Initialize trackType global field; holds switch for
    // which track to use (abbreviated, normal, or textured).
    SoSFInt32 *typeField = (SoSFInt32 *)
    	SoDB::createGlobalField("trackType", SoSFInt32::getClassTypeId());
    if (typeField == NULL) {
    	fprintf(stderr, "Couldn't initialize trackType global field\n");
	exit(1);
    }

     // Create a blank texture node to be used to turn off textures
    // at certain points in the graph (curve shoulder lines, cars.)
    ///SoTexture2 *offTexture = new SoTexture2;
    ///offTexture->setName("OffTexture");
    ///offTexture->ref();

    // See if it takes too much time. If so,
    // setup the simpler scene.
    SbTime timeElapsed = endTime - startTime;
    double elapsed = timeElapsed.getValue();

    if (elapsed > ABBREV_THRESH) {
	typeField->setValue(0); // abbreviated track
    }
    else {
    	
	// Open the file for texture timing (to see if
	// the current machine is fast enough to run a textured
	// version of the game).
    	reader.openFile("$OIVHOME/examples/source/Inventor/SlotCar/data/TexturePolygonTime.iv");
    	testRoot = SoDB::readAll(&reader);
    	reader.closeFile();
    	testRoot->ref();
	myRenderArea->setSceneGraph(testRoot);
    	
	// Scale as above
    	polyScale = (SoScale *) SoNode::getByName("PolygonTimeScale");
    	polyScale->scaleFactor.setValue(scale, scale, scale);
    	
    	myRenderArea->render();
     	myRenderArea->render();

    	// Draw scene and time results
	startTime.setToTimeOfDay();
    	for (i=0; i<3; i++) 
    	    myRenderArea->render();
    	endTime.setToTimeOfDay();
    	testRoot->unref();
   	
	elapsed = (endTime - startTime).getValue();
	
        if (useTexturedTrack)
                {//fprintf(stderr,"Using the textured track.\n"); 
		typeField->setValue(2);} // textured track
        else if (disableTexturedTrack) 
                {//fprintf(stderr,"Texture disabled.\n"); 
		typeField->setValue(1);} // normal track
        else if (elapsed > TEXTURE_THRESH)
                {//fprintf(stderr,"Using normal track since textures are slow.\n"); 
		typeField->setValue(1);} // normal track
	else
                {//fprintf(stderr,"Using textured track since textures are fast.\n"); 
		typeField->setValue(2);} // textured track
    }
    setupGame();

    delete polySensor;
		polySensor = NULL;
}

////////////////////////////////////////////////////////////////////////////
//
// This callback is called when the render area window becomes mapped
// and it does its first render. It schedules a sensor that will do
// the texture timing stuff.
//
////////////////////////////////////////////////////////////////////////////

static void
waitForWindowCB(void *, SoAction *action)
{
    if (polySensor == NULL && action->getTypeId() == SoGLRenderAction::getClassTypeId()) {
	polySensor = new SoOneShotSensor(timePolygonCB, NULL);
	polySensor->setPriority(1);
	polySensor->schedule();
    }
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    This is the mainline program for the slotcar game.
//

#include <Inventor/SoWinApp.h>

int main( int argc, char **argv )
//
////////////////////////////////////////////////////////////////////////
{
	FILE *f;
	f = SbFileHelper::open("$OIVHOME/examples/source/Inventor/SlotCar/data/billboard.iv","r");
	if (f == NULL) {
		char message[256];
		sprintf(message, "Data file not found or not installed. \nYou may need to download the optional Open Inventor data files. \n");
#ifdef _WIN32
		MessageBox(NULL, message, "Machining",MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
		printf("%s\n", message) ;
#endif
		exit(1);
	}
	fclose(f);

    // Initialize Inventor and Xt
    Widget myWindow = SoXt::init(argv[0]);  
    if (myWindow == NULL) exit(1);     

    // Initialize new classes:
    Sky::initClass();
    TrackSegment::initClass();
    Straight::initClass();
    StartFinish::initClass();
    Curve::initClass();
    LODD::initClass();

    // Parse the command line
    int carNum       = -1;
    SbBool quietGame  = TRUE;
#if defined(_WIN32)
    char *buffer = new char[256];
    DWORD len = 256;
    GetUserName( buffer, &len );
    SbString playerName( buffer );
    delete [] buffer;
#elif defined(__APPLE__)
    SbString playerName( getlogin() );
#else
    SbString playerName(cuserid(NULL));
#endif

    int err = 0;	// Flag: error in options?
    int c;

    // Note: optind and optarg are declared in getopt.h
    while ((c = getopt(argc, argv, "c:g:n:r:sqotTh")) != -1)
    {
	switch(c)
	{
	  case 'c':
	    carNum = atoi(optarg);
            if ((carNum < 1) || (carNum > 8)) err = 1;
	    break;
	  case 'g':
	    if (gameNum != 0) {
		fprintf(stderr, "Only one -g or -s option allowed\n");
		err = 1;
	    }
	    gameNum = atoi(optarg);
            if ((gameNum < 0) || (gameNum > 100)) err = 1;
	    break;
	  case 'n':
	    playerName = optarg;
	    if (playerName.getLength() > 15) err = 1;
	    break;
	  case 'q':
	    quietGame = 1;
	    break;
	  case 'o':
	    disableOverlays = TRUE;
	    break;
	  case 'r':
	    numRobotCars = atoi(optarg);
            if ((numRobotCars < 0) || (numRobotCars > 8)) err = 1;
	    break;
	  case 's':
	    if (gameNum != 0) {
		fprintf(stderr, "Only one -g or -s option allowed\n");
		err = 1;
	    }
	    gameNum = DISABLE_NETWORK_GAMEID;
	    break;
	  case 't':
	    useTexturedTrack = TRUE;
	    break;
	  case 'T':
	    disableTexturedTrack = TRUE;
	    break;
	  case 'h':	// Help
	  default:
	    err = 1;
	    break;
	}
    }
  if (err)   {
    usage();
    exit(1);
  }

    if(useTexturedTrack)
    {
        gameNum=1;
    }
#ifdef _WIN32
    // Make sure slotcar can find its data files
    SbString oivhome = SbFileHelper::expandString( "$OIVHOME" );
    if (! oivhome.isEmpty() ) {
        oivhome += "/examples/source/demos/SlotCar" ;
        SoInput::addDirectoryLast(oivhome.toLatin1() );
    }
#endif

    // Init random-number generator:
#ifdef _WIN32
    srand((unsigned long)_getpid());
#else
    srand48((long)getpid());
#endif

    // Initialize the game state
    gameState = new GameState;
    gameState->playQuietly    = quietGame;
    gameState->playerName     = playerName;
    gameState->myCarNumber    = carNum;
    if (gameNum%2)
        gameState->startDelay = SHORT_START_DELAY;
    else
        gameState->startDelay = LONG_START_DELAY;   // System long
#ifdef _WIN32
    gameState->numRobotCars   = numRobotCars;
#endif

    //Create the render area
    myRenderArea = new SoXtRenderArea(myWindow);
#ifdef _WIN32
    // Note: Normal UNIX size may be too big for some PC screens!
    myRenderArea->setSize(SbVec2s(500, 400));
#else
    myRenderArea->setSize(SbVec2s(650, 400));
#endif
    myRenderArea->setTitle("Slotcar Race");
    myRenderArea->setWindowCloseCallback(exitCB, gameState);

    // The first thing we want to do is to figure out whether to use 
    // a simplified scene or not. We'll do this by drawing and timing
    // a scene graph containing a  polygon. We have to make sure the
    // window is mapped first, though, so we set up a callback node to
    // do that.
    SoCallback *mapCB = new SoCallback;
    mapCB->setCallback(waitForWindowCB, NULL);
    // Disable auto-redraw so that we render only when we get the
    // expose event for the window - this means we will render this
    // node only once.
    myRenderArea->setAutoRedraw(FALSE);
    myRenderArea->setSceneGraph(mapCB);

    myRenderArea->show();
    SoXt::show(myWindow);

    SoXt::mainLoop();

    delete myRenderArea;
    SoXt::finish();
    /* NOTREACHED */

    return 0;
}

////////////////////////////////////////////////////////////////////////
//
// Description:
//    Setup the scene and the game state.
//

void
setupGame(void)

//
////////////////////////////////////////////////////////////////////////
{
    SoSeparator *root = new SoSeparator;
    root->ref();
   
#ifdef PERF_METER  
    numFramesRendered = 0;
#endif

    // Switch, used to change between modes (racing, practicing,
    // startup screen, etc):
    SoSwitch *modeSwitch = new SoSwitch;
    root->addChild(modeSwitch);

    // This global field is changed to switch between modes.  By using
    // a global field, we can automatically change the scene graphs
    // read in without writing any code.
    SoSFInt32 *modeField = (SoSFInt32 *)
	SoDB::createGlobalField("GameMode",
				SoSFInt32::getClassTypeId());
    if (modeField == NULL) {
	fprintf(stderr, "Couldn't find GameMode global field\n");
	exit(1);
    }
    modeSwitch->whichChild.connectFrom(modeField);
    
    // Read in start screens
    SoInput reader;
    reader.openFile("$OIVHOME/examples/source/Inventor/SlotCar/data/Startup.iv");
    SoSeparator *startScreen = SoDB::readAll(&reader);
    reader.closeFile();
    if (startScreen == NULL ) {
	fprintf(stderr, "Couldn't read startup screens\n");
	exit(1);
    }
    modeSwitch->addChild(startScreen);

    // And read in the overlay heads-up display:
    reader.openFile("$OIVHOME/examples/source/Inventor/SlotCar/data/HeadsUp.iv");
    SoSeparator *headsUpGraph = SoDB::readAll(&reader);
    reader.closeFile();
    if (headsUpGraph == NULL) {
	fprintf(stderr, "Couldn't read overlay screen\n");
	exit(1);
    }
    headsUpGraph->ref();

    // Make sure that if the long track was selected, the      // System long
    // trackType globalField is either 0 or 1 - not 2 (used
    // only with the short track to turn textures on).
    SoSFInt32 *typeField = (SoSFInt32 *)
    	SoDB::getGlobalField("trackType");
    if ( (typeField->getValue() > 1) && ((gameNum%2) == 0) )
    	typeField->setValue(1);
    
    // Read the track scenery geometry from a file and add to the scene
    // Instanced five times, because the track looks the same in
    // Practicing/Starting/Racing/After-Race/Race-Over modes:
    Track *theTrack = new Track;
    SoSeparator *trackGeom = theTrack->getTrack(gameNum%2);
    modeSwitch->addChild(trackGeom);
    modeSwitch->addChild(trackGeom);
    modeSwitch->addChild(trackGeom);
    modeSwitch->addChild(trackGeom);
    modeSwitch->addChild(trackGeom);

    // Create and initialize the state structure for storing the
    // simulation data and the network for sending and receiving data
    // from other processes running the game.
    gameState->network = new Network(gameNum);
    gameState->packetTime.setToTimeOfDay();
    gameState->startTime.setToTimeOfDay();
    gameState->gameTime       = SbTime::zero();
    gameState->scene          = root;
    gameState->track          = theTrack;
    gameState->mode           = modeField;
    gameState->mode->setValue(GameState::START_SCREEN);
    gameState->carBox[0].setValue(CAR_X_MIN, CAR_Y_MIN, CAR_Z_MIN);
    gameState->carBox[1].setValue(CAR_X_MIN, CAR_Y_MIN, CAR_Z_MAX);
    gameState->carBox[2].setValue(CAR_X_MAX, CAR_Y_MIN, CAR_Z_MIN);
    gameState->carBox[3].setValue(CAR_X_MAX, CAR_Y_MIN, CAR_Z_MAX);
    gameState->carBox[4].setValue(CAR_X_MIN, CAR_Y_MAX, CAR_Z_MIN);
    gameState->carBox[5].setValue(CAR_X_MIN, CAR_Y_MAX, CAR_Z_MAX);
    gameState->carBox[6].setValue(CAR_X_MAX, CAR_Y_MAX, CAR_Z_MIN);
    gameState->carBox[7].setValue(CAR_X_MAX, CAR_Y_MAX, CAR_Z_MAX);
    gameState->cameraUpDown = 10;
    gameState->cameraBackForth = 10;

#ifndef _WIN32
    // Now done in mouseInputCB to allow runtime option setting
    // Create robot cars, and add them to the track:
    // Turn textures off if necessary
    if ( typeField->getValue() == 2)
        trackGeom->addChild(SoNode::getByName("OffTexture"));
    for (int i = 0; i < numRobotCars; i++) {
	RobotCar *car = new RobotCar;
	// old: trackGeom->addChild(car->getCarRoot());
	theTrack->addCar(car);
	gameState->addCar(car, FALSE);
    }
#endif

    // Add event callback AS FIRST CHILD for handling mouse events:
    SoEventCallback *evCB = new SoEventCallback;
    root->insertChild(evCB, 0); 	// old: root->addChild(evCB);
    evCB->addEventCallback(SoLocation2Event::getClassTypeId(),
			   mouseInputCB, gameState);
    evCB->addEventCallback(SoMouseButtonEvent::getClassTypeId(),
			   mouseInputCB, gameState);
    evCB->addEventCallback(SoKeyboardEvent::getClassTypeId(),
			   mouseInputCB, gameState);

    // Setup the camera to be parented to the car.  
    SoPerspectiveCamera  *camera    = new SoPerspectiveCamera;
    camera->position.setValue(SbVec3f(0.0, 1.5, 0.0));
    camera->nearDistance = 1.0;
    trackGeom->insertChild(camera, 0);
    gameState->camera = camera;

    // Search for zbuffer callback nodes and setup their callbacks
    SoCallback *cbNode = (SoCallback *)SoNode::getByName("Z_BUFFER_ENABLE");
    if (cbNode != NULL)
        cbNode->setCallback(zEnableCB);
    cbNode = (SoCallback *)SoNode::getByName("Z_BUFFER_DISABLE");
    if (cbNode != NULL)
        cbNode->setCallback(zDisableCB);
    
    // Simulated sunlight:
    SoDirectionalLight   *hot_sun       = new SoDirectionalLight;
    // old: hot_sun->direction.setValue(SbVec3f(0.1, -1.0, 0.1));
    hot_sun->direction.setValue(SbVec3f(-0.5, -1.0, -0.5));
    trackGeom->insertChild(hot_sun, 1);

    // Set farDistance for the camera.
    // Currently, this is fixed, but should be dynamically calculated.
    //camera->farDistance = 1200;
    if (typeField->getValue() == 2)
    	camera->farDistance = 7500;
    else
    	camera->farDistance = 5000;

    // Initialize the idle sensor for running the simulation
    gameState->simulationSensor = new SoIdleSensor(runSimulation,
						   gameState);

#ifdef PERF_METER
    // Create a performance meter for showing frames per second
    SoSeparator *oldRoot = root;
    root = addUpdateCounter(oldRoot);
#endif

#ifdef PERF_METER
    if (!disableOverlays)
        initPerfMeter(headsUpGraph);
#endif

    myRenderArea->setAutoRedraw(TRUE);

#ifdef _WIN32
    if (!disableOverlays)
      root->addChild(headsUpGraph);
    myRenderArea->setSceneGraph(root);
#else
    myRenderArea->setSceneGraph(root);
#endif
    myRenderArea->setBackgroundColor(SbColor(0.25f, 0.45f, 0.25f));

    // Start the game!!
    SoDB::setDelaySensorTimeout(SbTime(.03));

    if (gameNum != DISABLE_NETWORK_GAMEID) {
#if !defined(_WIN32) && !defined(__APPLE__)
	XtAppAddInput(
	    SoXt::getAppContext(),
	    gameState->network->getfd(),
	    (XtPointer) XtInputReadMask,
	    packetCB,
	    (XtPointer) gameState);
#endif
    }

}


#ifdef PERF_METER
////////////////////////////////////////////////////////////////////
//
// Initialize the performance meter in the overlay planes
//
static void
initPerfMeter(SoGroup *overlayGroup)
{
    //
    // Create the overlay scene    
    //
    SoSeparator *perfMeter = new SoSeparator;
    perfMeter->ref();

    // Add the text string to display the tris per second
    SoFont *perfMeterFont = new SoFont;
    SoColorIndex *cind    = new SoColorIndex;
    cind->index = 3;
    perfMeterFont->size = 18;
    perfMeterFont->name = "Helvetica";
    perfMeter->addChild(perfMeterFont);
    perfMeter->addChild(cind);
    SoText2 *perfMeterText = new SoText2;
    SoTranslation *trans = new SoTranslation;
    trans->translation.setValue(-0.8f, -0.95f, 0.0f);
    perfMeter->addChild(trans);
    perfMeterText->justification = SoText2::LEFT;
    perfMeter->addChild(perfMeterText);

    overlayGroup->addChild(perfMeter);

    // Update the performance meter every two seconds
    SoTimerSensor *perfSensor = 
		new SoTimerSensor(perfSensorCB, perfMeterText);
    perfSensor->setInterval(2.0f);
    perfSensor->schedule();
}


////////////////////////////////////////////////////////////////////
//
// A sensor callback in the overlay scene updates the
// performance meter while the object is animating.
// 
static void
perfSensorCB(void *data, SoSensor *)
{
    SoText2 *perfText = (SoText2 *) data;
    static SbBool initialized = FALSE;
    static SbTime startTime; 
    static double lastFrames = 0;
    char str[100];
    
    if (initialized) {
	SbTime timeDiff = SbTime::getTimeOfDay() - startTime;
	double frames = 
	    numFramesRendered * 1000.0 / timeDiff.getMsecValue();
	frames = floor(frames * 100.0 + 0.5) / 100.0;

	// Update the perf meter display if the numbers have changed
	if (frames != lastFrames) {
	    sprintf (str, "%.2lf frames/sec", frames);
	    perfText->string.setValue(str);
	    lastFrames = frames;
	}
	        
	numFramesRendered = 0;
	startTime = SbTime::getTimeOfDay();
    }
    else {
	// Reset the timer when we detect the object has
	// started animating.
	numFramesRendered = 0;
	startTime = SbTime::getTimeOfDay();
	initialized = TRUE;
    }
}

////////////////////////////////////////////////////////////////////
//
// Add a callback node to the main scene which will count the number 
// of frames that have been rendered (to compute the tris/sec).
// Make sure the callback doesn't get cached or ruin the caching for
// the rest of the scene
//
static SoSeparator *
addUpdateCounter(SoNode *root)
{
    SoSeparator *newRoot = new SoSeparator;
    newRoot->ref();

    SoCallback *countUpdates = new SoCallback;
    countUpdates->setCallback(countUpdatesCB);
    newRoot->addChild(countUpdates);

    newRoot->addChild(root);

    return (newRoot);
}
	
static void
countUpdatesCB(void *, SoAction *action)
{
    if (action->isOfType(SoGLRenderAction::getClassTypeId())) {
        SoCacheElement::invalidate(action->getState());
	numFramesRendered++;
    }
}

#endif


