/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Fabien ARNAUD & Romain BERTHON (MMM YYYY)
** Modified by : Tristan Mehamli (MMM YYYY)
**=======================================================================*/

#include <QtCore/qglobal.h>

#include <Inventor/Qt/OivQtCompat.h>
#include "QLmv.h"
#include "QLmvIcons.h"
#include "QLmvAdaptiveViewing.h"
#include "QLmvSplitAction.h"
#include "InlineLoader.h"

#include <QtCore/QTimer>
#include <QtCore/QSettings>
#include <QPixmap>
#include <QCloseEvent>
#include <QDropEvent>
#include <QDragEnterEvent>
#include <QApplication>
#include <QMainWindow>
#include <QMenuBar>
#include <QToolButton>
#include <QLineEdit>
#include <QPushButton>
#include <QProgressDialog>
#include <QMessageBox>
#include <QMenu>
#include <QAction>
#include <QToolBar>
#include <QFileDialog>
#include <QtCore/QUrl>
#include <QtCore/QList>
#include <QMimeData>

#ifdef WIN32
#pragma warning( disable : 4996 ) // Ignore VRML deprecation warnings
#endif

#include <Inventor/Qt/viewers/SoQtExaminerViewer.h>
#include <Inventor/Qt/viewers/SoQtWalkViewer.h>
#include <Inventor/Qt/viewers/SoQtPlaneViewer.h>
#include <Inventor/Qt/viewers/SoQtFlyViewer.h>
#include <Inventor/Qt/viewers/SoQtCollisionViewer.h>
#include <Inventor/SoInput.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoInfo.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoFile.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/engines/SoTrackFollower.h>
#include <Inventor/components/stereo/SoStereoDialog.h>
#include <Inventor/components/stereo/SoAnaglyphStereo.h>
#include <Inventor/actions/SoReorganizeAction.h>
#include <Inventor/actions/SoShapeSimplifyAction.h>
#include <Inventor/actions/SoGlobalSimplifyAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/actions/SoRayPickAction.h>
#include <Inventor/actions/SoGetPrimitiveCountAction.h>
#include <Inventor/simplifier/SoDecimator.h> 
#include <Inventor/sensors/SoOneShotSensor.h> 
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/components/stereo/SoInterlacedStereo.h>
#include <Inventor/components/stereo/SoAnaglyphStereo.h>
#include <Inventor/components/stereo/SoHalfScreenStereo.h>
#include <Inventor/components/stereo/SoRawStereo.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/antialiasing/SoFullSceneAntialiasingParameters.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>

// SpaceBall
#include <Inventor/Qt/devices/SoQtSpaceball.h>
#include <Inventor/events/SoSpaceballButtonEvent.h>
#include <Inventor/events/SoMotion3Event.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/sys/SoGL.h>
#include "MoveAround.h"


enum MenuItemsId 
{
  MNU_VP_VIEW_ALL = 0,
  MNU_ANIM_INSIDE_VIEW = 0,
  MNU_ANIM_VIEW_FROM_BEHIND = 1,
  MNU_ANIM_TOP_VIEW = 2,
  MNU_ANIM_FREE_VIEW = 3,
  MNU_ANIM_POINT_AT_OBJECT = 4,
  MNU_ANIM_DISPLAY_OBJECT = 5,
  MNU_ANIM_ROLL_WITH_OBJECT = 6,
  MNU_ANIM_NONE = 7,
  MNU_ANIM_FIRST_TRACK = 8
};

#define local_MAX(_A,_B) (((_A)<(_B))?(_B):(_A))
#define local_MIN(_A,_B) (((_A)>(_B))?(_B):(_A))
#define M_EPSILON 0.001f

#ifdef _WIN32
#define QLMV_HELP_FILE_NAME "../DemosTools/Demos/QtLargeModelViewer.chm"
#else
#define QLMV_HELP_FILE_NAME "QtLargeModelViewer.html"
#endif

//to store Most Recent Files
const QString WINDOWS_REGISTRY = "VsgOivCppDemo810";
const QString APP_KEY = "/QtLargeModelViewer/MRU/";

////////////////////////////////////////////////////////////////////////
// This class updates a QProgressDialog during iv file loading.
class LiveInput : public SoInput
{
public:
  LiveInput( QProgressDialog* progressDialog = NULL );

  virtual ~LiveInput();

protected:

  virtual void updateReadPercent( double readPercentage );

private:

  QProgressDialog* m_progressDialog;
};


////////////////////////////////////////////////////////////////////////
LiveInput::LiveInput( QProgressDialog* progressDialog )
:SoInput(), m_progressDialog( progressDialog )
{
}


////////////////////////////////////////////////////////////////////////
LiveInput::~LiveInput()
{
}


////////////////////////////////////////////////////////////////////////
void 
LiveInput::updateReadPercent( double readPercentage )
{
  if (m_progressDialog)
    m_progressDialog->setValue( int(readPercentage) );
}


////////////////////////////////////////////////////////////////////////
// Constructor
QLmv::QLmv (const char* configFileName, QWidget* parent, const char* name) : QMainWindow (parent)
{
  setObjectName( name );
  // Main window
  setWindowTitle(tr ("QtLargeModelViewer"));
  setWindowIcon(QPixmap (appIconXpmString));
  setGeometry (100, 100, 520, 550);

  //Accept Drop
  setAcceptDrops(TRUE);

  // Make initializations
  SoQt::init (this);
  if (configFileName != NULL) SoPreferences::readFile (configFileName);
  MoveAround::initClass();
  SoFile::setSearchOK (TRUE);
  //set a default color if there is no config file
  SbString bkgColor = SoPreferences::getString("OIV_BACKGROUND_COLOR","");
  if (bkgColor.getLength() == 0)
	SoPreferences::setColor ("OIV_BACKGROUND_COLOR", SbColor (0, 0, 0.3f));

  // Build the toolbar
  m_toolbar = new QToolBar( this );
  addToolBar( m_toolbar );
  QIcon* fileopenIcon = new QIcon( QPixmap(openfileXpmString) );
  fileopenIcon->addPixmap( QPixmap(openfileLightXpmString), QIcon::Disabled );
  m_fileOpenBtn = new QToolButton( m_toolbar );
  m_fileOpenBtn->setIcon( *fileopenIcon );
  m_fileOpenBtn->setText( "Open file" );
  connect( m_fileOpenBtn, SIGNAL(clicked()), this, SLOT(onMnuFileOpen()) );
  m_toolbar->addWidget( m_fileOpenBtn );
  QIcon* filereopenIcon = new QIcon( QPixmap(reopenfileXpmString) );
  filereopenIcon->addPixmap( QPixmap(reopenfileLightXpmString), QIcon::Disabled );
  m_fileReopenBtn = new QToolButton( m_toolbar );
  m_fileReopenBtn->setIcon( *filereopenIcon );
  m_fileReopenBtn->setText( "Reload file" );
  connect( m_fileReopenBtn, SIGNAL(clicked()), this, SLOT(onMnuFileReload()) );
  m_toolbar->addWidget( m_fileReopenBtn );
  m_fileReopenBtn->setEnabled( false );
  QIcon* stereoIcon = new QIcon( QPixmap(stereoXpmString) );
  stereoIcon->addPixmap( QPixmap(stereoLightXpmString), QIcon::Disabled );
  m_stereoBtn = new QToolButton( m_toolbar );
  m_stereoBtn->setIcon( *stereoIcon );
  m_stereoBtn->setText( "Stereo" );
  connect( m_stereoBtn, SIGNAL(clicked()), this, SLOT(onBtnStereo()) );
  m_toolbar->addWidget( m_stereoBtn );
  QIcon* gravityIcon = new QIcon( QPixmap(gravityXpmString) );
  gravityIcon->addPixmap( QPixmap(gravityLightXpmString), QIcon::Disabled );
  m_gravityBtn = new QToolButton( m_toolbar );
  m_gravityBtn->setIcon( *gravityIcon );
  m_gravityBtn->setText( "Apply gravity" );
  connect( m_gravityBtn, SIGNAL(clicked()), this, SLOT(onMnuApplyGravity()) );
  m_toolbar->addWidget( m_gravityBtn );

  // Build popup menus
  m_fileMenu = new QMenu;
  m_fileMenu->addAction( *fileopenIcon, tr("&Open..."), this, SLOT (onMnuFileOpen()) );
  m_fileReloadMenuItem = m_fileMenu->addAction(*filereopenIcon, tr("&Reload"), this, SLOT (onMnuFileReload()));
  m_fileReloadMenuItem->setCheckable( true );
  m_fileReloadMenuItem->setEnabled( false );
  m_fileMenu->addSeparator(); 
  m_recentFilesMenu = m_fileMenu->addMenu( "Recent Files..." );
  m_fileMenu->addMenu( m_recentFilesMenu );
  m_fileMenu->addSeparator();
  // add the last separator
	m_fileMenu->addSeparator();
  m_fileMenu->addAction( tr("E&xit"), this, SLOT (fileQuit()) );


  //Most Recent Files List
#ifdef WIN32
  QSettings settings( QSettings::SystemScope, WINDOWS_REGISTRY );
#else
  QSettings settings;
#endif
  m_fileMenuConnected = false;
  //this line works only for Windows and does nothing on UNIX
  for ( int i = 0; i < MAX_RECENTFILES; ++i ) {
	  QString filename = settings.value( APP_KEY + "File" + QString::number(i + 1) ).toString();
	  if ( !filename.isEmpty() )
		  m_recentFiles.push_back( filename );
  }
  if ( m_recentFiles.count() )
	  updateRecentFilesMenu();

  m_transparencyMenu = new QMenu;
  m_noSortTransparencyMenuItem = m_transparencyMenu->addAction(tr("No sort"), this, SLOT(onMnuNoSortTransparency()));
  m_opaqueFirstTransparencyMenuItem = m_transparencyMenu->addAction(tr("Opaque First"), this, SLOT(onMnuOpaqueFirstTransparency()));
  m_sortedObjectTransparencyMenuItem = m_transparencyMenu->addAction(tr("Sorted object"), this, SLOT(onMnuSortedObjectTransparency()));
  m_sortedPixelTransparencyMenuItem = m_transparencyMenu->addAction(tr("Sorted pixel"), this, SLOT(onMnuSortedPixelTransparency()));
  m_transparencyMenu->addSeparator();
  m_alphaTestTransparencyMenuItem = m_transparencyMenu->addAction(tr("Alpha Test"), this, SLOT (onMnuAlphaTestTransparency()));

  m_viewMenu = new QMenu;
  m_examinerViewerMenuItem = m_viewMenu->addAction(tr("Examiner Viewer"), this, SLOT (onMnuExaminerViewer()));
  m_walkViewerMenuItem = m_viewMenu->addAction(tr("Walk Viewer"), this, SLOT (onMnuWalkViewer()));
  m_flyViewerMenuItem = m_viewMenu->addAction(tr("Fly Viewer"), this, SLOT (onMnuFlyViewer()));
  m_planeViewerMenuItem = m_viewMenu->addAction(tr("Plane Viewer"), this, SLOT (onMnuPlaneViewer()));
  m_viewMenu->addSeparator();
  m_collisionDetectionMenuItem = m_viewMenu->addAction(tr("Collision Detection"), this, SLOT (onMnuCollisionDetection()));
  m_applyGravityMenuItem = m_viewMenu->addAction(*gravityIcon, tr("Apply Gravity"), this, SLOT (onMnuApplyGravity()));
  m_viewMenu->addSeparator();
  m_stereoMenuItem = m_viewMenu->addAction(*stereoIcon, tr("Stereo"), this, SLOT (onMnuStereo()));
  m_stereoDialogMenuItem = m_viewMenu->addAction(tr("Stereo Settings..."), this, SLOT (onMnuStereoDialog()));
  m_adaptiveViewMenuItem = m_viewMenu->addAction(tr("Adaptive Viewing..."), this, SLOT (onMnuAdaptiveView()));
  m_viewMenu->addSeparator();
  m_transparencyMenu->setTitle("Transparency");
  m_viewMenu->addMenu(m_transparencyMenu);

  // - FSAA
  m_viewMenu->addSeparator();
  m_fsaaMenuItem = m_viewMenu->addAction(tr("Full scene antialiasing"), this, SLOT (onMenuViewFsaa()));
  m_fsaaMenuItem->setCheckable( true );
  m_fsaaMenuItem->setChecked( false );

  m_viewMenu->addSeparator();
  m_viewToolbarMenuItem = m_viewMenu->addAction(tr("Tool Bar"), this, SLOT (onMnuViewToolbar()));
  m_viewToolbarMenuItem->setCheckable( true );
  m_viewToolbarMenuItem->setChecked( true );
  m_viewStatusBarMenuItem = m_viewMenu->addAction(tr("Status Bar"), this, SLOT (onMnuViewStatusbar()));
  m_viewStatusBarMenuItem->setCheckable( true );
  m_viewStatusBarMenuItem->setChecked( true );
  m_viewRenderStatsMenuItem = m_viewMenu->addAction(tr("Render Statistics"), this, SLOT (onMnuViewRenderStats()));
  m_viewRenderStatsMenuItem->setCheckable( true );
  m_viewRenderStatsMenuItem->setChecked( true );
  m_viewSceneInfosMenuItem = m_viewMenu->addAction(tr("Scene Informations"), this, SLOT (onMnuViewSceneInfos()));
  m_viewSceneInfosMenuItem->setVisible( false );

  m_processingMenu = new QMenu;
  m_loadInlinesItem = m_processingMenu->addAction(tr("Load Inlines Immediately"), this, SLOT (onMnuLoadInlines()));
  m_processingMenu->addSeparator();
  m_reorganizeMenuItem = m_processingMenu->addAction(tr("Reorganize (triangle strips)"), this, SLOT (onMnuReorganizeProcess()));
  m_shapeSimplifyMenuItem = m_processingMenu->addAction(tr("Shape Simplify"), this, SLOT (onMnuShapeSimplifyProcess()));
  m_globalSimplifyMenuItem = m_processingMenu->addAction(tr("Global Simplify"), this, SLOT (onMnuGlobalSimplifyProcess()));
  m_shapeSimplifyTSMenuItem = m_processingMenu->addAction(tr("Shape Simplify (triangle strips)"), this, SLOT (onMnuShapeSimplifyTSProcess()));
  m_globalSimplifyTSMenuItem = m_processingMenu->addAction(tr("Global Simplify (triangle strips)"), this, SLOT (onMnuGlobalSimplifyTSProcess()));
  m_processingMenu->addSeparator();
  m_splitShapeMenuItem = m_processingMenu->addAction(tr("Split Shapes..."), this, SLOT (onMnuSplitShapesProcess()));
  m_loadInlinesItem->setCheckable( true );
  m_loadInlinesItem->setChecked( true );

  m_viewpointsMenu = new QMenu;
  m_vpMenuIdsMap.clear();
  m_vpMenuIdsMap[MNU_VP_VIEW_ALL] = m_viewpointsMenu->addAction( tr("View All") );
  connect( m_viewpointsMenu, SIGNAL(triggered(QAction*)), this, SLOT(onMnuViewpoints(QAction*)) );

  m_animationsMenu = new QMenu;
  m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW] = m_animationsMenu->addAction( tr("Inside View") );
  m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setCheckable( true );
  m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND] = m_animationsMenu->addAction( tr("View From Behind") );
  m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setCheckable( true );
  m_animMenuIdsMap[MNU_ANIM_TOP_VIEW] = m_animationsMenu->addAction(tr( "Top View") );
  m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setCheckable( true );
  m_animMenuIdsMap[MNU_ANIM_FREE_VIEW] = m_animationsMenu->addAction( tr("Free View") );
  m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setCheckable( true );
  m_animationsMenu->addSeparator();
  m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]= m_animationsMenu->addAction( tr("Point At Object") );
  m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setCheckable( true );
  m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT] = m_animationsMenu->addAction( tr("Show Object") );
  m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setCheckable( true );
  m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT] = m_animationsMenu->addAction( tr("Roll With Object") );
  m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setCheckable( true );
  m_animationsMenu->addSeparator();
  m_animMenuIdsMap[MNU_ANIM_NONE] = m_animationsMenu->addAction( tr("No Selection") );
  m_animMenuIdsMap[MNU_ANIM_NONE]->setCheckable( true );
  connect( m_animationsMenu, SIGNAL(triggered(QAction*)), this, SLOT (onMnuAnimations(QAction*)) );

  // Build the menu bar
  m_menubar = new QMenuBar(this);
  setMenuBar( m_menubar );
  m_fileMenu->setTitle( "&File" );
  m_viewMenu->setTitle( "&View" );
  m_processingMenu->setTitle( "&Processing" );
  m_viewpointsMenu->setTitle( "Vie&wPoints" );
  m_animationsMenu->setTitle( "&Animations" );
  m_fileMenuId = m_menubar->addMenu( m_fileMenu );
  m_viewMenuId = m_menubar->addMenu(  m_viewMenu );
  m_processingMenuId = m_menubar->addMenu(  m_processingMenu );
  m_viewpointsMenuId = m_menubar->addMenu( m_viewpointsMenu );
  m_animationsMenuId = m_menubar->addMenu(  m_animationsMenu );
  m_viewSceneInfosMenuItem->setEnabled( false );
  m_processingMenuId->setEnabled( false );
  m_viewpointsMenuId->setEnabled( false );
  m_animationsMenuId->setEnabled( false );

  // Add a status bar
  m_fpsDisplay = new QLineEdit (this);
  m_fpsDisplay->setFixedWidth (m_fpsDisplay->fontMetrics().horizontalAdvance (" 000.0 fps ") + 10);
  m_fpsDisplay->setReadOnly (TRUE);
  m_fpsDisplay->setAlignment (Qt::AlignRight);
  m_decimationDisplay = new QLineEdit (this);
  m_decimationDisplay->setFixedWidth( m_decimationDisplay->fontMetrics().horizontalAdvance( " 100.0 % " ) + 10 );
  m_decimationDisplay->setReadOnly (TRUE);
  m_decimationDisplay->setAlignment (Qt::AlignRight);
  m_numTrianglesDisplay = new QLineEdit (this);
  m_numTrianglesDisplay->setFixedWidth( m_numTrianglesDisplay->fontMetrics().horizontalAdvance( " 0000000 triangles " ) + 10 );
  m_numTrianglesDisplay->setReadOnly (TRUE);
  m_numTrianglesDisplay->setAlignment (Qt::AlignRight);
  m_statusbar = new QStatusBar (this);
  setStatusBar( m_statusbar );
  m_statusbar->addPermanentWidget( m_fpsDisplay );
  m_statusbar->addPermanentWidget( m_decimationDisplay );
  m_statusbar->addPermanentWidget( m_numTrianglesDisplay );

  // Dialogs
  m_traversalTuning = NULL;
  m_adaptiveViewing = NULL;

  // Create the basics scene graphs
  m_loadedSG = NULL;
  m_displayedSG = new SoSeparator;
  m_displayedSG->ref ();

// Inline Fetcher is used to resolve WWWInlines on local file system
  inlineLoader = new InlineLoader;
  inlineLoader->setReadingInlineCallback(readingInlineCB, this);

  // Shape Hints node for controlling back-face culling:
  shapeHints = new SoShapeHints;
  shapeHints->shapeType = SoShapeHints::SOLID;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeHints->faceType.setIgnored(TRUE);
  shapeHints->creaseAngle = 0.5;
  shapeHints->creaseAngle.setIgnored(TRUE);
  m_displayedSG->addChild(shapeHints);

  // SpaceBall
  m_spaceballTranslation = new SoTranslation;
  m_spaceballTranslation->ref();
  m_spaceballRotation = new SoRotation;
  m_spaceballRotation->ref();

  if (SoQtSpaceball::exists()) {
    m_spaceball_available = TRUE;
    m_spaceballCB = new SoEventCallback;
    m_displayedSG->addChild(m_spaceballCB);

    // Set up event callbacks
    m_spaceballCB->addEventCallback(SoSpaceballButtonEvent::getClassTypeId(),
                         onSpaceballButtonCB, 
                         this);
    m_spaceballCB->addEventCallback(SoMotion3Event::getClassTypeId(),
                         onMotion3TranslationCB, 
                         this);
    m_spaceballCB->addEventCallback(SoMotion3Event::getClassTypeId(),
                         onMotion3RotationCB, 
                         this);
    m_sb = new SoQtSpaceball;
  } 
  else {
    m_spaceball_available = FALSE;
    m_sb = NULL;
  }

  
  // Mouse event for moving around a followed object
  m_movingAroundCB = new SoEventCallback;
  m_movingAroundCB->addEventCallback (SoLocation2Event::getClassTypeId(), onMouseMoveCB, this);
  m_movingAroundCB->addEventCallback (SoMouseButtonEvent::getClassTypeId(), onMouseMoveCB, this);
  m_displayedSG->addChild (m_movingAroundCB);
  m_freeViewMode = NONE;

  // Traversal nodes
  m_occlusionStatsNumObjRendered = -1;
  m_occlusionStatsNumObj = -1;
  m_occlusionStatsNumOccluders = -1;

  m_separatorTraversalNode = new SoSeparator;
  m_separatorTraversalNode->ref ();

  // Selection node
  m_selectionNode = new SoSelection;
  m_selectionNode->ref();

  // Viewers
  m_examinerViewer = NULL;
  m_walkViewer = NULL;
  m_collisionViewer = NULL;
  m_collisionDistance = 0.25f;
  m_flyViewer = NULL;
  m_planeViewer = NULL;
  m_currentViewer = NULL;

  //stereo
  m_stereoMenuItem->setCheckable( true );
  m_stereoMenuItem->setChecked( false );

  //stereoDialog
  posStereoDialog = QPoint(0,0);

  // Cameras, viewpoints and tracks
  m_cameraParent = new SoGroup;
  m_cameraParent->ref();
  m_trackSwitch = new SoSwitch;
  m_trackSwitch->ref();
  displayFollowedObject (FALSE);
  m_moveAroundEngine = new MoveAround;
  m_moveAroundEngine->ref();
  rollWithObject (TRUE);
  m_currentTrack = -1;
  m_animMenuIdsMap[MNU_ANIM_NONE]->setCheckable( true );
  m_animMenuIdsMap[MNU_ANIM_NONE]->setChecked( true );
  followAsInsideView ();

  // Render statistics
  m_countTrianglesAction = new SoGetPrimitiveCountAction;
  m_countTrianglesSensor = new SoOneShotSensor(onCountTrianglesCB, this);

  // Set the viewer
  SbString viewerType = SoPreferences::getString("LMV_VIEWER_TYPE", "EXAMINER");
  m_examinerViewerMenuItem->setCheckable( true );
  m_examinerViewerMenuItem->setChecked( true );
  if (viewerType == "WALK") {
    setViewerType (WALK);
  }
  else if (viewerType == "FLY") {
    setViewerType (FLY);
  }
  else if (viewerType == "PLANE") {
    setViewerType (PLANE);
  }
  else {
    setViewerType (EXAMINER);
  }

  // Retrieve the transparency from environment
  switch (m_currentViewer->getTransparencyType()) {
    case SoGLRenderAction::NO_SORT:
      m_noSortTransparencyMenuItem->setCheckable(true);
      m_noSortTransparencyMenuItem->setChecked(true);
      break;
    case SoGLRenderAction::OPAQUE_FIRST:
      m_opaqueFirstTransparencyMenuItem->setCheckable(true);
      m_opaqueFirstTransparencyMenuItem->setChecked(true);
      break;
    case SoGLRenderAction::SORTED_OBJECT:
      m_sortedObjectTransparencyMenuItem->setCheckable(true);
      m_sortedObjectTransparencyMenuItem->setChecked(true);
      break;
    case SoGLRenderAction::SORTED_PIXEL:
      m_sortedPixelTransparencyMenuItem->setCheckable(true);
      m_sortedPixelTransparencyMenuItem->setChecked(true);
      break;
    default:
      setTransparency (SoGLRenderAction::SORTED_OBJECT);
      break;
  }
  m_alphaTestTransparencyMenuItem->setCheckable( true );
  m_alphaTestTransparencyMenuItem->setChecked( m_currentViewer->getGLRenderAction()->isAlphaTest() );
  
  m_gravityOffset = SoPreferences::getFloat ("LMV_GRAVITY_OFFSET", 1.7f);

  resize( 520, 550 );
}


////////////////////////////////////////////////////////////////////////
// Destructor
QLmv::~QLmv ()
{
  m_spaceballTranslation->unref();
  m_spaceballRotation->unref();
  m_displayedSG->unref();
  m_cameraParent->unref();
  m_selectionNode->unref();
  m_separatorTraversalNode->unref ();
  m_trackSwitch->unref();
  m_moveAroundEngine->unref();
  delete inlineLoader;
  delete m_countTrianglesAction;
  delete m_countTrianglesSensor;
  
  delete m_examinerViewer;
  delete m_walkViewer;
  delete m_flyViewer;
  delete m_planeViewer;
  delete m_collisionViewer;

  MoveAround::exitClass();
  SoQt::finish();
}


////////////////////////////////////////////////////////////////////////
// Set the viewer to be used
void QLmv::setViewerType (ViewerType t)
{
  switch (t) {
    case WALK:
      if ( !m_walkViewerMenuItem->isChecked() ) 
      {
        if ( !m_walkViewerMenuItem->isCheckable() )
          m_walkViewerMenuItem->setCheckable( true );
        m_walkViewerMenuItem->setChecked( true );
        m_examinerViewerMenuItem->setChecked( false );
        m_planeViewerMenuItem->setChecked( false );
        m_flyViewerMenuItem->setChecked( false );
        //return;
      }
      break;
    case FLY:
      if ( !m_flyViewerMenuItem->isChecked() ) 
      {
        if ( !m_flyViewerMenuItem->isCheckable() )
          m_flyViewerMenuItem->setCheckable( true );
        m_flyViewerMenuItem->setChecked( true );
        m_examinerViewerMenuItem->setChecked( false );
        m_walkViewerMenuItem->setChecked( false );
        m_planeViewerMenuItem->setChecked( false );
        //return;
      }
      break;
    case PLANE:
      if ( !m_planeViewerMenuItem->isChecked() ) 
      {
        if ( !m_planeViewerMenuItem->isCheckable() )
          m_planeViewerMenuItem->setCheckable( true );
        m_planeViewerMenuItem->setChecked( true );
        m_examinerViewerMenuItem->setChecked( false );
        m_walkViewerMenuItem->setChecked( false );
        m_flyViewerMenuItem->setChecked( false );
        //return;
      }
      break;
    case EXAMINER:
      if ( !m_examinerViewerMenuItem->isChecked() ) 
      {
        if ( !m_examinerViewerMenuItem->isCheckable() )
          m_examinerViewerMenuItem->setCheckable( true );
        m_examinerViewerMenuItem->setChecked( true );
        m_walkViewerMenuItem->setChecked( false );
        m_planeViewerMenuItem->setChecked( false );
        m_flyViewerMenuItem->setChecked( false );
        //return;
      }
      break;
  }

  if ( m_currentViewer ) // Avoids setCentralWidget to delete which it doesn't own after all!!
    layout()->removeWidget( m_currentViewer->getWidget() );

  SoBaseStereo *myCurrentViewType = NULL;
  if (m_currentViewer !=NULL) {
    SoQtStereoDialog *stereoDialog = m_currentViewer->getStereoDialog();
    if (stereoDialog != NULL) {
      posStereoDialog = (stereoDialog->getWidget()->pos());
      stereoDialog->getWidget()->close();
      delete stereoDialog;
      stereoDialog = NULL;
    }
  }

  // Disable the old viewer
  bool shouldDisplayRenderStats = m_viewRenderStatsMenuItem->isChecked();
  if (m_currentViewer != NULL) {
    m_currentViewer->setSceneGraph (NULL);
    m_currentViewer->getWidget()->hide();
    if (shouldDisplayRenderStats) {
      displayRenderStats (FALSE);
    }
  }

  m_examinerViewerMenuItem->setChecked( false );
  m_walkViewerMenuItem->setChecked( false );
  m_flyViewerMenuItem->setChecked( false );
  m_planeViewerMenuItem->setChecked( false );
  if ( !m_collisionDetectionMenuItem->isCheckable() )
    m_collisionDetectionMenuItem->setCheckable( true );
  m_collisionDetectionMenuItem->setChecked( false );
  m_collisionDetectionMenuItem->setEnabled( false );
  if ( !m_alphaTestTransparencyMenuItem->isCheckable() )
    m_alphaTestTransparencyMenuItem->setCheckable( true );
  m_alphaTestTransparencyMenuItem->setChecked( false );
  if ( !m_applyGravityMenuItem->isCheckable() )
    m_applyGravityMenuItem->setCheckable( true );
  m_applyGravityMenuItem->setChecked( false );
  m_gravityBtn->setEnabled( false );

  //SpaceBall

if ((m_spaceball_available) &&(m_currentViewer != NULL)) {
    m_currentViewer->unregisterDevice(m_sb);
  }
  else if((m_spaceball_available) &&(m_currentViewer == NULL))
  {
    //set the default SpaceBall mode
    m_spaceball_rotationMode    = 1;
    m_spaceball_translationMode = 0;
  }

  // Switch to the new viewer (create it if necessary)
  switch (t) {
    case EXAMINER:
      if (m_examinerViewer == NULL) {
        m_examinerViewer = new SoQtExaminerViewer (this);
        m_examinerViewer->helpFileName = QLMV_HELP_FILE_NAME;
      }
      if (m_currentViewer != NULL) {
        m_examinerViewer->getGLRenderAction()->setAlphaTest( m_alphaTestTransparencyMenuItem->isChecked() );
        m_examinerViewer->setTransparencyType (m_currentViewer->getTransparencyType ());
        myCurrentViewType = m_currentViewer->getStereoViewType ();
        m_examinerViewer->setStereoActive (m_currentViewer->isStereoActive());
      }
      m_currentViewer = m_examinerViewer;
      m_examinerViewerMenuItem->setChecked( true );
     break;

    case WALK:
      if (m_walkViewer == NULL) {
        m_walkViewer = new SoQtWalkViewer (this);
        m_walkViewer->helpFileName = QLMV_HELP_FILE_NAME;
      }
      if (m_currentViewer != NULL) {
        m_walkViewer->getGLRenderAction()->setAlphaTest( m_alphaTestTransparencyMenuItem->isChecked() );
        m_walkViewer->setTransparencyType (m_currentViewer->getTransparencyType ());
        myCurrentViewType = m_currentViewer->getStereoViewType ();
        m_walkViewer->setStereoActive (m_currentViewer->isStereoActive());
      }
      m_currentViewer = m_walkViewer;
      if (m_collisionViewer == NULL) {
        m_collisionViewer = new SoQtCollisionViewer(m_walkViewer);
        m_collisionViewer->setNumSteps(6); 
        m_collisionViewer->setDistance (m_collisionDistance);
        m_collisionViewer->setZBufferOptimization( TRUE, TRUE );
      }
      m_collisionViewer->setCollisionDetection( m_collisionDetectionMenuItem->isChecked() );

      m_walkViewerMenuItem->setChecked( true );
      m_collisionDetectionMenuItem->setChecked( true );
      m_applyGravityMenuItem->setChecked( true );
      m_gravityBtn->setEnabled( true );
      break;

    case FLY:
      if (m_flyViewer == NULL) {
        m_flyViewer = new SoQtFlyViewer (this);
#ifdef _WIN32
        m_flyViewer->helpFileName = "QtLargeModelViewer.chm";
#endif
      }
      if (m_currentViewer != NULL) {
        m_flyViewer->getGLRenderAction()->setAlphaTest( m_alphaTestTransparencyMenuItem->isChecked() );
        m_flyViewer->setTransparencyType (m_currentViewer->getTransparencyType ());
        myCurrentViewType = m_currentViewer->getStereoViewType ();
        m_flyViewer->setStereoActive (m_currentViewer->isStereoActive());
      }
      m_currentViewer = m_flyViewer;
      m_flyViewerMenuItem->setChecked( true );
      break;

    case PLANE:
      if (m_planeViewer == NULL) {
        m_planeViewer = new SoQtPlaneViewer (this);
        m_planeViewer->helpFileName = QLMV_HELP_FILE_NAME;
      }
      if (m_currentViewer != NULL) {
        m_planeViewer->getGLRenderAction()->setAlphaTest( m_alphaTestTransparencyMenuItem->isChecked() );
        m_planeViewer->setTransparencyType (m_currentViewer->getTransparencyType ());
        myCurrentViewType = m_currentViewer->getStereoViewType ();
        m_planeViewer->setStereoActive (m_currentViewer->isStereoActive());
      }
      m_currentViewer = m_planeViewer;
      m_planeViewerMenuItem->setChecked( true );
      break;
  }

  //SpaceBall
  if ((m_spaceball_available) &&(m_currentViewer != NULL)) {
    m_currentViewer->registerDevice(m_sb);
    m_spaceball_rotScaleFactor   = m_sb->getRotationScaleFactor();
    m_spaceball_transScaleFactor = m_sb->getTranslationScaleFactor();
    
    printf("Default rotation scale factor %f\n",m_spaceball_rotScaleFactor);
    printf("Default translation scale factor %f\n",m_spaceball_transScaleFactor);
  }

  //set the stereo configuration to be the same as the previous viewer
  SoBaseStereo *myStereo = NULL;
  // set the stereoViewType
  if (myCurrentViewType != NULL) {
    switch (myCurrentViewType->getStereoViewType()) {
    case 0:
      //NO_STEREO_VIEW
      break;
    case 1:
      //INTERLACED_STEREO
      myStereo = new SoInterlacedStereo((SoStereoViewer *)m_currentViewer);
      ((SoInterlacedStereo *)myStereo)->setInterlacedMode(((SoInterlacedStereo *)myCurrentViewType)->getInterlacedMode());
      ((SoInterlacedStereo *)myStereo)->setFastInterlacing(((SoInterlacedStereo *)myCurrentViewType)->isFastInterlacing());
      ((SoInterlacedStereo *)myStereo)->setScreenInterlacing(((SoInterlacedStereo *)myCurrentViewType)->isScreenInterlacing());
      ((SoInterlacedStereo *)myStereo)->reverseStereoView(((SoInterlacedStereo *)myCurrentViewType)->isStereoViewReversed());
      break;
    case 2:
      //HALF_SCREEN_STEREO
      myStereo = new SoHalfScreenStereo((SoStereoViewer *)m_currentViewer);
      ((SoHalfScreenStereo *)myStereo)->setHalfScreenMode(((SoHalfScreenStereo *)myCurrentViewType)->getHalfScreenMode());
      ((SoHalfScreenStereo *)myStereo)->reverseStereoView(((SoHalfScreenStereo *)myCurrentViewType)->isStereoViewReversed());
      break;
    case 3:
      //RAW_STEREO or OPENGL_STEREO
      myStereo = new SoRawStereo((SoStereoViewer *)m_currentViewer);
      ((SoAnaglyphStereo *)myStereo)->reverseStereoView(((SoAnaglyphStereo *)myCurrentViewType)->isStereoViewReversed());
      break;
    case 4:
      //ANAGLYPH_STEREO
      myStereo = new SoAnaglyphStereo((SoStereoViewer *)m_currentViewer);
      ((SoAnaglyphStereo *)myStereo)->reverseStereoView(((SoAnaglyphStereo *)myCurrentViewType)->isStereoViewReversed());
      ((SoAnaglyphStereo *)myStereo)->setColorFilter(((SoAnaglyphStereo *)myCurrentViewType)->getColorFilter());
      break;
    default:
      break;
    }
    if (myStereo !=NULL)
      m_currentViewer->setStereoViewType(myStereo);
    //set the offset, the parallax and active or not the stereo
    myStereo->getStereoViewer()->setStereoBalance(myCurrentViewType->getStereoViewer()->getStereoBalance());
    myStereo->getStereoViewer()->setStereoOffset(myCurrentViewType->getStereoViewer()->getStereoOffset());
    myStereo->getStereoViewer()->setStereoActive(myCurrentViewType->getStereoViewer()->isStereoActive());

  }

  m_currentViewer->getWidget()->show();
  setCentralWidget(m_currentViewer->getWidget());
  QApplication::processEvents();

  if (m_adaptiveViewing != NULL) m_adaptiveViewing->setViewer (m_currentViewer);

  // Set the scene graph and other parameters
  m_currentViewer->setSceneGraph (m_displayedSG);
  if (m_cameraParent->getNumChildren()) {
    if (m_cameraParent->getChild (0)->isOfType (SoCamera::getClassTypeId())) {
      m_currentViewer->setCamera ((SoCamera*)(m_cameraParent->getChild(0)));
    }
  }
  if (shouldDisplayRenderStats) {
    displayRenderStats (TRUE);
  }
}

////////////////////////////////////////////////////////////////////////
// Accept drag
void QLmv::dragEnterEvent(QDragEnterEvent* event)
{
    event->accept();
}

////////////////////////////////////////////////////////////////////////
//
void QLmv::dropEvent(QDropEvent* event)
{
  // get the path and file name 
  if ( event->mimeData()->hasUrls() ) 
  {
    QList<QUrl> urls = event->mimeData()->urls();
    openFile(urls[0].toLocalFile());
  }
}

////////////////////////////////////////////////////////////////////////
// Open a file
void QLmv::openFile (const QString& filename)
{
  SbString fname;

  fname.fromUtf16( filename.utf16() );

  SbString dirName = SbFileHelper::getDirName( fname );

  // Set up directory to help SoInput find the rest of the included files:
  if (!dirName.isEmpty())
    SoInput::addDirectoryFirst(dirName);

  QProgressDialog progress( QString("Loading ") + filename, "Cancel", 0, 100, this);
  progress.setWindowModality(Qt::WindowModal);
  progress.setCancelButton( NULL );
  progress.setValue( 0 );


  // Load the file
  int i;
  LiveInput inp(&progress);
  if ( !inp.openFile(fname, FALSE, TRUE) ) 
  {
    QString msg = QString (tr("Unable to open file %1.")).arg(filename);
    const QString caption = tr("Error");
    QMessageBox *messagePopup = new  QMessageBox (caption, msg, QMessageBox::Critical,
                 QMessageBox::Abort, Qt::NoButton, Qt::NoButton,
                 this);
    messagePopup->show();
    removeRecentFiles(filename);
    return;
  }
  SoGroup* newSG = SoDB::readAll (&inp);
  if (newSG == NULL) {
    QString msg = QString (tr("Unable to read the file %1.\nIt may be corrupted.")).arg(filename);
    const QString caption = tr("Error");
    QMessageBox *messagePopup = new  QMessageBox (caption, msg, QMessageBox::Critical,
                 QMessageBox::Abort, Qt::NoButton, Qt::NoButton,
                 this);
    messagePopup->show();
    removeRecentFiles(filename);
    return;
  }

  //add the file in the recent file list
  updateRecentFiles( filename );

  // Don't follow any track on the old scene
  followTrack (-1);

  // Store the filename and enable reload
  m_filename = filename;
  m_fileReloadMenuItem->setEnabled( true );
  m_fileReopenBtn->setEnabled( true );

  // Add the file to the displayed scene graph
  m_loadedSG = newSG;
  m_selectionNode->removeAllChildren();
  m_selectionNode->addChild (m_loadedSG);

  //First reset scenegraph
  m_currentViewer->setSceneGraph ( NULL );
  //Set correct scene graph
  m_currentViewer->setSceneGraph( m_displayedSG );

  inlineLoader->ClearLists();

  // Search for viewpoints in the file
  for ( i=0; i<m_vpList.getLength(); i++ )
    m_viewpointsMenu->removeAction( m_vpMenuIdsMap[i+1] );
  if ( m_vpList.getLength() > 0 )
    m_viewpointsMenu->removeAction( m_vpMenuIdsMap[1] );
  while (m_vpList.getLength() > 0)
    m_viewpointsMenu->removeAction( m_vpMenuIdsMap[0] );

  m_viewpointsMenuId->setEnabled( false );

  // Search for the animations in the scene graph
  for (i=0; i<m_trackList.getLength(); i++) {
    m_animationsMenu->removeAction( m_animMenuIdsMap[m_animationsMenu->actions().count()-1] );
  }
  while (m_trackList.getLength() > 0) {
    m_trackList.remove(0);
  }
  SoSearchAction searchAnims;
  searchAnims.setType (SoGroup::getClassTypeId(), TRUE);
  searchAnims.setName (SbName ("LMV_ANIM"));
  searchAnims.setInterest (SoSearchAction::ALL);
  searchAnims.setSearchingAll (TRUE);
  searchAnims.apply (m_loadedSG);
  SoPathList trackPathList = searchAnims.getPaths();
  int numAnimationInserted = 0;
  for (i=0; i<trackPathList.getLength(); i++) {
    SoFullPath* path = (SoFullPath*) trackPathList[i];
    QString trackName;
    if (((SoGroup*)(path->getTail()))->getChild(0)->getName() == SbName("")) {
      char buffer[256];
      sprintf (buffer, "Track %d", numAnimationInserted + 1);
      trackName = QString (buffer);
    }
    else {
      trackName = QString (((SoGroup*)(path->getTail()))->getChild(0)->getName().getString());
      trackName.replace ('_', ' ');
      trackName = trackName.simplified();
    }
    m_trackList.append (path->getTail());
    QAction* action = new QAction( trackName, m_animationsMenu );
    m_animationsMenu->insertAction( m_animMenuIdsMap[m_trackList.getLength()+ MNU_ANIM_FIRST_TRACK - 1], action );
    numAnimationInserted ++;
  }
  if ( numAnimationInserted )
    m_animationsMenuId->setEnabled( true );
  else
    m_animationsMenuId->setEnabled( false );

  // Search for a camera in the loaded scene graph
  SoSearchAction searchCams;
  searchCams.setType (SoCamera::getClassTypeId());
  searchCams.setInterest (SoSearchAction::FIRST);
  searchCams.apply (m_loadedSG);
  SoFullPath* cameraPath = (SoFullPath*) searchCams.getPath();

  // Put the camera in the right place in the scene graph
  if (cameraPath != NULL) {
    m_cameraParent->removeAllChildren ();
    m_cameraParent->addChild (cameraPath->getTail ());
    SoGroup* oldParent = (SoGroup*)cameraPath->getNodeFromTail (1);
    oldParent->replaceChild (cameraPath->getTail (), m_cameraParent);
  }
  else {
    // Clean the camera parent tree and put the same kind of camera as before
    if (m_cameraParent->getNumChildren()) {
      if (m_cameraParent->getChild (0)->isOfType (SoOrthographicCamera::getClassTypeId())) {
        m_cameraParent->removeAllChildren ();
        m_cameraParent->addChild (new SoOrthographicCamera);
      }
      else {
        m_cameraParent->removeAllChildren ();
        m_cameraParent->addChild (new SoPerspectiveCamera);
      }
    }
    else {
      m_cameraParent->addChild (new SoPerspectiveCamera);
    }
  }
  //SpaceBall
  if ((m_spaceball_available) &&(m_currentViewer != NULL)) {
    m_cameraParent->addChild (m_spaceballTranslation);
    m_cameraParent->addChild (m_spaceballRotation);
  }

  // Auto follow a track
  int trackToFollow = SoPreferences::getInt ("LMV_TRACK_TO_FOLLOW", -1);
  if (trackToFollow >= 0) {
    if (trackToFollow < m_trackList.getLength()) {
      followTrack (trackToFollow);
    }
  }
  SbString trackView = SoPreferences::getString ("LMV_TRACK_VIEW", "INSIDE");
  if (trackView == "FROM_BEHIND") {
    followAsFromBehindView ();
  }
  else if (trackView == "TOP") {
    followAsTopView ();
  }
  else if (trackView == "FREE") {
    followAsFreeView ();
  }
  else {
    followAsInsideView ();
  }

  // Process scene graph to handle Inlines:
  inlineLoader->FixWWWInlines(m_loadedSG);
    
  // Force in all of the empty WWWInlines:
  inlineLoader->FetchAllEmpty();

  // Restore the processing menu
  m_processingMenuId->setEnabled( true );
  if ( !m_reorganizeMenuItem->isCheckable() )
    m_reorganizeMenuItem->setCheckable( true );
  m_reorganizeMenuItem->setChecked( false );
  m_reorganizeMenuItem->setEnabled( true );
  if ( !m_shapeSimplifyMenuItem->isCheckable() )
    m_shapeSimplifyMenuItem->setCheckable( true );
  m_shapeSimplifyMenuItem->setChecked( false );
  m_shapeSimplifyMenuItem->setEnabled( true );
  if ( !m_globalSimplifyMenuItem->isCheckable() )
    m_globalSimplifyMenuItem->setCheckable( true );
  m_globalSimplifyMenuItem->setChecked( false );
  m_globalSimplifyMenuItem->setEnabled( true );
  if ( !m_shapeSimplifyTSMenuItem->isCheckable() )
    m_shapeSimplifyTSMenuItem->setCheckable( true );
  m_shapeSimplifyTSMenuItem->setChecked( false );
  m_shapeSimplifyTSMenuItem->setEnabled( true );
  if ( !m_globalSimplifyTSMenuItem->isCheckable() )
    m_globalSimplifyTSMenuItem->setCheckable( true );
  m_globalSimplifyTSMenuItem->setChecked( false );
  m_globalSimplifyTSMenuItem->setEnabled( true );
  if ( !m_splitShapeMenuItem->isCheckable() )
    m_splitShapeMenuItem->setCheckable( true );
  m_splitShapeMenuItem->setChecked( false );
  m_splitShapeMenuItem->setEnabled( true );

  // Set the traversal mode to separator
  m_separatorTraversalNode->addChild (m_selectionNode);
  m_displayedSG->addChild (m_separatorTraversalNode);
}

////////////////////////////////////////////////////////////////////////
// Slot to open a recent file
void QLmv::fileOpenRecent( QAction* index )
{
  QString file = index->text();
  if ( file == "&Open..." )
    return;
  file.remove( 0, 3 );
  openFile( file );
}

////////////////////////////////////////////////////////////////////////
// Update the registry
void QLmv::updateRecentFiles( const QString& filename )
{
  if ( m_recentFiles.size() > 0 )
  {
    if ( filename != m_recentFiles[m_recentFiles.size()-1] )
      m_recentFiles.removeAt( m_recentFiles.indexOf(filename) );
  }

  m_recentFiles.push_front( filename );
  if ( m_recentFiles.count() > MAX_RECENTFILES )
    m_recentFiles.pop_back();

  if ( !m_fileMenuConnected ) 
  {
    connect( m_fileMenu, SIGNAL(aboutToShow()), this, SLOT(updateRecentFilesMenu()) );
    m_fileMenuConnected = true;
  }
}

////////////////////////////////////////////////////////////////////////
// Update the registry
void QLmv::removeRecentFiles( const QString& filename )
{
  if ( filename == m_recentFiles[m_recentFiles.size()-1] )
    return;

  m_recentFiles.removeAt( m_recentFiles.indexOf(filename) );

  if ( !m_fileMenuConnected ) 
  {
    connect( m_recentFilesMenu, SIGNAL(aboutToShow()), this, SLOT(updateRecentFilesMenu()) );
    m_fileMenuConnected = true;
  }
}

////////////////////////////////////////////////////////////////////////
// Add recent files to file menu
void QLmv::updateRecentFilesMenu()
{
  int count = m_recentFiles.count();

  m_recentFilesMenu->clear();

  if (count > MAX_RECENTFILES)
    count = MAX_RECENTFILES;

  //  update the menu
    for (int i = 0; i < count; i++)
     m_recentFilesMenu->addAction( QString("&%1 %2").arg(i+1).arg(m_recentFiles[i]) );

  connect( m_recentFilesMenu, SIGNAL(triggered(QAction*)), this, SLOT(fileOpenRecent(QAction*)) );
 
  if ( m_fileMenuConnected )
  {
    disconnect( m_fileMenu,SIGNAL(aboutToShow()), this, SLOT(updateRecentFilesMenu()) );
    m_fileMenuConnected = false;
  }
}

////////////////////////////////////////////////////////////////////////
// Send all Close Event to the FileQuit function
void QLmv::closeEvent( QCloseEvent * )
{
  fileQuit();
}

////////////////////////////////////////////////////////////////////////
// Save recent files in the registry before exiting
void QLmv::fileQuit()
{
	saveOptions();
	qApp->exit( 0 );
}

void QLmv::saveOptions()
{
  QSettings settings( QSettings::SystemScope, WINDOWS_REGISTRY );
  for ( int i = 0; i < int(m_recentFiles.count()); ++i )
    settings.setValue( APP_KEY + "File" + QString::number( i + 1 ),
    m_recentFiles[i] );
}

////////////////////////////////////////////////////////////////////////
// Set or remove the collision detection
void QLmv::setCollisionDetection( bool onOrOff )
{
  m_collisionViewer->setCollisionDetection(onOrOff);
  if ( !m_collisionDetectionMenuItem->isCheckable() )
    m_collisionDetectionMenuItem->setCheckable( true );
  m_collisionDetectionMenuItem->setChecked(onOrOff);
}


////////////////////////////////////////////////////////////////////////
void QLmv::setAlphaTest (bool onOrOff)
{
  m_currentViewer->getGLRenderAction()->setAlphaTest (onOrOff);
  if ( !m_alphaTestTransparencyMenuItem->isCheckable() )
    m_alphaTestTransparencyMenuItem->setCheckable( true );
  m_alphaTestTransparencyMenuItem->setChecked(onOrOff);
}

////////////////////////////////////////////////////////////////////////
void QLmv::setFsaa( bool onOff )
{
  m_currentViewer->setAntialiasing( (onOff) ? 1.0f : 0.0f, SoSceneManager::FSAA );
}


////////////////////////////////////////////////////////////////////////
void QLmv::setTransparency (SoGLRenderAction::TransparencyType type)
{
  if ( !m_noSortTransparencyMenuItem->isCheckable() )
    m_noSortTransparencyMenuItem->setCheckable( true );
  m_noSortTransparencyMenuItem->setChecked( false );
  if ( !m_opaqueFirstTransparencyMenuItem->isCheckable() )
    m_opaqueFirstTransparencyMenuItem->setCheckable( true );
  m_opaqueFirstTransparencyMenuItem->setChecked( false );
  if ( !m_sortedObjectTransparencyMenuItem->isCheckable() )
    m_sortedObjectTransparencyMenuItem->setCheckable( true );
  m_sortedObjectTransparencyMenuItem->setChecked( false );
  if ( !m_sortedPixelTransparencyMenuItem->isCheckable() )
    m_sortedPixelTransparencyMenuItem->setCheckable( true );
  m_sortedPixelTransparencyMenuItem->setChecked( false );

  switch (type) {
    case SoGLRenderAction::OPAQUE_FIRST:
      m_currentViewer->setTransparencyType (SoGLRenderAction::OPAQUE_FIRST);
      m_opaqueFirstTransparencyMenuItem->setChecked( true );
      break;
    case SoGLRenderAction::SORTED_OBJECT:
      m_currentViewer->setTransparencyType(SoGLRenderAction::SORTED_OBJECT);
      m_sortedObjectTransparencyMenuItem->setChecked(true);
      break;
    case SoGLRenderAction::SORTED_PIXEL:
      m_currentViewer->setTransparencyType(SoGLRenderAction::SORTED_PIXEL);
      m_sortedPixelTransparencyMenuItem->setChecked(true);
      break;
    case SoGLRenderAction::NO_SORT:
    default:
      m_currentViewer->setTransparencyType (SoGLRenderAction::NO_SORT);
      m_noSortTransparencyMenuItem->setChecked( true );
      break;

  }
}


////////////////////////////////////////////////////////////////////////
// Set or remove stereo features
void QLmv::setStereo(bool onOff, bool button)
{
  if( button )
    onOff = !onOff;
  if ( onOff ) 
  {
    // If no stereo type set yet, set it to a safe type, ie. Red/Cyan
    SoQtStereoViewType *viewType = m_currentViewer->getStereoViewType();
    if (viewType == NULL) {
      viewType = new SoQtAnaglyphStereo();
      ((SoQtAnaglyphStereo *)viewType)->setColorFilter( SoQtAnaglyphStereo::RED_CYAN );
      m_currentViewer->setStereoViewType( viewType );
    }
    m_currentViewer->setStereoActive( TRUE );
  }
  else
    m_currentViewer->setStereoActive( FALSE );
  if ( !m_stereoMenuItem->isCheckable() )
    m_stereoMenuItem->setCheckable( true );
  m_stereoMenuItem->setChecked( onOff );
  m_stereoBtn->setDown( onOff );

	// Update the props dialog if it exists
	SoQtStereoDialog *stereoDialog = m_currentViewer->getStereoDialog();
  if ( stereoDialog )
    stereoDialog->update();
}

////////////////////////////////////////////////////////////////////////
// Hide or displays the stereo dialog
void QLmv::onMnuStereoDialog() 
{
  SoQtStereoDialog *stereoDialog = m_currentViewer->getStereoDialog();
  if (stereoDialog == NULL) {
    stereoDialog = new SoQtStereoDialog( NULL, (SoQtStereoViewer *)m_currentViewer ) ;
    m_currentViewer->setStereoDialog (stereoDialog);
    stereoDialog->getWidget()->move(posStereoDialog);
  }
  stereoDialog->show();
}


////////////////////////////////////////////////////////////////////////
// Display statistics
void QLmv::displayRenderStats(bool onOrOff)
{
  if (onOrOff) {
    if ( !m_viewRenderStatsMenuItem->isCheckable() )
      m_viewRenderStatsMenuItem->setCheckable( true );
    m_viewRenderStatsMenuItem->setChecked( true );
    m_fpsDisplay->setDisabled (FALSE);
    m_currentViewer->setFramesPerSecondCallback (onFpsChangeCB, this);
    m_currentViewer->setNumSamples (10);
    m_numTrianglesDisplay->setDisabled (FALSE);
    m_currentViewer->setDecimationPercentageCallback (onNewPercentageStatisticsCB, this);
    m_decimationDisplay->setDisabled (FALSE);
  }
  else {
    if ( !m_viewRenderStatsMenuItem->isCheckable() )
      m_viewRenderStatsMenuItem->setCheckable( true );
    m_viewRenderStatsMenuItem->setChecked( false );
    m_fpsDisplay->setDisabled (TRUE);
    m_currentViewer->setFramesPerSecondCallback (NULL, this);
    m_currentViewer->setNumSamples (-1);
    m_numTrianglesDisplay->setDisabled (TRUE);
    m_currentViewer->setDecimationPercentageCallback (NULL);
    m_decimationDisplay->setDisabled (TRUE);
  }
  m_fpsDisplay->setText (tr("fps"));
  m_numTrianglesDisplay->setText ("");
  m_decimationDisplay->setText ("%");
  onSetStatusMessage ("");
}

void 
QLmv::readingInlineCB(const SbString& filename, void * userData, InlineLoader *)
{
  QString infoString("Reading inline: ");
  infoString += QString::fromUtf16( filename.toUtf16() );
  ((QLmv*)userData)->onSetStatusMessage(infoString);
}

void QLmv::onMnuLoadInlines()
{
  inlineLoader->setLoadAllImmediate(!inlineLoader->isLoadAllImmediate());
  if ( !m_loadInlinesItem->isCheckable() )
    m_loadInlinesItem->setCheckable( true );
  m_loadInlinesItem->setChecked( m_loadInlinesItem->isChecked() );
}

////////////////////////////////////////////////////////////////////////
//	Reorganizes each child of the LevelOfSimplification
//  nodes under "graph".  This will cause each representation
//  to be in triangle strips instead of face sets.
//      It isn't always possible to reorganize a subgraph, because
//  it may lose the coordinates, materials, or shape hints that are
//  higher in the scene graph; but since we using this
//  on the results of the SimplifyActions, we know that the shapes
//  are using vertex property, and therefore contain all of the 
//  coordinate and normal info.
void QLmv::reorganizeSimplified(SoNode *graph)
{
  graph->ref();
  
  SoSearchAction search;
  search.setType(SoLevelOfSimplification::getClassTypeId());
  search.setInterest(SoSearchAction::ALL);
  search.apply(graph);
  
  SoNodeList alreadyDone;
  
  SoPathList paths = search.getPaths();
  for (int i=0; i<paths.getLength(); i++) {
    SoFullPath *fullpath = (SoFullPath*)paths[i];
    SoLevelOfSimplification *los = 
      (SoLevelOfSimplification*)fullpath->getTail();
    for (int c=0; c<los->getNumChildren(); c++) {
      SoNode *node = los->getChild(c);
      
      if (alreadyDone.find(node) >=0) {
        // We have already tri-stripped this node,
        // so skip it now:
        continue;
      }
      alreadyDone.append(node);
      
      SoReorganizeAction tristripper;
      tristripper.generateNormals(TRUE);
      tristripper.generateTriangleStrips(TRUE);
      tristripper.apply(node);
      if (tristripper.getSimplifiedSceneGraph() != NULL) {
        SoNode *result = tristripper.getSimplifiedSceneGraph();
        if (result != node)
          los->replaceChild(node,result);
      }
    }
  }
  
  graph->unrefNoDelete();
}


////////////////////////////////////////////////////////////////////////
// Reorganize the scene
void QLmv::reorganize()
{
  if (m_loadedSG == NULL) return;

  m_statusbar->showMessage( tr("Reorganizing scene...") );

  // Organize scene under new separator:
  SoSeparator *sep = new SoSeparator;
  sep->ref();
  for (int i=0; i<m_selectionNode->getNumChildren(); i++) {
    sep->addChild( m_selectionNode->getChild(i) );
  }
  
  // Do the action
  SoReorganizeAction* simplify;
  simplify = new SoReorganizeAction; ;
  simplify->generateNormals(TRUE);
  simplify->generateTriangleStrips(TRUE);
  simplify->apply(sep);
  sep->unref();

  // Use the simplified scene graph
  if (simplify->getSimplifiedSceneGraph() != NULL) {
    m_loadedSG = simplify->getSimplifiedSceneGraph();
    m_selectionNode->removeAllChildren ();
    m_selectionNode->addChild (m_loadedSG);
    if ( !m_reorganizeMenuItem->isCheckable() )
      m_reorganizeMenuItem->setCheckable( true );
    m_reorganizeMenuItem->setChecked( true );
    m_reorganizeMenuItem->setEnabled( false );
    if ( !m_shapeSimplifyMenuItem->isCheckable() )
      m_shapeSimplifyMenuItem->setCheckable( true );
    m_shapeSimplifyMenuItem->setChecked( false );
    m_shapeSimplifyMenuItem->setEnabled( false );
    if ( !m_globalSimplifyMenuItem->isCheckable() )
      m_globalSimplifyMenuItem->setCheckable( true );
    m_globalSimplifyMenuItem->setChecked( false );
    m_globalSimplifyMenuItem->setEnabled( false );
    m_statusbar->showMessage(tr("Scene reorganized"), 3000);
  }
  else {
    m_statusbar->showMessage(tr("Problems in scene reorganization"), 10000);
  }
  delete simplify;
}


////////////////////////////////////////////////////////////////////////
// Simplify the shapes in the scene
void QLmv::shapeSimplify (bool doTriangleStrips)
{
  // Exit if nothing to do or allready done !
  if (m_loadedSG == NULL) return;

  // Starting the job
  m_statusbar->showMessage(tr("Shape simplifying scene..."));
  
  int numLevels = 3;
  static float levels[5] = {1.0f, 0.3f, 0.1f};

  // The Simplifier object: uses geometric simplification
  SoDecimator *decimer = new SoDecimator;
  SoShapeSimplifyAction simplify(decimer);
  simplify.setSimplificationLevels(numLevels,levels);
  simplify.setSizeFactor(4.);
  simplify.setMinTriangles(20);
  simplify.apply(m_loadedSG);
  delete decimer;
  
  // Do triangle stripping on results:
  if (doTriangleStrips) {
	  m_statusbar->showMessage(tr("Triangle stripping simplified result..."));  
    reorganizeSimplified(m_loadedSG);
	  m_statusbar->showMessage(tr("Scene shape simplified and triangle stripped"), 3000);  
    if ( !m_shapeSimplifyTSMenuItem->isCheckable() )
      m_shapeSimplifyTSMenuItem->setCheckable( true );
    m_shapeSimplifyTSMenuItem->setChecked( true );
  }
  else {
 	  m_statusbar->showMessage(tr("Scene shape simplified"), 3000);
    if ( !m_shapeSimplifyTSMenuItem->isCheckable() )
      m_shapeSimplifyMenuItem->setCheckable( true );
    m_shapeSimplifyMenuItem->setChecked( true );
  }
  m_reorganizeMenuItem->setEnabled( false );
  m_shapeSimplifyMenuItem->setEnabled( false );
  m_shapeSimplifyTSMenuItem->setEnabled( false );
  m_globalSimplifyMenuItem->setEnabled( false );
  m_globalSimplifyTSMenuItem->setEnabled( false );
}


////////////////////////////////////////////////////////////////////////
// Simplify the scene
void QLmv::globalSimplify (bool doTriangleStrips)
{
  // Exit if nothing to do or allready done !
  if (m_loadedSG == NULL) return;

  // Starting the job
  m_statusbar->showMessage(tr("Global simplifying scene..."));
 
  // Replace scene with reorganized scene graph:
  int numLevels = 3;
  static float levels[5] = {1.0f, 0.3f, 0.1f};
  
  SoDecimator *decimer = new SoDecimator;
  SoGlobalSimplifyAction *simplify = new SoGlobalSimplifyAction(decimer);
  simplify->setSimplificationLevels(numLevels,levels);
  simplify->setSizeFactor(4.);
  simplify->setMinTriangles(20);
  simplify->setVerbosity(TRUE);
  simplify->setSimplificationStrategy(SoGlobalSimplifyAction::SIMPLIFY_GLOBALLY);
  
  // Organize scene under new separator:
  SoSeparator *sep = new SoSeparator;
  sep->ref();
  for (int i=0; i<m_selectionNode->getNumChildren(); i++) {
    sep->addChild (m_selectionNode->getChild(i));
  }
  simplify->apply(sep);
  sep->unref();
  
  if (simplify->getSimplifiedSceneGraph() != NULL) {
    m_loadedSG = simplify->getSimplifiedSceneGraph();
    m_selectionNode->removeAllChildren ();
    m_selectionNode->addChild (m_loadedSG);
    
    if (doTriangleStrips) {
	    m_statusbar->showMessage(tr("Triangle stripping simplified result..."));  
      reorganizeSimplified(m_loadedSG);
	    m_statusbar->showMessage(tr("Scene simplified and triangle stripped"), 3000);  
      if ( !m_globalSimplifyTSMenuItem->isCheckable() )
        m_globalSimplifyTSMenuItem->setCheckable( true );
      m_globalSimplifyTSMenuItem->setChecked( true );
    }
    else {
 	    m_statusbar->showMessage(tr("Scene simplified"), 3000);  
      if ( !m_globalSimplifyTSMenuItem->isCheckable() )
        m_globalSimplifyMenuItem->setCheckable( true );
      m_globalSimplifyMenuItem->setChecked( true );
    }
    m_reorganizeMenuItem->setEnabled( false );
    m_shapeSimplifyMenuItem->setEnabled( false );
    m_shapeSimplifyTSMenuItem->setEnabled( false );
    m_globalSimplifyMenuItem->setEnabled( false );
    m_globalSimplifyTSMenuItem->setEnabled( false );
  }
  else {
    m_statusbar->showMessage(tr("Problems in global simplification"), 10000);
  }
  
  delete simplify;
  delete decimer;
}


////////////////////////////////////////////////////////////////////////
void QLmv::splitShapes()
{
  if ( !m_splitShapeMenuItem->isCheckable() )
    m_splitShapeMenuItem->setCheckable( true );
  m_splitShapeMenuItem->setChecked( false );
  QLmvSplitAction* dlg = new QLmvSplitAction (this, m_loadedSG);
  dlg->exec ();
}


////////////////////////////////////////////////////////////////////////
// Follow one of the tracks of the scene
void QLmv::followTrack (int trackId)
{
  if ( trackId == -1 ) 
  {
    m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setEnabled( false );
    if (m_currentTrack >= 0) 
    {
      SoGroup* grp = (SoGroup*)m_trackList[m_currentTrack];

      // remove the switch from the scene graph
      for (int i=0; i<m_trackSwitch->getNumChildren(); i++)
        grp->addChild (m_trackSwitch->getChild(i));

      grp->removeChild (m_trackSwitch);
      m_trackSwitch->removeAllChildren();

      // Disconnect the camera
      m_currentViewer->getCamera()->position.disconnect();
      m_currentViewer->getCamera()->orientation.disconnect();
    }
  }
  else 
  {
    // Clean the scene graph
    followTrack (-1);
    // enable the menu items that can now be selected
    m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setEnabled( true );
    m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setEnabled( true );
    m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setEnabled( true );
    m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setEnabled( true );
    m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setEnabled( true );
    m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setEnabled( true );

    SoGroup* grp = (SoGroup*)m_trackList[trackId];

    // add the switch to display ot not the object
    for (int i=1; i<grp->getNumChildren(); i++)
      m_trackSwitch->addChild(grp->getChild (i));
    while (grp->getNumChildren() > 1)
      grp->removeChild (1);
    grp->addChild (m_trackSwitch);

    // Attach the engines together and the camera to the engine
    SoTransform* tr = (SoTransform*) grp->getChild(0);
    m_moveAroundEngine->position.connectFrom (&(tr->translation));
    m_moveAroundEngine->orientation.connectFrom (&(tr->rotation));
    m_currentViewer->getCamera()->position.connectFrom (&(m_moveAroundEngine->outPosition));
    m_currentViewer->getCamera()->orientation.connectFrom (&(m_moveAroundEngine->outOrientation));

    // Tune the distance for TOP and FROM BEHINF views
    if ( m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->isChecked() ) 
    {
      if (m_trackSwitch->getNumChildren () > 0) 
      {
        SoGetBoundingBoxAction action (SbViewportRegion (100, 100));
        m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
        action.apply (m_trackSwitch);
        float dim_obj = local_MAX (action.getBoundingBox().getMax()[0] - action.getBoundingBox().getMin()[0],
                            local_MAX (action.getBoundingBox().getMax()[1] - action.getBoundingBox().getMin()[1],
                                  action.getBoundingBox().getMax()[2] - action.getBoundingBox().getMin()[2]));
        m_moveAroundEngine->distance.setValue (dim_obj * 4);
        displayFollowedObject( m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
      }
      else
        m_moveAroundEngine->distance.setValue (4);
    }
    else if ( m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->isChecked() )
    {
      if (m_trackSwitch->getNumChildren () > 0) {
        SoGetBoundingBoxAction action (SbViewportRegion (100, 100));
        m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
        action.apply (m_trackSwitch);
        float dim_obj = local_MAX (action.getBoundingBox().getMax()[0] - action.getBoundingBox().getMin()[0],
                            local_MAX (action.getBoundingBox().getMax()[1] - action.getBoundingBox().getMin()[1],
                                  action.getBoundingBox().getMax()[2] - action.getBoundingBox().getMin()[2]));
        m_moveAroundEngine->distance.setValue (dim_obj * 10);
        displayFollowedObject( m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
      }
      else
        m_moveAroundEngine->distance.setValue (6);
    }
  }

  m_animMenuIdsMap[MNU_ANIM_FIRST_TRACK + m_currentTrack]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_FIRST_TRACK + trackId]->setChecked( true );
  m_currentTrack = trackId;
}


////////////////////////////////////////////////////////////////////////
// Display or hide the followed object
void QLmv::displayFollowedObject (bool onOff)
{
  if (onOff) {
    m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
    m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setChecked( true );
  }
  else {
    m_trackSwitch->whichChild.setValue (SO_SWITCH_NONE);
    m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setChecked( false );
  }
}


////////////////////////////////////////////////////////////////////////
// Roll with the object or stay flat
void QLmv::rollWithObject (bool onOff)
{
  if (onOff) {
    m_moveAroundEngine->flatOrientation.setValue (FALSE);
    m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setChecked( true );
  }
  else {
    m_moveAroundEngine->flatOrientation.setValue (TRUE);
    m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setChecked( false );
  }
}


////////////////////////////////////////////////////////////////////////
// Point the camera in direction of the object or keep the engine
// orientation
void QLmv::pointAtObject (bool onOff)
{
  if (onOff) {
    m_moveAroundEngine->keepOrientation.setValue (FALSE);
    m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setChecked( true );
  }
  else {
    m_moveAroundEngine->keepOrientation.setValue (TRUE);
    m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setChecked( false );
  }
}


////////////////////////////////////////////////////////////////////////
// 
void QLmv::followAsInsideView ()
{
  if ( m_animMenuIdsMap[MNU_ANIM_NONE]->isChecked() ) 
  { 
    m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setEnabled( false );
  }
  else  
  { 
    m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->setEnabled( false );
    m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->setEnabled( false );
  }
    m_moveAroundEngine->azimuth.setValue (0);
    m_moveAroundEngine->site.setValue (0);
    m_moveAroundEngine->distance.setValue (0);
    m_moveAroundEngine->keepOrientation.setValue (FALSE);
    displayFollowedObject (FALSE);
    m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setEnabled( false );
     m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setChecked( true );
    m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setChecked( false );
    m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setChecked( false );
    m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setChecked( false );
}

////////////////////////////////////////////////////////////////////////
// 
void QLmv::followAsFromBehindView ()
{
  m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setEnabled( true );
  m_moveAroundEngine->azimuth.setValue (0.3f);
  m_moveAroundEngine->site.setValue (0.f);
  if (m_trackSwitch->getNumChildren () > 0) {
    SoGetBoundingBoxAction action (SbViewportRegion (100, 100));
    m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
    action.apply (m_trackSwitch);
    float dim_obj = local_MAX (action.getBoundingBox().getMax()[0] - action.getBoundingBox().getMin()[0],
                        local_MAX (action.getBoundingBox().getMax()[1] - action.getBoundingBox().getMin()[1],
                              action.getBoundingBox().getMax()[2] - action.getBoundingBox().getMin()[2]));
    m_moveAroundEngine->distance.setValue (dim_obj * 4);
    displayFollowedObject( m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
  }
  else {
    m_moveAroundEngine->distance.setValue (4);
  }

  displayFollowedObject (TRUE);
  pointAtObject (TRUE);
  m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setChecked( true );
  m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setChecked( false );
}


////////////////////////////////////////////////////////////////////////
// 
void QLmv::followAsTopView ()
{
  m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setEnabled( true );
  m_moveAroundEngine->azimuth.setValue (float(M_PI_2));
  m_moveAroundEngine->site.setValue (0.f);
  if (m_trackSwitch->getNumChildren () > 0) {
    SoGetBoundingBoxAction action (SbViewportRegion (100, 100));
    m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
    action.apply (m_trackSwitch);
    float dim_obj = local_MAX (action.getBoundingBox().getMax()[0] - action.getBoundingBox().getMin()[0],
                        local_MAX (action.getBoundingBox().getMax()[1] - action.getBoundingBox().getMin()[1],
                              action.getBoundingBox().getMax()[2] - action.getBoundingBox().getMin()[2]));
    m_moveAroundEngine->distance.setValue (dim_obj * 10);
    displayFollowedObject( m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
  }
  else {
    m_moveAroundEngine->distance.setValue (6);
  }

  pointAtObject (TRUE);
  displayFollowedObject (TRUE);
  m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setChecked( true );
  m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setChecked( false );
}


////////////////////////////////////////////////////////////////////////
// 
void QLmv::followAsFreeView ()
{
  m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setEnabled( true );
  m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->setEnabled( true );
  m_moveAroundEngine->azimuth.setValue (0.3f);
  m_moveAroundEngine->site.setValue (0.f);
  if (m_trackSwitch->getNumChildren () > 0) {
    SoGetBoundingBoxAction action (SbViewportRegion (100, 100));
    m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
    action.apply (m_trackSwitch);
    float dim_obj = local_MAX (action.getBoundingBox().getMax()[0] - action.getBoundingBox().getMin()[0],
                        local_MAX (action.getBoundingBox().getMax()[1] - action.getBoundingBox().getMin()[1],
                              action.getBoundingBox().getMax()[2] - action.getBoundingBox().getMin()[2]));
    m_moveAroundEngine->distance.setValue (dim_obj * 4);
    displayFollowedObject( m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
  }
  else {
    m_moveAroundEngine->distance.setValue (4);
  }

  pointAtObject (TRUE);
  displayFollowedObject (TRUE);
  m_animMenuIdsMap[MNU_ANIM_INSIDE_VIEW]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_VIEW_FROM_BEHIND]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_TOP_VIEW]->setChecked( false );
  m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->setChecked( true );
}


////////////////////////////////////////////////////////////////////////
// Choose the filename of the file to open
void QLmv::onMnuFileOpen ()
{
  SbString oivDataPath = SbFileHelper::expandString( "$OIVHOME/data/models" );

  QString filter0 = tr("Inventor files (*.iv *.ivz)");
  QString filter1 = tr("All files (*.*)");
  QString filename = QFileDialog::getOpenFileName( this,
                                                   "Load an iv file",
                                                   QString::fromUtf16( oivDataPath.toUtf16() ),
                                                   filter0 + ";;" +
                                                   filter1 );
  if ( !filename.isEmpty() )
    openFile( filename );
}

////////////////////////////////////////////////////////////////////////
// Reload the last opened file
void QLmv::onMnuFileReload ()
{
  openFile (m_filename);
}

////////////////////////////////////////////////////////////////////////
void QLmv::onMnuApplyGravity ()
{
  if ( !m_walkViewerMenuItem->isChecked() )
    return;

  SoCamera *camera = m_currentViewer->getCamera();
  SbVec3f camPos = camera->position.getValue();
  // do the picking
  SbVec2s size = m_currentViewer->getSize();
  SoRayPickAction pick (size);
  SbVec3f downDir(0, -1, 0);
  // Note we must set the "nearDistance" to a small value to force picking
  // only objects that are below us.  By default near clipping is off,
  // allowing bogus picking of objects above us.
  pick.setRay(camPos, downDir, .001f);
  pick.setPickAll(FALSE);
  pick.apply(m_currentViewer->getSceneManager()->getSceneGraph());
        
  // makes sure something got picked, then
  // get the picked point.
  SoPickedPoint *pp = pick.getPickedPoint();
  if (pp == NULL) return;
  SbVec3f normal = pp->getNormal();
    
  // check whether the normal is pointing toward the camera, else
  // flip the normal around.
  SbVec3f point = pp->getPoint();;
  if (normal.dot(camera->position.getValue() - point) < 0) {
    normal.negate();    
  }
  SbVec3f upDir = m_walkViewer->getUpDirection();
  camera->position.setValue(point + normal * m_gravityOffset);
  SbRotation rot(upDir, normal);
    
  // rotate the camera 
  if (camera != NULL) {
    camera->orientation = rot * camera->orientation.getValue();
  }
}


////////////////////////////////////////////////////////////////////////
void QLmv::onMnuAdaptiveView ()
{
  if (m_adaptiveViewing == NULL) {
    m_adaptiveViewing = new QLmvAdaptiveViewing (this);
  }
  m_adaptiveViewing->show ();
  m_adaptiveViewing->setViewer (m_currentViewer);
}


////////////////////////////////////////////////////////////////////////
// Toggle the toolbar
void QLmv::onMnuViewToolbar()
{
  if ( !m_viewToolbarMenuItem->isChecked() ) 
  {
    m_viewToolbarMenuItem->setChecked( false );
    m_toolbar->hide();
  }
  else {
    m_viewToolbarMenuItem->setChecked( true );
    m_toolbar->show();
  }
}


////////////////////////////////////////////////////////////////////////
// Toggle the toolbar
void QLmv::onMnuViewStatusbar()
{
  if ( !m_viewStatusBarMenuItem->isChecked() ) 
  {
    m_viewStatusBarMenuItem->setChecked( false );
    m_statusbar->hide();
  }
  else {
    m_viewStatusBarMenuItem->setChecked( true );
    m_statusbar->show();
  }
}


////////////////////////////////////////////////////////////////////////
void QLmv::onMnuViewSceneInfos()
{
  if ( m_viewSceneInfosMenuItem->isChecked() ) 
    m_viewSceneInfosMenuItem->setChecked( false );
  else
    m_viewSceneInfosMenuItem->setChecked( true );
}

////////////////////////////////////////////////////////////////////////
void QLmv::onMnuAnimations(QAction* id)
{
  int key = -1;
  std::map<int, QAction*>::iterator it;

  // Find the key
  for ( it = m_animMenuIdsMap.begin(); it != m_animMenuIdsMap.end(); it++ )
  {
    if ( id->text() == (it->second)->text() )
      key = it->first;
  }
  
  switch ( key ) 
  {
    case MNU_ANIM_INSIDE_VIEW:
      followAsInsideView ();
      break;
    case MNU_ANIM_VIEW_FROM_BEHIND:
      followAsFromBehindView ();
      break;
    case MNU_ANIM_TOP_VIEW:
      followAsTopView ();
      break;
    case MNU_ANIM_FREE_VIEW:
      followAsFreeView ();
      break;
    case MNU_ANIM_POINT_AT_OBJECT:
      pointAtObject( !m_animMenuIdsMap[MNU_ANIM_POINT_AT_OBJECT]->isChecked() );
      break;
    case MNU_ANIM_DISPLAY_OBJECT:
      displayFollowedObject( !m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
      break;
    case MNU_ANIM_ROLL_WITH_OBJECT:
      rollWithObject( !m_animMenuIdsMap[MNU_ANIM_ROLL_WITH_OBJECT]->isChecked() );
      break;
    case MNU_ANIM_NONE:
      followTrack( -1 );
      break;
    default:
      followTrack( key - MNU_ANIM_FIRST_TRACK );
      break;
  }
}

////////////////////////////////////////////////////////////////////////
// Displays the new fps rate
void QLmv::onFpsChange (float fps)
{
  if (m_fpsDisplay->isVisible()) {
    char strValue[128];
    sprintf(strValue, "%.1f", fps);
    m_fpsDisplay->setText (QString(tr("%1 fps")).arg(strValue));
    m_countTrianglesSensor->schedule();
  }
}

////////////////////////////////////////////////////////////////////////
// Function called when decimation statistics arrive from the viewer.
void QLmv::onNewPercentageStatistics (float percentage)
{
  if (!m_fpsDisplay->isEnabled()) return;

  char buf[256];
  sprintf(buf, "%.1f %%", 100.0 * percentage);
  if ( strcmp(buf, (m_decimationDisplay->text()).toLocal8Bit().data()) )
    m_decimationDisplay->setText( buf );
}


////////////////////////////////////////////////////////////////////////
// Function called to count the number of triangles.
void QLmv::onCountTriangles (SoSensor*)
{
  if (!m_fpsDisplay->isEnabled()) return;

  // Get full scene graph, and current decimation values:
  SoNode *graph = m_currentViewer->getSceneManager()->getSceneGraph();
  SoGLRenderAction *glAction = m_currentViewer->getGLRenderAction();
  m_countTrianglesAction->setDecimationValue(glAction->getDecimationType(), glAction->getDecimationPercentage());

  // Apply primitive count action:	
  m_countTrianglesAction->apply(graph);
  int32_t numTris   = m_countTrianglesAction->getTriangleCount();
  char buf[256];
  if (numTris <= 0)
    sprintf(buf, "%d triangle", numTris);
  else
    sprintf(buf, "%d triangles", numTris);
  if ( strcmp(buf, (m_numTrianglesDisplay->text()).toLocal8Bit().data()) )
    m_numTrianglesDisplay->setText( buf );
}


////////////////////////////////////////////////////////////////////////
// Function called to count the number of triangles.
void QLmv::onMouseMove (SoEventCallback* node)
{
  if ( m_animMenuIdsMap[MNU_ANIM_NONE]->isChecked() ) return;
  if ( !m_animMenuIdsMap[MNU_ANIM_FREE_VIEW]->isChecked() ) return;

  if (node->getEvent()->isOfType (SoMouseButtonEvent::getClassTypeId())) {
    if (m_freeViewMode == ANGLE_CHANGING) {
      if (SoMouseButtonEvent::isButtonReleaseEvent (node->getEvent(), SoMouseButtonEvent::BUTTON1)) {
        m_freeViewMode = NONE;
      }
    }
    else if (m_freeViewMode == DISTANCE_CHANGING) {
      if (SoMouseButtonEvent::isButtonReleaseEvent (node->getEvent(), SoMouseButtonEvent::BUTTON2)) {
        m_freeViewMode = NONE;
      }
    }
    else {
      if (SoMouseButtonEvent::isButtonPressEvent (node->getEvent(), SoMouseButtonEvent::BUTTON1)) {
        m_freeViewMode = ANGLE_CHANGING;
        m_freeViewStartingAzimuth = m_moveAroundEngine->azimuth.getValue();
        m_freeViewStartingSite = m_moveAroundEngine->site.getValue();
        m_freeViewStartingMousePos = node->getEvent()->getPosition();
      }
      else if (SoMouseButtonEvent::isButtonPressEvent (node->getEvent(), SoMouseButtonEvent::BUTTON2)) {
        m_freeViewMode = DISTANCE_CHANGING;
        float minimumStartingDistance;
        if (m_trackSwitch->getNumChildren () > 0) {
          SoGetBoundingBoxAction action (SbViewportRegion (100, 100));
          m_trackSwitch->whichChild.setValue (SO_SWITCH_ALL);
          action.apply (m_trackSwitch);
          float dim_obj = local_MAX (action.getBoundingBox().getMax()[0] - action.getBoundingBox().getMin()[0],
                              local_MAX (action.getBoundingBox().getMax()[1] - action.getBoundingBox().getMin()[1],
                                    action.getBoundingBox().getMax()[2] - action.getBoundingBox().getMin()[2]));
          minimumStartingDistance = dim_obj * 4;
          displayFollowedObject( m_animMenuIdsMap[MNU_ANIM_DISPLAY_OBJECT]->isChecked() );
        }
        else {
          minimumStartingDistance = 4;
        }
        m_freeViewStartingDistance = local_MAX (minimumStartingDistance, m_moveAroundEngine->distance.getValue());
        m_freeViewStartingMousePos = node->getEvent()->getPosition();
      }
    }
  }
  else {
    if (m_freeViewMode == ANGLE_CHANGING) {
      float newSite = m_freeViewStartingSite +
                      ((((float)(node->getEvent()->getPosition()[0])) * M_PI) / ((float)(m_currentViewer->getSize()[0]))) -
                      ((((float)(m_freeViewStartingMousePos[0])) * M_PI) / ((float)(m_currentViewer->getSize()[0])));
      while (newSite > M_PI) newSite = newSite - M_PI - M_PI;
      while (newSite < -M_PI) newSite = newSite + M_PI + M_PI;
      float newAzimuth = m_freeViewStartingAzimuth +
                      ((((float)(node->getEvent()->getPosition()[1])) * M_PI_2) / ((float)(m_currentViewer->getSize()[1]))) -
                      ((((float)(m_freeViewStartingMousePos[1])) * M_PI_2) / ((float)(m_currentViewer->getSize()[1])));
      newAzimuth = local_MAX (-M_PI_2, newAzimuth);
      newAzimuth = local_MIN (M_PI_2, newAzimuth);
      m_moveAroundEngine->site.setValue (newSite);
      m_moveAroundEngine->azimuth.setValue (newAzimuth);
    }
    else if (m_freeViewMode == DISTANCE_CHANGING) {
      float newDistance = local_MAX(0.F, ((float)(node->getEvent()->getPosition()[1])) * m_freeViewStartingDistance / ((float)(m_freeViewStartingMousePos[1])));
      m_moveAroundEngine->distance.setValue(newDistance);
    }
  }
}


void
QLmv::onSpaceballButton(SoEventCallback* node)
{
  const SoSpaceballButtonEvent *ev = 
    (const SoSpaceballButtonEvent *) node->getEvent();
  
  if (ev->getState() == SoButtonEvent::DOWN) {
    int which = ev->getButton();
    printf(" button %d\n",which);
    switch (which) {
    case 1:
      printf("rotation only mode \n");
      m_spaceball_rotationMode    = 1;
      m_spaceball_translationMode = 0;
      break;
    case 2:
      printf("translation only mode \n");
      m_spaceball_translationMode = 1;
      m_spaceball_rotationMode    = 0;
      break;
    case 3:
      printf("translation and rotation mode \n");
      m_spaceball_translationMode = 1;
      m_spaceball_rotationMode    = 1;
      break;
    case 4:
      m_spaceballTranslation->translation.setValue(SbVec3f(0.0,0.0,0.0));
      m_spaceballRotation->rotation.setValue(SbRotation(0.0,0.0,0.0,1.0));
      m_currentViewer->resetToHomePosition();
      break;
    case 5:
      m_spaceball_rotScaleFactor *= 2.;
      printf("Increasing Rotation Scale Factor : %f\n",m_spaceball_rotScaleFactor);
      m_sb->setRotationScaleFactor(m_spaceball_rotScaleFactor);
      break;
    case 6:
      m_spaceball_rotScaleFactor /= 2.;
      printf("Decreasing Rotation Scale Factor : %f\n",m_spaceball_rotScaleFactor);
      m_sb->setRotationScaleFactor(m_spaceball_rotScaleFactor);
      break;
    case 7:
      m_spaceball_transScaleFactor *= 2.;
      printf("Increasing Translation Scale Factor: %f\n",m_spaceball_transScaleFactor);
      m_sb->setTranslationScaleFactor(m_spaceball_transScaleFactor);
      break;
    case 8:
      m_spaceball_transScaleFactor /= 2;
      printf("Decreasing Translation Scale Factor: %f\n",m_spaceball_transScaleFactor);
      m_sb->setTranslationScaleFactor(m_spaceball_transScaleFactor);
      break;
    default: 
      break;
    }
  }
}

void
QLmv::onMotion3Translation(SoEventCallback* node)
{
  if (!m_spaceball_translationMode) return;
  
  const SoMotion3Event *ev = 
    (const SoMotion3Event *) node->getEvent();
  
  m_spaceballTranslation->translation = m_spaceballTranslation->translation.getValue() + ev->getTranslation();
}   

void 
QLmv::onMotion3Rotation(SoEventCallback* node)
{
  if (!m_spaceball_rotationMode) return;
  
  const SoMotion3Event *ev = 
    (const SoMotion3Event *) node->getEvent();
  m_spaceballRotation->rotation = m_spaceballRotation->rotation.getValue() * ev->getRotation();
}


