#ifndef _Q_LMV_H_
#define _Q_LMV_H_

#include <QMenu>
#include <QStatusBar>
#include <QCloseEvent>
#include <QDropEvent>
#include <QDragEnterEvent>
#include <QToolButton>
#include <QMainWindow>

#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/SbPList.h>

#include <Inventor/STL/map>

class QTimer;
class QToolButton;
class SoQtViewer;
class SoQtExaminerViewer;
class SoQtWalkViewer;
class SoQtFlyViewer;
class SoQtPlaneViewer;
class SoQtFullViewer;
class SoQtSpaceball;
class QLineEdit;
class QToolBar;
class SoGroup;
class SoSeparator;
class SoTranslation;
class SoRotation;
class SoAction;
class SoCallback;
class SoQtCollisionViewer;
class SoSelection;
class SoShapeHints;
class QLmvTraversalTuning;
class QLmvAdaptiveViewing;
class SoNode;
class SoOneShotSensor;
class MoveAround;
class SoSwitch;
class SoEventCallback;
class InlineLoader;

enum FreeViewMode {
  NONE,
  ANGLE_CHANGING,
  DISTANCE_CHANGING
};

////////////////////////////////////////////////////////////////////////
class QLmv : public QMainWindow
{
  Q_OBJECT

public:
  QLmv (const char* configFileName, QWidget *parent=0, const char *name=0);
  ~QLmv ();

// types
  enum ViewerType {
    EXAMINER,
    WALK, 
    FLY,
    PLANE
  };

  enum { MAX_RECENTFILES = 5 };

// functions
  void openFile (const QString& filename);
  void setViewerType (ViewerType t);
  void setCollisionDetection (bool onOff = TRUE);
  void setStereo (bool onOff = TRUE, bool button = TRUE);
  void setTransparency (SoGLRenderAction::TransparencyType);
  void setAlphaTest (bool onOrOff);
  void displayRenderStats (bool);
  void reorganize();
  void shapeSimplify(bool doTriangleStrips);
  void globalSimplify(bool doTriangleStrips);
  void splitShapes();
  void reorganizeSimplified(SoNode* graph);
  void followTrack (int trackId);
  void displayFollowedObject (bool onOff = TRUE);
  void rollWithObject (bool onOff = TRUE);
  void pointAtObject (bool onOff = TRUE);
  void followAsInsideView ();
  void followAsFromBehindView ();
  void followAsTopView ();
  void followAsFreeView ();
  void setFsaa( bool onOff );

//drop functions
  void dragEnterEvent(QDragEnterEvent* event);
  void dropEvent(QDropEvent* event);

private:
// Status bar objects
  QStatusBar* m_statusbar;
  QLineEdit* m_fpsDisplay;
  QLineEdit* m_decimationDisplay;
  QLineEdit* m_numTrianglesDisplay;

// Tool buttons
  QToolBar* m_toolbar;
  QToolButton* m_fileOpenBtn;
  QToolButton* m_fileReopenBtn;
  QToolButton* m_stereoBtn;
  QToolButton* m_gravityBtn;

// Menus and menu items
  QMenuBar* m_menubar;
  QMenu* m_fileMenu;
  QMenu* m_recentFilesMenu;
  QAction* m_fileMenuId;
  QAction* m_fileReloadMenuItem;
  QMenu* m_viewMenu;
  QAction* m_viewMenuId;
  QAction* m_examinerViewerMenuItem;
  QAction* m_flyViewerMenuItem;
  QAction* m_walkViewerMenuItem;
  QAction* m_planeViewerMenuItem;
  QAction* m_collisionDetectionMenuItem;
  QAction* m_applyGravityMenuItem;
  QAction* m_stereoMenuItem;
  QAction* m_stereoDialogMenuItem;
  QAction* m_viewToolbarMenuItem;
  QAction* m_viewStatusBarMenuItem;
  QAction* m_viewRenderStatsMenuItem;
  QAction* m_viewSceneInfosMenuItem;
  QAction* m_adaptiveViewMenuItem;
  QMenu* m_transparencyMenu;
  QAction* m_transparencyMenuId;
  QAction* m_noSortTransparencyMenuItem;
  QAction* m_opaqueFirstTransparencyMenuItem;
  QAction* m_sortedObjectTransparencyMenuItem;
  QAction* m_sortedPixelTransparencyMenuItem;
  QAction* m_alphaTestTransparencyMenuItem;
  QAction* m_fsaaMenuItem;
  QAction* m_traversalMenuId;
  QAction* m_editTraversalParametersMenuItem;
  QMenu* m_processingMenu;
  QAction* m_processingMenuId;
  QAction* m_loadInlinesItem;
  QAction* m_reorganizeMenuItem;
  QAction* m_shapeSimplifyMenuItem;
  QAction* m_globalSimplifyMenuItem;
  QAction* m_shapeSimplifyTSMenuItem;
  QAction* m_globalSimplifyTSMenuItem;
  QAction* m_splitShapeMenuItem;
  QMenu* m_viewpointsMenu;
  QAction* m_viewpointsMenuId;
  std::map<int, QAction*> m_vpMenuIdsMap;
  QMenu* m_animationsMenu;
  QAction* m_animationsMenuId;
  std::map<int, QAction*> m_animMenuIdsMap;

// Dialogs
  QLmvTraversalTuning* m_traversalTuning;
  QLmvAdaptiveViewing* m_adaptiveViewing;

// Viewers, cameras and viewpoints
  SoQtExaminerViewer* m_examinerViewer;
  SoQtWalkViewer* m_walkViewer;
  SoQtFlyViewer* m_flyViewer;
  SoQtPlaneViewer* m_planeViewer;
  SoQtFullViewer* m_currentViewer;
  SoQtCollisionViewer* m_collisionViewer;
  float m_collisionDistance;
  SoGroup* m_cameraParent;
  SbPList m_vpList;
  SbPList m_trackList;
  SoSwitch* m_trackSwitch;
  MoveAround* m_moveAroundEngine;
  int m_currentTrack;
  FreeViewMode m_freeViewMode;
  float m_freeViewStartingAzimuth;
  float m_freeViewStartingSite;
  float m_freeViewStartingDistance;
  SbVec2s m_freeViewStartingMousePos;
  SoEventCallback* m_movingAroundCB;

// Scene graphs
  QString m_filename;
  SoGroup* m_loadedSG;
  SoSeparator* m_displayedSG;
  SoSeparator* m_separatorTraversalNode;
  SoSelection* m_selectionNode;
  SoShapeHints     *shapeHints;

// Misc
  SoOneShotSensor* m_countTrianglesSensor;
  SoGetPrimitiveCountAction* m_countTrianglesAction;
  QPoint posStereoDialog;
  float m_gravityOffset;

//To Load Inlines
  InlineLoader *inlineLoader;
  static void readingInlineCB(const SbString& filename, void *userData, InlineLoader *);

// Most Recent Files List and functions
  QStringList m_recentFiles;
  void updateRecentFiles( const QString& filename );
  void removeRecentFiles( const QString& filename );
  SbBool m_fileMenuConnected;

// private callbacks
  static void onFpsChangeCB (float fps, void* userData, SoQtViewer*)
    {((QLmv*)userData)->onFpsChange(fps);}
  int m_occlusionStatsNumObjRendered;
  int m_occlusionStatsNumObj;
  int m_occlusionStatsNumOccluders;
  static void onCountTrianglesCB (void *userData, SoSensor* sensor)
    {((QLmv*)userData)->onCountTriangles (sensor);}
  static void onNewPercentageStatisticsCB (float decimation, void *userData, SoQtViewer*)
    {((QLmv*)userData)->onNewPercentageStatistics (decimation);}
  static void onMouseMoveCB (void *userData, SoEventCallback* node)
    {((QLmv*)userData)->onMouseMove (node);}
  static void onSpaceballButtonCB (void *userData, SoEventCallback* node)
    {((QLmv*)userData)->onSpaceballButton (node);}
  static void onMotion3TranslationCB (void *userData, SoEventCallback* node)
    {((QLmv*)userData)->onMotion3Translation (node);}
  static void onMotion3RotationCB (void *userData, SoEventCallback* node)
    {((QLmv*)userData)->onMotion3Rotation (node);}

// SpaceBall
  SbBool m_spaceball_available;
  SoTranslation* m_spaceballTranslation;
  SoRotation* m_spaceballRotation;
  SoQtSpaceball* m_sb;
  SoEventCallback* m_spaceballCB;
  SbBool m_spaceball_rotationMode;
  SbBool m_spaceball_translationMode;
  float m_spaceball_rotScaleFactor;
  float m_spaceball_transScaleFactor;

protected:
    virtual void closeEvent( QCloseEvent * );

private Q_SLOTS:
  void onMnuFileOpen ();
  void onMnuFileReload ();
  void fileOpenRecent( QAction* index );
  void updateRecentFilesMenu();
  void saveOptions();
  void fileQuit();
  void onMnuExaminerViewer () {setViewerType (EXAMINER);}
  void onMnuWalkViewer () {setViewerType (WALK);}
  void onMnuFlyViewer () {setViewerType (FLY);}
  void onMnuPlaneViewer () {setViewerType (PLANE);}
  void onMnuCollisionDetection() {setCollisionDetection(m_collisionDetectionMenuItem->isChecked());}
  void onMnuApplyGravity();
  void onMnuStereo() {setStereo(m_stereoMenuItem->isChecked(), FALSE);}
  void onBtnStereo() {setStereo(/*m_stereoBtn->isDown()*/m_stereoMenuItem->isChecked(), TRUE);}
  void onMnuStereoDialog();
  void onMnuAdaptiveView();
  void onMnuNoSortTransparency() {setTransparency (SoGLRenderAction::NO_SORT);}
  void onMnuOpaqueFirstTransparency() {setTransparency (SoGLRenderAction::OPAQUE_FIRST);}
  void onMnuSortedObjectTransparency() {setTransparency (SoGLRenderAction::SORTED_OBJECT);}
  void onMnuSortedPixelTransparency() { setTransparency( SoGLRenderAction::SORTED_PIXEL); };
  void onMnuAlphaTestTransparency() {setAlphaTest(m_alphaTestTransparencyMenuItem->isChecked());}
  void onMenuViewFsaa() {setFsaa(m_fsaaMenuItem->isChecked());}
  void onMnuViewToolbar();
  void onMnuViewStatusbar();
  void onMnuViewRenderStats() {displayRenderStats (m_viewRenderStatsMenuItem->isChecked());}
  void onMnuViewSceneInfos();
  void onMnuLoadInlines();

  void onMnuReorganizeProcess() {reorganize();}
  void onMnuShapeSimplifyProcess() {shapeSimplify (FALSE);}
  void onMnuGlobalSimplifyProcess() {globalSimplify (FALSE);}
  void onMnuShapeSimplifyTSProcess() {shapeSimplify (TRUE);}
  void onMnuGlobalSimplifyTSProcess() {globalSimplify (TRUE);}
  void onMnuSplitShapesProcess() {splitShapes();}
  void onMnuAnimations(QAction*);

  void onFpsChange (float fps);
  void onNewPercentageStatistics (float percentage);
  void onCountTriangles (SoSensor* sensor);
  void onMouseMove (SoEventCallback* node);
  void onSpaceballButton (SoEventCallback* node);
  void onMotion3Translation (SoEventCallback* node);
  void onMotion3Rotation (SoEventCallback* node);
  void onSetStatusMessage (QString string) {m_statusbar->showMessage(string);}
};


#endif // _Q_LMV_H_


