/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Fabien ARNAUD & Romain BERTHON (MMM YYYY)
** Modified by : Tristan Mehamli (MMM YYYY)
**=======================================================================*/

#include <QtCore/qglobal.h>
#include <Inventor/Qt/OivQtCompat.h>

#include "QLmvAdaptiveViewing.h"

#include <QtCore/QSize>
#include <QSlider>
#include <QCheckBox>
#include <QLabel>
#include <QSpinBox>
#include <QLayout>
#include <QVBoxLayout>
#include <QBoxLayout>
#include <QRadioButton>
#include <QHBoxLayout>
#include <QButtonGroup>
#include <QGroupBox>
#include <QWidget>

#include <Inventor/Qt/viewers/SoQtViewer.h>


////////////////////////////////////////////////////////////////////////
// Constructor
QLmvAdaptiveViewing::QLmvAdaptiveViewing (QWidget* parent) :
QDialog( parent, Qt::MSWindowsFixedSizeDialogHint | 
         Qt::WindowTitleHint | 
         Qt::WindowSystemMenuHint )
{
  // Tune the dialog
  setModal( false );
  m_viewer = NULL;
  setWindowTitle(tr("Adaptive Viewing"));

  // Build widgets
  QGroupBox* bggb = new QGroupBox(this);
  QButtonGroup* buttonGroup = new QButtonGroup(bggb);

  m_normalRButton = new QRadioButton(tr("Normal"), bggb);
  QLabel* emptyLabel = new QLabel("", bggb);
  buttonGroup->addButton( m_normalRButton );
  connect(m_normalRButton, SIGNAL(toggled(bool)), this, SLOT(onNormalToggled(bool)));

  m_numOfTrianglesRButton = new QRadioButton (tr("Fixed Number of Triangles"), bggb);
  buttonGroup->addButton( m_numOfTrianglesRButton );
  connect(m_numOfTrianglesRButton, SIGNAL(toggled(bool)), this, SLOT(onNumOfTrianglesToggled(bool)));
  QWidget* numOfTrianglesHBox = new QWidget( bggb );
  QHBoxLayout* hlnot = new QHBoxLayout( numOfTrianglesHBox );
  hlnot->setStretchFactor (new QLabel ("         ", numOfTrianglesHBox), 0);
  m_numOfTrianglesSpinBox = new QSpinBox (numOfTrianglesHBox);
  connect( m_numOfTrianglesSpinBox, SIGNAL(valueChanged(int)), this, SLOT(onNumOfTrianglesValueChanged (int)));
  m_numOfTrianglesSpinBox->setMinimum(0);
  m_numOfTrianglesSpinBox->setMaximum(10000000);
  m_numOfTrianglesSpinBox->setValue(100000);
  m_numOfTrianglesSpinBox->setSingleStep (100);
  m_numOfTrianglesSpinBox->setEnabled (FALSE);
  hlnot->setStretchFactor (m_numOfTrianglesSpinBox, 0);
  hlnot->setStretchFactor (new QLabel ("", numOfTrianglesHBox), 1);

  m_fpsRButton = new QRadioButton(tr("Frames Per Second"), bggb);
  buttonGroup->addButton( m_fpsRButton );
  connect(m_fpsRButton, SIGNAL(toggled(bool)), this, SLOT(onFpsToggled (bool)));
  QWidget* fpsHBox = new QWidget(bggb);
  QHBoxLayout* hlfps = new QHBoxLayout( fpsHBox );
  hlfps->setStretchFactor (new QLabel ("         ", fpsHBox), 0);
  m_fpsSlider = new QSlider (Qt::Horizontal, fpsHBox);
  connect(m_fpsSlider, SIGNAL(valueChanged(int)), this, SLOT (onFpsValueChanged (int)));
  m_fpsSlider->setEnabled (FALSE);
  m_fpsSlider->setMinimumWidth (150);
  m_fpsSlider->setRange (0, 300);
  hlfps->setStretchFactor (m_fpsSlider, 0);
  hlfps->setStretchFactor (new QLabel ("", fpsHBox), 1);
  m_fpsLabel = new QLabel ("12345", fpsHBox);
  m_fpsLabel->setEnabled (FALSE);
  hlfps->setStretchFactor (m_fpsLabel, 0);

  m_percentageRButton = new QRadioButton (tr("Fixed Percentage"), bggb);
  buttonGroup->addButton( m_percentageRButton );
  connect(m_percentageRButton, SIGNAL(toggled(bool)), this, SLOT (onPercentageToggled(bool)));
  QWidget* percentageHBox = new QWidget(bggb);
  QHBoxLayout* hlp = new QHBoxLayout( percentageHBox );
  hlp->setStretchFactor (new QLabel ("         ", percentageHBox), 0);
  m_percentageSlider = new QSlider (Qt::Horizontal, percentageHBox);
  connect (m_percentageSlider, SIGNAL(valueChanged(int)), this, SLOT (onPercentageValueChanged(int)));
  m_percentageSlider->setEnabled (FALSE);
  m_percentageSlider->setMinimumWidth (150);
  m_percentageSlider->setRange (0, 100);
  hlp->setStretchFactor (m_percentageSlider, 0);
  hlp->setStretchFactor (new QLabel ("", percentageHBox), 1);
  m_percentageLabel = new QLabel ("12345", percentageHBox);
  m_percentageLabel->setEnabled (FALSE);
  hlp->setStretchFactor (m_percentageLabel, 0);

  QBoxLayout* groupLayout = new QVBoxLayout (bggb);
  groupLayout->setContentsMargins( 10, 10, 10, 10 );
  groupLayout->setSpacing( 5 );
  groupLayout->addWidget (m_normalRButton);
  groupLayout->addWidget (emptyLabel);
  groupLayout->addWidget (m_numOfTrianglesRButton);
  groupLayout->addWidget (numOfTrianglesHBox);
  groupLayout->addWidget (m_fpsRButton);
  groupLayout->addWidget (fpsHBox);
  groupLayout->addWidget (m_percentageRButton);
  groupLayout->addWidget (percentageHBox);

  m_renderWhenStillCheckBox = new QCheckBox (tr("Do Full Rendering When Still"), this);
  connect (m_renderWhenStillCheckBox, SIGNAL(toggled(bool)), this, SLOT (onRenderWhenStillToggled (bool)));

  QBoxLayout* main_layout = new QVBoxLayout (this);
  main_layout->setContentsMargins( 10, 10, 10, 10 );
  main_layout->addWidget (bggb);
  main_layout->addSpacing (10);
  main_layout->addWidget (m_renderWhenStillCheckBox);

  // Default values
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::setViewer (SoQtViewer* viewer)
{
  m_viewer = viewer;
  if (m_viewer == NULL) {
    return;
  }
  m_renderWhenStillCheckBox->setChecked (m_viewer->isFullRenderingWhenStill());
  switch (m_viewer->getDecimationStrategy ()) {
  case SoQtViewer::NORMAL :
    m_normalRButton->setChecked (TRUE);
    break;
  case SoQtViewer::FIXED_NUM_TRIANGLES :
    m_numOfTrianglesRButton->setChecked (TRUE);
    break;
  case SoQtViewer::FRAMES_PER_SECOND :
    m_fpsRButton->setChecked (TRUE);
    break;
  case SoQtViewer::FIXED_PERCENTAGE :
    m_percentageRButton->setChecked (TRUE);
    break;
  }
	m_numOfTrianglesSpinBox->setValue (m_viewer->getGoalNumberOfTriangles());
	m_fpsSlider->setValue ((int)(m_viewer->getGoalFramesPerSecond() * 10));
	m_percentageSlider->setValue ((int)(m_viewer->getFixedPercentage() * 100.0));
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onNormalToggled (bool onOrOff)
{
  if (m_viewer != NULL) {
    if (onOrOff) {
      m_viewer->setDecimationStrategy (SoQtViewer::NORMAL);
    }
  }
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onNumOfTrianglesToggled (bool onOrOff)
{
  if (m_viewer != NULL) {
    if (onOrOff) {
      m_viewer->setDecimationStrategy (SoQtViewer::FIXED_NUM_TRIANGLES);
    }
  }
  m_numOfTrianglesSpinBox->setEnabled (onOrOff);
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onFpsToggled (bool onOrOff)
{
  if (m_viewer != NULL) {
    if (onOrOff) {
      m_viewer->setDecimationStrategy (SoQtViewer::FRAMES_PER_SECOND);
    }
  }
  m_fpsSlider->setEnabled (onOrOff);
  m_fpsLabel->setEnabled (onOrOff);
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onPercentageToggled (bool onOrOff)
{
  if (m_viewer != NULL) {
    if (onOrOff) {
      m_viewer->setDecimationStrategy (SoQtViewer::FIXED_PERCENTAGE);
    }
  }
  m_percentageSlider->setEnabled (onOrOff);
  m_percentageLabel->setEnabled (onOrOff);
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onRenderWhenStillToggled (bool onOrOff)
{
  if (m_viewer != NULL) {
    m_viewer->enableFullRenderingWhenStill (onOrOff);
  }
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onFpsValueChanged (int val)
{
  char newVal[255];
  sprintf (newVal, "%.1f", float (val) / 10.0f);
  m_fpsLabel->setText (newVal);
  if (m_viewer != NULL) {
    m_viewer->setGoalFramesPerSecond (float (val) / 10.0f);
  }
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onPercentageValueChanged (int val)
{
  char newVal[255];
  sprintf (newVal, "%d %%", val);
  m_percentageLabel->setText (newVal);
  if (m_viewer != NULL) {
    m_viewer->setFixedPercentage (float(val)/100.0f);
  }
}


////////////////////////////////////////////////////////////////////////
void QLmvAdaptiveViewing::onNumOfTrianglesValueChanged (int val)
{
  if (m_viewer != NULL) {
    m_viewer->setGoalNumberOfTriangles (val);
  }
}


