
#include <Inventor/Qt/SoQt.h>
#include <Inventor/Qt/viewers/SoQtExaminerViewer.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoCube.h>

#include <Inventor/Qt/devices/SoQtSpaceball.h>
#include "Inventor/events/SoSpaceballButtonEvent.h"
#include "Inventor/events/SoMotion3Event.h"

#include <Inventor/helpers/SbFileHelper.h>

SoQtSpaceball *sb = NULL;

SbBool rotationMode;
SbBool translationMode;
float rotScaleFactor;
float transScaleFactor;
int stereoMode;
SoQtExaminerViewer *vwr = NULL;

void
spaceballButtonCB(void *, SoEventCallback *cb)
{
  const SoSpaceballButtonEvent *ev = 
    (const SoSpaceballButtonEvent *) cb->getEvent();
  
  if (ev->getState() == SoButtonEvent::DOWN) {
    int which = ev->getButton();
    printf(" button %d\n",which);
    switch (which) {
    case 1:
      printf("rotation only mode \n");
      rotationMode    = 1;
      translationMode = 0;
      break;
    case 2:
      printf("translation only mode \n");
      translationMode = 1;
      rotationMode    = 0;
      break;
    case 3:
      printf("translation and rotation mode \n");
      translationMode = 1;
      rotationMode    = 1;
      break;
    case 4:
      rotScaleFactor *= 2.;
      printf("Increasing Rotation Scale Factor : %f\n",rotScaleFactor);
      sb->setRotationScaleFactor(rotScaleFactor);
      break;
    case 5:
      rotScaleFactor /= 2.;
      printf("Decreasing Rotation Scale Factor : %f\n",rotScaleFactor);
      sb->setRotationScaleFactor(rotScaleFactor);
      break;
    case 6:
      transScaleFactor *= 2.;
      printf("Increasing Translation Scale Factor: %f\n",transScaleFactor);
      sb->setTranslationScaleFactor(transScaleFactor);
      break;
    case 7:
      transScaleFactor /= 2;
      printf("Decreasing Translation Scale Factor: %f\n",transScaleFactor);
      sb->setTranslationScaleFactor(transScaleFactor);
      break;
    case 8:
      stereoMode = 1-stereoMode;
      printf("Stereo mode %d\n",stereoMode);
      vwr->setStereoViewing(stereoMode);
      break;
    default: 
      break;
    }
  }
}

void
motion3TranslationCB(void *userData, SoEventCallback *cb)
{
  if (!translationMode) return;
  
  const SoMotion3Event *ev = 
    (const SoMotion3Event *) cb->getEvent();
  SoTranslation *translation = (SoTranslation *) userData;
  
  
  translation->translation = translation->translation.getValue() + ev->getTranslation();
  cb->setHandled();
}   

void
motion3RotationCB(void *userData, SoEventCallback *cb)
{
  if (!rotationMode) return;
  
  const SoMotion3Event *ev = 
    (const SoMotion3Event *) cb->getEvent();
  
  SoRotation *rotation = (SoRotation *) userData;
  
  rotation->rotation = rotation->rotation.getValue() * ev->getRotation();
  cb->setHandled();
}   

SoNode *
buildSceneGraph(int argc, char *argv[])
{
  // load the axis scenegraph
  SoInput input;
  if (!input.openFile("$OIVHOME/examples/source/Inventor/SoQt/QtSpaceMouse/axis.iv"))
  {
    printf("Cannot open file $OIVHOME/examples/source/Inventor/SoQt/QtSpaceMouse/axis.iv \n");
    exit(-1);
  }
  SoRef<SoSeparator> axis = SoDB::readAll(&input);
  if ( axis.ptr()==NULL)
    return NULL;

  SoSeparator *sep = new SoSeparator;
  SoEventCallback *cb = new SoEventCallback;
  SoRotation *rotation = new SoRotation;
  SoTranslation *translation = new SoTranslation;
  
  sep->addChild(translation);
  sep->addChild(rotation);
  sep->addChild(cb);
  sep->addChild(axis.ptr());
  
  // Set up event callbacks
  cb->addEventCallback(
    SoSpaceballButtonEvent::getClassTypeId(),
    spaceballButtonCB, 
    sep);
  cb->addEventCallback(
    SoMotion3Event::getClassTypeId(),
    motion3TranslationCB, 
    translation);
  cb->addEventCallback(
    SoMotion3Event::getClassTypeId(),
    motion3RotationCB, 
    rotation);
  
  // if the user gave us a scenegraph then add it to the global scenegraph
  if (argc == 2)
  {
    if (!input.openFile(SbFileHelper::expandString(argv[1])))
    {
      printf("Cannot open file %s \n",SbFileHelper::expandString(argv[1]).toLatin1()); 
    }
    else
    {
      SoRef<SoSeparator> scene = SoDB::readAll(&input);
      if ( scene.ptr()!=NULL )
        sep->addChild(scene.ptr());
    }
  }

  return sep;
}

int
main(int argc, char **argv)
{
  QWidget *mainWindow = SoQt::init(argv[0]);
  
  vwr = new SoQtExaminerViewer(mainWindow);
  vwr->setSceneGraph(buildSceneGraph(argc,argv));
  vwr->setTitle("Space Ball and Space Mouse Device");
  vwr->setViewing(FALSE);   // come up in pick mode
  vwr->setHeadlight(TRUE); 
  
  
  if (! SoQtSpaceball::exists()) {
    fprintf(stderr, "Sorry, no Space Ball or Magellan Space Mouse on this display!\n");
  } 
  else {
    sb = new SoQtSpaceball;
    vwr->registerDevice(sb);
    rotScaleFactor   = sb->getRotationScaleFactor();
    transScaleFactor = sb->getTranslationScaleFactor();
    
    printf("Default rotation scale factor %f\n",rotScaleFactor);
    printf("Default translation scale factor %f\n",transScaleFactor);
  }
  stereoMode = 0;
  
  rotationMode    = 1;
  translationMode = 0;
  
  vwr->show();
  SoQt::show(mainWindow);
  SoQt::mainLoop();

  delete vwr;
  delete sb;

  SoQt::finish();
  return 0;
}


