/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Fabien ARNAUD & Romain BERTHON (MMM YYYY)
** Modified by : Tristan Mehamli (MMM YYYY)
**=======================================================================*/

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4996) // Disable VRML deprecation warnings
#endif

#include <QtCore/QVariant>
#include <QApplication>
#include <QClipboard>
#include <QDragEnterEvent>
#include <QDropEvent>
#include <QMessageBox>
#include <QImage>
#include <QPrinter>
#include <QPainter>
#include <QPrintDialog>
#include <QVBoxLayout>
#include <QFileDialog>
#include <QtCore/QUrl>
#include <QtCore/QList>
#include <QtCore/QMimeData>

#include "QtTViewTreeView.h"
#include "QtTViewViewer.h"

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoOutput.h>
#include <Inventor/actions/SoLineHighlightRenderAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoWWWInline.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoTexture2.h>


/////////////////////////////////////////////////
// Event Callback :
// saves the scene to file when F1 key is pressed

static void keyCB( void* data, SoEventCallback* node )
{
  SoNode* root = (SoNode*)data;

  if ( SO_KEY_PRESS_EVENT(node->getEvent(), F1) ) 
  {
    SoOutput output;
    output.openFile( "out.iv" );
    SoWriteAction action( &output );
    action.apply( root );
    output.closeFile();
  }
}

///////////////////////////////////////
// Constructor

QtTViewViewer::QtTViewViewer( QWidget* parent, const char* name )
: QWidget( parent )
{
  m_vblayout = new QVBoxLayout( this );
  setObjectName( name );
  setAcceptDrops( true );

  m_pViewer = new SoQtExaminerViewer( this, "Examiner" );
  m_vblayout->setContentsMargins( 0, 0, 0, 0 );
  m_vblayout->setSpacing( 0 ) ;
  m_vblayout->addWidget( m_pViewer->getWidget() );

  m_pSelection = new SoSelection();
  m_pSelection->policy = SoSelection::SINGLE;
  m_pSelection->addSelectionCallback( addSelectPath, this );
  m_pViewer->setGLRenderAction( new SoLineHighlightRenderAction() );
  m_pViewer->redrawOnSelectionChange( m_pSelection );
  SoWWWInline::setFetchURLCallBack( WWWInlineFetchURLCB, this );
  
  m_pTreeWindow = NULL;
  showFullScreen();
}

QtTViewViewer::~QtTViewViewer()
{
}

void 
QtTViewViewer::WWWInlineFetchURLCB( const SbString& url, void* /*userdata*/, SoWWWInline* node )
{
  SoInput input;
  if ( !input.openFile(url) ) 
    return;
  SoSeparator* root = SoDB::readAll( &input );
  if ( !root ) 
    return ;
  input.closeFile();
  node->setChildData( root );
}


void 
QtTViewViewer::dropEvent( QDropEvent* event )  
{
  QString buffer;
  QList<QUrl> fileList;
  SoInput input;

  if ( event->mimeData()->hasUrls() ) 
  {
    fileList = event->mimeData()->urls();
    if ( !input.openFile( SbString().fromUtf16(fileList[0].path().utf16()), TRUE ) ) 
      return;
  }
  else if ( event->mimeData()->hasText()  )
  {
    buffer = event->mimeData()->text();
    SbString str;
    str.fromUtf16( buffer.utf16() );
    input.setBuffer( (void*)((const char *)str.toLatin1()), str.getLength() );
  }
  else 
    return;

  SbString dir = SbFileHelper::getDirName( input.getCurStringFileName() );

  if (!dir.isEmpty())
  {
    SoInput::removeDirectory(dir);
    SoInput::addDirectoryLast(dir);
  }

  SoSeparator* root = SoDB::readAll( &input );

  input.closeFile();
  if ( root ) 
    setScene( (SoNode*)root );
}

void 
QtTViewViewer::dragEnterEvent( QDragEnterEvent* event )  
{
  if ( event->mimeData()->hasText() || event->mimeData()->hasUrls() )
    event->accept();
}

void 
QtTViewViewer::setTreeView( QtTViewTreeView* treeView ) 
{
  m_pTreeWindow = treeView;
  treeView->m_pGraphicWindow = this;
}

void 
QtTViewViewer::setScene( SoNode* node ) 
{
  m_pSelection->deselectAll();
  m_pSelection->removeAllChildren();
  if ( node ) 
  {
    m_pSelection->addChild( node );
    m_pViewer->setSceneGraph( m_pSelection );
  }
  if (m_pTreeWindow) 
    m_pTreeWindow->initTree( node );

  SoEventCallback* cbnode = new SoEventCallback();

  cbnode->addEventCallback( SoKeyboardEvent::getClassTypeId(), keyCB, m_pSelection );

  m_pSelection->addChild( cbnode );
}

/////////////////////////////////
// Handle selection highlighting

SoPath*
QtTViewViewer::getPathFromItem( QTreeWidgetItem* item )
{
  if ( !item ) 
    return (SoPath*)NULL;

  SbPList list;
  SoNode* node;

  do 
  {
    node = (SoNode*)(uintptr_t)item->text(2).toLong();
    list.append( node );
  } while( (item = item->parent()) );

  int numItems = list.getLength();
  SoPath* path = new SoPath( numItems + 1 );
  path->ref();
  path->setHead( m_pSelection );
  for ( int i = numItems - 1; i >= 0; i-- ) 
  {
    node = (SoNode*)list[i];
    path->append( node );
  }

  return path;
}

QTreeWidgetItem* 
QtTViewViewer::getItemFromPath( SoPath* path )
{
  QTreeWidgetItem* item = m_pTreeWindow->topLevelItem( 0 );

  int pathLen = path->getLength();
  for ( int i = 1; i < pathLen; i++ ) 
  {
    SoNode* node = path->getNode( i );
    int index = 0;
    while ( item && ((SoNode*)(uintptr_t)(item->text(2).toLong())) != node ) 
      item = item->parent()->child(index++);

    if ( !item ) 
      return NULL;
    if ( i != pathLen-1 ) 
      item = item->child( 0 );
  }

  return item;
}

void 
QtTViewViewer::highlightPath( QTreeWidgetItem* item )
{
  SoPath* path = getPathFromItem( item );
  m_pSelection->deselectAll();

  if ( path ) 
    m_pSelection->select( path );
}

void 
QtTViewViewer::addSelectPath( void* data, SoPath* path )
{
  QtTViewViewer* view = (QtTViewViewer*)data;
  QTreeWidgetItem* item = view->getItemFromPath( path );

  if ( item ) 
  {
    QTreeWidgetItem* parent = item->parent();
    while ( parent ) 
    {
      parent->setHidden( false );
      parent = parent->parent();
    }
    item->setSelected( true );
    view->m_pTreeWindow->setCurrentItem( item );
    view->m_pTreeWindow->scrollToItem( item );
  }
  else 
    view->m_pTreeWindow->clearSelection();
}

/////////////////////////////
// Cut & paste scene graphs

void 
QtTViewViewer::cut()
{
  if ( !m_pSelection->getNumSelected() ) 
    return;

  copy();

  SoPath* path = m_pSelection->getPath( 0 );
  path->ref();
  m_pSelection->deselectAll();
  SoGroup* group = (SoGroup*)path->getNodeFromTail( 1 );
  group->removeChild( path->getTail() );
  SbBool shapeFound = FALSE;
  int j = 0;
  while ( (! shapeFound) && (j < path->getLength()-1) ) 
  {
    SoNode* node = path->getNodeFromTail( j );
    if ( node->isOfType(SoSeparator::getClassTypeId()) ) 
    {
      SoSearchAction searchAction;
      searchAction.setFind( SoSearchAction::TYPE );
      searchAction.setType( SoShape::getClassTypeId() );
      searchAction.apply( node );
      // If no other shapes under this separator, delete it!
      if ( searchAction.getPath() == NULL ) 
      {
        group = (SoGroup*) path->getNodeFromTail( j+1 );
        group->removeChild( node );
        j = 0;
      }
      else 
        shapeFound = TRUE;
    }
    // Else a group with no children?
    else if ( node->isOfType(SoGroup::getClassTypeId()) &&
              (((SoGroup*)node)->getNumChildren() == 0) ) 
    {
        group = (SoGroup*)path->getNodeFromTail( j+1 );
        group->removeChild( node );
        j = 0;
    }
    // Else continue up the path looking for separators
    else 
      j++;
  }
  path->unref();
  m_pTreeWindow->initTree( m_pSelection->getChild(0) );
}

void 
QtTViewViewer::copy()
{
  if ( !m_pSelection->getNumSelected() ) 
    return;

  char* buffer = (char*)malloc( 32000 );
  SoOutput output;
  SoWriteAction writeAction( &output );
  output.setBuffer( buffer, 32000, reallocBufCB );
  writeAction.apply( m_pSelection->getPath(0) );
  if ( !buffer ) 
  {
    QMessageBox::warning( NULL, "Copy", "Can't copy data to clipboard. Out of memory" );
    return;
  }
  QMimeData* data = new QMimeData();
  data->setText( buffer );
  QApplication::clipboard()->setMimeData( data );
}

void 
QtTViewViewer::paste()
{
  QString buffer;

  if ( QApplication::clipboard()->mimeData()->hasText() ) 
  {
    buffer = QApplication::clipboard()->mimeData()->text();
    SoInput input;
    input.setBuffer( (void*)((const char *)buffer.toLocal8Bit().data()), buffer.length() );
    SoPath* path;
    SbBool res = SoDB::read( &input, path );
    if ( res ) 
    {
      m_pSelection->addChild( path->getNode(1) );
      m_pTreeWindow->initTree( m_pSelection->getChild(0) );
    }
  }
}

void* 
QtTViewViewer::reallocBufCB( void* ptr, size_t newSize )
{
  ptr = (char*)realloc( ptr, newSize );
  return( ptr );
}

void 
QtTViewViewer::printFile()
{
  QString result = QFileDialog::getSaveFileName( this, "Choose a file", "./",
#ifdef _WIN32
                                                 "Images (*.ps *.jpg *.bmp *.tif *.rgb)"
#else
                                                 "Images (*.ps *.jpg *.tif *.rgb)"
#endif
                                               );
  if ( !result.isEmpty() ) 
  {
#if 0
    SoOffscreenRenderer offscreen( m_pViewer->getViewportRegion() );
    offscreen.render( m_pViewer->getSceneManager()->getSceneGraph() );

    FILE* fp = SbFileHelper::open( SbString().fromUtf16( result.utf16() ), "wb" );

    if ( result.contains(QString(".ps"), Qt::CaseInsensitive) )
      offscreen.writeToPostScript( fp );
    else if ( result.contains(QString(".jpg"), Qt::CaseInsensitive) )
      offscreen.writeToJPEG( fp );
    else if ( result.contains(QString(".tif"), Qt::CaseInsensitive) ) 
      offscreen.writeToTIFF( fp );
    else if ( result.contains(QString(".rgb"), Qt::CaseInsensitive) )
      offscreen.writeToRGB( fp );
#ifdef _WIN32
    else if ( result.contains(QString(".bmp"), Qt::CaseInsensitive) )
      offscreen.writeToBMP( fp );
#endif
    SbFileHelper::close( fp );
#endif
    SoDebugError::postInfo("tTViewViewer::printFile", "Not implemented for oiv10");
  }
}

void 
QtTViewViewer::printQt()
{
#if 0
  SoOffscreenRenderer offscreen( m_pViewer->getViewportRegion() );
  // Set image buffer format to RGBA :
  offscreen.setComponents( SoOffscreenRenderer::RGB_TRANSPARENCY );
  // Render
  offscreen.render( m_pViewer->getSceneManager()->getSceneGraph() );

  short width,height;
  m_pViewer->getViewportRegion().getWindowSize().getValue(width, height);

  QImage image( width, height, QImage::Format_RGB32 );
  // memcpy(image.bits(), image.getBuffer(), image.byteCount());

  // Note: QImage most significant byte is used for the alpha buffer,
  // while SoOffscreenRendering stores in RGBA order. 

  QPrinter printer;
  QPrintDialog dialog( &printer, this );
  if ( dialog.exec() ) 
  {
    QPainter paint( &printer );
    paint.drawImage( 0, 0, image.mirrored(FALSE, TRUE) ); 
    // ...scanline order needs be reversed
    printer.newPage();
  }
#endif
  SoDebugError::postInfo("tTViewViewer::printQt", "Not implemented for oiv10");
}


#if defined( _WIN32 )
#pragma warning( pop )
#endif
