// Example - Get normals
//
// Copyright 2001 Template Graphics Software, Inc.
//
// Original: MMH
// Modified: 
//
// Notes:
// When left mouse button is pressed, searchs for vertex shapes
// and tries to find the normals that will be used.
//
// Viewer must be in selection mode (arrow cursor) to trigger
// the event callback.
//
// It doesn't matter how you get a pointer to the geometry
// node.  A callback action is just one way (you could also use
// picking for example).
//
// Only vertex shapes (FaceSet, etc) have explicit normals.
// It is not currently possible to access the normals used for
// a Cone, Sphere, etc.
//
// If there is no normal cache, getNormalCache returns NULL.
// This could happen because normals were explicitly specified
// either in an SoNormal or SoVertexProperty node.  It could
// also happen because the geometry node has not yet been
// traversed by a render action.
//
// Special Note: If the Open Inventor version is older than 3.0
// the getNormalCache method is not public.  In order to call
// it we cast the SoVertexShape pointer to "myVertexShape",
// which has the same header except getNormalCache is public.
// This hack is not necessary with version 3.0 and later.

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/nodes/SoVertexShape.h>
#include <Inventor/SoInput.h>
#include <Inventor/actions/SoCallbackAction.h>

// Header file we need specifically for normal cache
#include <Inventor/caches/SoNormalCache.h>

#include <Inventor/helpers/SbFileHelper.h>

// Header file we need for older versions of Open Inventor
#if SO_INVENTOR_VERSION < 3000
#include "myVertexShape.h"
#endif

/////////////////////////////////////////////////////////////
//
// Callback for vertex shapes

int numShapes = 0;

SoCallbackAction::Response
callback( void* /*userData*/, SoCallbackAction *action, const SoNode *node)
{
  numShapes++;
  SoVertexShape *pShape = (SoVertexShape *)node;
  int numNormals = 0;

  // Best case is normals specified in vertexProperty node
  SoVertexProperty *pVProp =
    (SoVertexProperty *)pShape->vertexProperty.getValue();
  if (pVProp) {
    int numNormals = pVProp->normal.getNum();
    if (numNormals > 0) {
      // Get pointer to array of normals (not actually used here)
      // Note that normalBinding affects number of normals and
      // how they relate to the geometry.
      printf("  %s : %d normals from vertexProperty node\n",
             node->getTypeId().getName().getString(), numNormals);
      return SoCallbackAction::CONTINUE;
    }
  }

  // Next best is normals inherited from traversal state
  numNormals = action->getNumNormals();
  if (numNormals > 0) {
    // Get normals one-by-one from traversal state
    // Note that normalBinding affects number of normals and
    // how they relate to the geometry.
    printf("  %s : %d normals from traversal state\n",
            node->getTypeId().getName().getString(), numNormals);
    return SoCallbackAction::CONTINUE;
  }

  // Else Open Inventor may have computed normals
#if SO_INVENTOR_VERSION < 3000
  myVertexShape *pFake = (myVertexShape*)pShape; // Hack for non-public method
#else
  SoVertexShape *pFake = pShape;
#endif
  SoNormalCache *pNormCache = pFake->getNormalCache();
  if (pNormCache) {
    numNormals = pNormCache->getNum();
    if (numNormals > 0) {
      // Get pointer to array of normals (not actually used here).
      // Note that number of normals is *not* affected by normal
      // binding or how vertex coords are shared between faces!
      //
      // There should be one normal for each vertex of each face
      // of the primitive.  Eg. an IndexedFaceSet containing two
      // quads will have 8 normals in the cache.  An
      // IndexedTriangleStripSet with 5 vertices will contain 4
      // triangles and so will have 12 (4 x 3) normals in cache.
      printf("  %s : %d normals from normal cache\n",
             node->getTypeId().getName().getString(), numNormals);
      return SoCallbackAction::CONTINUE;
    }
  }

  // Hmm, possibly normals have not been computed yet
  printf("Shape %p: No normals found\n", pShape);

  return SoCallbackAction::CONTINUE;
}

/////////////////////////////////////////////////////////////
//
// Callback for mouse click

void 
EventCallback(void *userData, SoEventCallback *node)
{
  const SoEvent *event = node->getEvent();

  // If left mouse button pressed...
  if (SO_MOUSE_PRESS_EVENT(event, BUTTON1)) {
    // Traverse the scene graph and call callback for
    // each vertex shape
    printf("Vertex shapes:\n");
    SoCallbackAction cba;
    cba.addPreCallback(SoVertexShape::getClassTypeId(), callback, NULL);
    cba.apply((SoNode*)userData);
    printf("Found %d vertex shapes\n", numShapes);
  }
}

/////////////////////////////////////////////////////////////

int
main(int argc, char **argv)
{
  // Check if input file exists

  SbString IvFileName = SbFileHelper::expandString( "$OIVHOME" ) + "/examples/source/Inventor/Techniques/GetNormals/test.iv";

  char *filename = (char*) IvFileName.toLatin1 ();
  if (argc > 1)
    filename = argv[1];
  FILE *fp = fopen(filename, "r");
  if (fp == NULL) {
    printf("Unable to open '%s'\n", filename);
    exit(1);
  }
  fclose(fp);
  if ( !SoPreferences::getBool("ATT_RUNNING", FALSE) )
    printf("Opened data file '%s'\n", filename);

  // Initialize Inventor
  Widget myWindow = SoXt::init(argv[0]);

  // Load the file (or create some dummy geometry)
  SoSeparator *pRoot = NULL;
  if (filename != NULL) {
    SoInput in;
    in.openFile(filename);
    pRoot = SoDB::readAll(&in);
    in.closeFile();
  }
  else {
    SoMaterial *pMatl = new SoMaterial;
    SoCone *pCone = new SoCone;

    pMatl->diffuseColor.setValue(1,0,0);

    pRoot = new SoSeparator;
    pRoot->addChild(pMatl);
    pRoot->addChild(pCone);
  }

  // Request a callback for mouse clicks
  SoEventCallback *pEvCB = new SoEventCallback;
  pEvCB->addEventCallback(SoMouseButtonEvent::getClassTypeId(),
                           EventCallback, (void*)pRoot);
  pRoot->addChild(pEvCB);

  // Setup and display viewer
  SoXtExaminerViewer *myViewer = 
    new SoXtExaminerViewer(myWindow);
  myViewer->setBackgroundColor(SbColor(0,.75,.75));
  myViewer->setSceneGraph(pRoot);
  myViewer->setTitle("GetNormals - click to start");
  myViewer->setViewing(FALSE);
  myViewer->show();

  // On with the show...
  SoXt::show(myWindow);
  SoXt::mainLoop();
  SoXt::finish();
  return 0;
}


