/**
* @file IvGLUT.cpp
*
* This simple example shows how to use Open Inventor and IvTune with the GLUT.
*
* http://freeglut.sourceforge.net/
*
* Commands :
*  + Pressing <SHIFT><F12>, launches IvTune
*  + Pressing the key '+', performs a zoom in and the key '-' a zoom out.
*  + Pressing the left/right/up/down arrow keys rotates the camera
*  + Pressing the left mouse button and moving the mouse, rotates the camera
*    (but only if the button event was not handled by a node in the scene graph,
*     for example an SoSelection or a dragger)
*  + Pressing the right mouse button, shows a popup menu.
*  + Pressing the space key reset the camera to its original position.
*
* @author Thomas ROBERT / Mike HECK
* @date June 2007 / December 2010
*
* Revised version (December 2010) adds:
* - Headlight rotates with camera (like OIV viewers)
* - Save and restore camera settings
* - Automatic redraw on scene graph change (like OIV viewers)
* - Draggers and manipulators work
*   (some limitations on shift/ctrl key behaviors because of GLUT)
* - SoSelection and selection callbacks work
* - Selection highlighting works
*
* Note that this is not a complete re-implementation of an OIV viewer!
* - Middle mouse button camera manipulations are not implemented.
* - Does not automatically adjust near/far clip planes.
* - Does not enforce separate "viewing" and "selection" modes.
*   Instead events are first sent to the scene graph, where they may be
*   handled by SoSelection, draggers, etc.  If not handled then they are
*   used to rotate the camera.  There are pluses and minuses to both
*   strategies.  Applications will have to decide what they want.
* - Keyboard events are not sent to the scene graph (easy to add).
* - Etc...
*/

// Define this macro to enable using IvTune with a GLUT/OIV application.
// But note that you must have a compatible Qt SDK installed for this to work!
//#define USE_IVTUNE 0

#include <Inventor/SoDB.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/nodekits/SoNodeKit.h>

#include <Inventor/SoSceneManager.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/devices/SoCpuDevice.h>
#include <Inventor/devices/SoGLContext.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoGradientBackground.h> 
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoBoxHighlightRenderAction.h>

#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>

#include <Inventor/manips/SoTransformerManip.h>

#include <Inventor/SoInput.h>

#ifdef USE_IVTUNE
#include <IvTune/SoIvTune.h>
//#include <Qt/qapplication.h>
#endif

#ifdef __APPLE__
#  include <OpenGL/gl.h>
#  include <GLUT/glut.h>
#else
#  include <GL/gl.h>
#  include <GL/glut.h>
#endif

#include <stdio.h>

/* +-----------------------------------------------------------------------+
|                           Global Variables                            |
+-----------------------------------------------------------------------+ */

int width  = 1024;    // window width
int height = 768;    // window height

int posX;            // mouse press x coordinate
int posY;            // mouse press y coordinate

bool mouseLBDown = false;

SoSceneManager      * sceneManager;
SoBoxHighlightRenderAction * renderAction;
SoPerspectiveCamera * camera;
SoRotation          * headlightRot;

SoMouseButtonEvent * gpMouseButtonEvent = NULL;
SoLocation2Event   * gpLocation2Event   = NULL;

const char *DEFAULT_FILENAME = "$OIVHOME/examples/source/Inventor/Techniques/IvGLUT/cubes.iv";

static SbBool     cameraSaved = FALSE;
static SbVec3f    cameraSavedPos;
static SbRotation cameraSavedRot;
static float      cameraSavedFoc;
static float      cameraSavedNear;
static float      cameraSavedFar;

static SoGLContext* m_SoGLContext = NULL;

/* +-----------------------------------------------------------------------+
|                              Prototypes                               |
+-----------------------------------------------------------------------+ */


void initScene(int argc, char ** argv);
void finishScene();
void renderCB(void * userData, SoSceneManager * mgr);
void displayCB();
void reshapeCB(int w, int h);
void keyboardCB(unsigned char ch, int x, int y);
void specialCB(int key, int x, int y);
void mouseCB(int button, int state, int x, int y);
void motionCB(int x, int y);
void visibilityCB(int state);
void menuCB(int value);
void idleCB();

void dollyCamera( int incr );
void rotateCamera( int xincr, int yincr );
void saveCamera();
void restoreCamera();

void selectionCB(void *userData, SoPath *path);
void deselectionCB(void *userData, SoPath *path);

/* +-----------------------------------------------------------------------+
|                                 Main                                  |
+-----------------------------------------------------------------------+ */


/**
* Default: A blue cone in a Glut window with the ability to zoom in/out and
* rotate the camera with the mouse when the left button is hold down.
* IvTune can be launched by pressing Shift+F12.
*/
int main(int argc, char ** argv)
{
  printf( "Press arrow keys to rotate camera.\nPress '+' and '-' keys to dolly camera.\nPress space bar to reset camera.\n" );

  // OpenGL setup
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH | GLUT_MULTISAMPLE);

  glutInitWindowSize(width, height);
  glutCreateWindow("GLUT/OIV Demo");

  glutCreateMenu(menuCB);
  glutAddMenuEntry("Enable Antialiasing", 1);
  glutAttachMenu(GLUT_RIGHT_BUTTON);

  glutDisplayFunc(displayCB);
  glutReshapeFunc(reshapeCB);
  glutKeyboardFunc(keyboardCB);
  glutSpecialFunc(specialCB);
  glutMouseFunc(mouseCB);
  glutMotionFunc(motionCB);
  glutVisibilityFunc(visibilityCB);
  glutIdleFunc(idleCB);

  // Set up Open Inventor stuff
  initScene( argc, argv );

  glutMainLoop();

  // Clean up Open Inventor stuff
  finishScene();
/*
#ifdef USE_IVTUNE
  if (qApp)
    delete qApp;
#endif
*/
  return 0;
}


/* +-----------------------------------------------------------------------+
|                               Functions                               |
+-----------------------------------------------------------------------+ */


/**
* Initializes Open Inventor and create the scene graph.
*/
void initScene(int argc, char ** argv)
{
  // Initialize core Open Inventor (no window system classes).
  // Initialize OIV extensions, e.g. VolumeViz, here if needed.
  SoDB::init();
  SoNodeKit::init();
  SoInteraction::init();

  // Create an OpenGL context wrapper object for Open Inventor rendering.
  // Passing true means we should be using the application's context.
  m_SoGLContext = SoGLContext::getCurrent( true );

  // Build the scene graph

  SoSeparator * sceneGraph = new SoSeparator;
  sceneGraph->setName("ROOT");
  sceneGraph->ref();

  SoGradientBackground* pBack = new SoGradientBackground();
  sceneGraph->addChild(pBack);

  // Camera is a global variable so we can modify it in keypress callback
  camera = new SoPerspectiveCamera();
  camera->setName("CAMERA");
  sceneGraph->addChild(camera);

  // HeadlightGroup - We have a rotation which keeps the headlight
  // moving whenever the camera moves,  and a reset xform so
  // that the rest of the scene is not affected by the first rot.
  SoTransformSeparator *headlightGroup  = new SoTransformSeparator( 2 );
  SoDirectionalLight *headlightNode = new SoDirectionalLight;
  headlightRot = new SoRotation; // Global variable
  headlightGroup->addChild( headlightRot );
  headlightGroup->addChild( headlightNode );
  headlightNode->direction.setValue( SbVec3f(.2f, -.2f, -.9797958971f) );
  sceneGraph->addChild( headlightGroup );

  // The selection node is optional, but can be useful because it does
  // automatic picking and maintains a list of selected geometry (paths).
  SoSelection *pSelection = new SoSelection();
  pSelection->addSelectionCallback  ( selectionCB   );
  pSelection->addDeselectionCallback( deselectionCB );
  sceneGraph->addChild( pSelection );

  // Try to load specified .iv file
  SoSeparator *pScene = NULL;
  SbString filename = (argc > 1) ? (argv[1]) : DEFAULT_FILENAME;
  SoInput input;
  if (input.openFile( filename )) {
    pScene = SoDB::readAll( &input );
    input.closeFile();
  }

  // Else create a scene
  if (! pScene) {
    pScene = new SoSeparator();

    // Create default scene
    SoMaterial *pMatl = new SoMaterial();
    pMatl->diffuseColor.setValue( .1f, .1f, .9f );
    pScene->addChild(pMatl);

    SoTransformerManip *pManip = new SoTransformerManip();
    pScene->addChild(pManip);

    pScene->addChild( new SoCone() );
  }
  pScene->setName( "Scene" );
  pSelection->addChild( pScene );

  // Adjust camera to view the entire scene
  // Then save those settings to allow reset
  camera->viewAll(sceneGraph, SbViewportRegion(width, height));
  saveCamera();

  // Instanciate an SoSceneManager and attach to it a render callback in
  // order to inform Glut that the render area must be refreshed.
  //
  // Note: For automatic redraws to happen, e.g. when the scene graph is
  //       modified, we must (at least) implement a GLUT idle callback
  //       and process the Open Inventor delay queue in that function.
  sceneManager = new SoSceneManager();
  sceneManager->setSceneGraph(sceneGraph);
  sceneManager->setRenderCallback(renderCB);
  sceneManager->getGLRenderAction()->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  sceneManager->activate();
  sceneManager->scheduleRedraw();

  // If we want to automatically highlight selected objects,
  // tell the scene manager to use the appropriate render action.
  renderAction = new SoBoxHighlightRenderAction();
  sceneManager->setGLRenderAction( renderAction );

  // Create some Open Inventor event objects so we can pass GLUT events
  // to the scene graph in the appropriate GLUT callback functions.
  gpMouseButtonEvent = new SoMouseButtonEvent();
  gpLocation2Event   = new SoLocation2Event();
}

/**
* Destroys the scene graph then finalizes Open Inventor.
*/
void finishScene()
{
  sceneManager->getSceneGraph()->unref();
  delete sceneManager;

  delete renderAction;

  delete gpMouseButtonEvent;
  delete gpLocation2Event;

  // Note the ref count on this object is 1 by default.
  m_SoGLContext->unref();

  SoInteraction::finish();
  SoNodeKit::finish();
  SoDB::finish();
}

/** This function is called by SoSelection when a path is
*  added to the selection list.
*/
void selectionCB(void* /*userData*/, SoPath *path)
{
  // Report the selection
  SoNode *pTail = path->getTail();
  printf( "Selected a %s (0x%p)\n",
    pTail->getTypeId().getName().getString(), pTail );

  // Changing the selection list is not considered a change to
  // the scene graph, so we need to explicitly schedule a redraw
  // if we are using a highlight render action.
  sceneManager->scheduleRedraw();
}

/** This function is called by SoSelection when a path is
*  removed from the selection list.
*/
void deselectionCB(void* /*userData*/, SoPath* /*path*/)
{
  // Changing the selection list is not considered a change to
  // the scene graph, so we need to explicitly schedule a redraw
  // if we are using a highlight render action.
  sceneManager->scheduleRedraw();
}

/**
* Rotate the camera around the focal point.
* Approximately same algorithm as the "Examiner" viewer class.
* xincr/yincr are in pixels (how far cursor moved).
*/
void rotateCamera( int xincr, int yincr )
{
  SbRotation rot;
  rot.setValue( (float)(.01 * yincr), (float)(.01 * xincr), 0, 1);

  SbRotation camRot = camera->orientation.getValue();
  float      radius = camera->focalDistance.getValue();

  SbMatrix mat;
  mat = camRot;
  SbVec3f forward( -mat[2][0], -mat[2][1], -mat[2][2]);
  SbVec3f center = camera->position.getValue() + radius * forward;

  camRot = rot * camRot;
  camera->orientation = camRot;
  mat = camRot;

  forward.setValue( -mat[2][0], -mat[2][1], -mat[2][2]);
  camera->position = center - radius * forward;

  // Keep headlight synchronized with camera
  headlightRot->rotation.setValue( camRot );
}

/** Dolly camera closer to or farther away from focal point.
*  Approximately same algorithm as the "Examiner" viewer class.
*  incr is in pixels.
*  Positive value means to move forward (toward the focal point).
*
*  Note: Moving camera closer has no effect with an ortho camera
*        so we simulate the desired result by zooming.
*  Note: The Open Inventor viewer classes automatically update
*        the near and far clip planes when the camera moves.
*        For now we'll just assume that the initial clip planes
*        are valid and move them the same distance (relative to
*        the focal point) that we moved the camera.  This will
*        prevent the scene from being clipped away as we move.
*/
void dollyCamera( int incr )
{
  float d = -incr / 40.0f;

  if ( camera->isOfType(SoOrthographicCamera::getClassTypeId()) ) 
  {
    // change the ortho camera height
    SoOrthographicCamera* cam = (SoOrthographicCamera*)camera;
    cam->height = cam->height.getValue() * (float)pow( 2.0f, d );
  }
  else 
  {
    // shorter/grow the focal distance given the mouse move
    float focalDistance = camera->focalDistance.getValue();
    float newFocalDist = focalDistance * (float)pow( 2.0f, d );

    // finally reposition the camera
    SbMatrix mx;
    mx = camera->orientation.getValue();
    SbVec3f forward( -mx[2][0], -mx[2][1], -mx[2][2] );
    camera->position = camera->position.getValue() + ( focalDistance - newFocalDist ) * forward;
    camera->focalDistance = newFocalDist;

    float delta = newFocalDist - focalDistance;
    camera->nearDistance = camera->nearDistance.getValue() + delta;
    camera->farDistance  = camera->farDistance.getValue() + delta;
  }
}

/** Save current camera position/orientation
*/
void saveCamera()
{
  cameraSaved = TRUE;
  cameraSavedPos  = camera->position.getValue();
  cameraSavedRot  = camera->orientation.getValue();
  cameraSavedFoc  = camera->focalDistance.getValue();
  cameraSavedNear = camera->nearDistance.getValue();
  cameraSavedFar  = camera->farDistance.getValue();
}

/** Restore saved camera position/orientation
*/
void restoreCamera()
{
  if (cameraSaved) {
    camera->position      = cameraSavedPos;
    camera->orientation   = cameraSavedRot;
    camera->focalDistance = cameraSavedFoc;
    camera->nearDistance  = cameraSavedNear;
    camera->farDistance   = cameraSavedFar;

    // Keep headlight synchronized with camera
    headlightRot->rotation.setValue( cameraSavedRot );
  }
}

/**
* Called by SoSceneManager when a redraw is needed.
*
* @param userData
*    the user data,
* @param mgr
*    the SoSceneManager.
*/
void renderCB(void* /*userData*/, SoSceneManager* /*mgr*/)
{
  glutPostRedisplay();
}

/**
* Render the scene.
* GLUT calls this function automatically.
*/
void displayCB()
{

  // Render the Open Inventor scene.
  // Note we need to "bind" the OpenGL context wrapper object to be sure
  // our rendering context is made current (wglMakeCurrent is called).
  // After rendering we unbind to allow other contexts to (possibly) be used.
  m_SoGLContext->bind();
  sceneManager->render();
  m_SoGLContext->unbind();

  // Last step of OpenGL rendering
  glutSwapBuffers();
}

/**
* Resize the render area.
* GLUT calls this automatically if window is resized.
*
* @param w
*    the new width of the render area,
* @param h
*    the new height of the render area.
*/
void reshapeCB(int w, int h)
{
  // Remember new window size
  width = w;
  height = h;

  // Notify Open Inventor scene manager of viewport change
  sceneManager->setViewportRegion(SbViewportRegion(w, h));

  // Re-render
  glutPostRedisplay();
}

/**
* Handle keyboard events.
* GLUT calls this function automatically.
*
* Pressing the key '+' perform a zoom in and the key '-' a zoom out.
*
* @param ch
*    the ascii character of the pressed key,
* @param x
*    the x coordinate of the mouse cursor when the key was hit,
* @param y
*    the y coordinate of the mouse cursor when the key was hit.
*/
void keyboardCB(unsigned char ch, int /*x*/, int /*y*/)
{
  if (ch == '+')         // zoom in
  {
    dollyCamera( 1 );
  }
  else if (ch == '-')    // zoom out
  {
    dollyCamera( -1 );
  }
  else if ( ch == ' ')    // reset camera
  {
    restoreCamera();
  }

  // TODO: Send keyboard events to Open Inventor scene graph
}

/**
* Handles special keyboard keys (F1 .. F2).
* GLUT calls this function automatically.
*
* Launches IvTune when Shift+F12 is hit.
*
* @param key
*    the enum value associated to the key pressed,
* @param x
*    the x coordinate of the mouse cursor when the key was hit,
* @param y
*    the y coordinate of the mouse cursor when the key was hit.
*/
void specialCB(int key, int /*x*/, int /*y*/)
{
#ifdef USE_IVTUNE
  if ((key == GLUT_KEY_F12)
    && (glutGetModifiers() == GLUT_ACTIVE_SHIFT)
    && (!SoIvTune::isInstance()))
  {
    //
    // Instantiates QApplication before launching IvTune in order to prevent
    // IvTune to start its own event loop. Thus we can nested the Qt event loop
    // into the Glut event loop.
    //
/*
    if (!qApp)
      QApplication * app = new QApplication(__argc, __argv);
	  */
    SoIvTune::setInstance(0, 0, sceneManager->getSceneGraph());
    return;
  }
#endif //USE_IVTUNE

  // Spin camera around focal point
  if (key == GLUT_KEY_LEFT) {
    rotateCamera(  1, 0 );
  }
  else if (key == GLUT_KEY_RIGHT) {
    rotateCamera( -1, 0 );
  }
  else if (key == GLUT_KEY_DOWN) {
    rotateCamera(  0, -1 );
  }
  else if (key == GLUT_KEY_UP) {
    rotateCamera(  0,  1 );
  }
  else if (key == GLUT_KEY_PAGE_UP) {
    dollyCamera( 1 );
  }
  else if (key == GLUT_KEY_PAGE_DOWN) {
    dollyCamera( -1 );
  }

  // TODO: Send keyboard events to Open Inventor scene graph
}

/**
* Handle mouse press events and mouse release events.
* GLUT calls this function automatically. 
*
* @param button
*    the mouse button (GLUT_LEFT_BUTTON, GLUT_RIGHT_BUTTON or GLUT_MIDDLE),
* @param state
*    specifies whether the button was pressed or released,
* @param x
*    the x coordinate of the mouse cursor,
* @param y
*    the y coordinatet of the mouse cursor.
*/
void mouseCB(int button, int state, int x, int y)
{
  if (button == GLUT_LEFT_BUTTON)
  {
    // Convert event to equivalent SoEvent object
    gpMouseButtonEvent->setButton( SoMouseButtonEvent::BUTTON1 );

    int oiv_y = (height - 1) - y; // Glut seems to return native (inverted) Y values
    gpMouseButtonEvent->setPosition( SbVec2s(x,oiv_y) );

    gpMouseButtonEvent->setState( 
      (state == GLUT_DOWN) ? SoMouseButtonEvent::DOWN : SoMouseButtonEvent::UP );

    gpMouseButtonEvent->setShiftDown( glutGetModifiers() & GLUT_ACTIVE_SHIFT );
    gpMouseButtonEvent->setCtrlDown ( glutGetModifiers() & GLUT_ACTIVE_CTRL  );

    // Allow OIV nodes (SoSelection, draggers, etc) to handle event
    SbBool wasHandled = sceneManager->processEvent( gpMouseButtonEvent );

    // If the event was NOT handled by a node in the scene graph (e.g.
    // a dragger or manipulator) then we'll assume the user wants to
    // rotate the camera.  mouseLBDown indicates we're rotating the camera.
    //
    // NOTE: This is not the same behavior as an Open Inventor viewer.
    // The viewer classes enforce separate viewing and selection modes
    // (toggled by ESC key).  There are advantages and disadvantages to
    // both approaches...
    if (! wasHandled) {
      if (state == GLUT_DOWN) {
        mouseLBDown = true;
        posX = x;
        posY = y;
      }
      else if (state == GLUT_UP){
        mouseLBDown = false;
      }
    }
  }
}

/**
* Handle mouse move events.
* GLUT calls this function automatically.
*
* If the left mouse button is pressed, this callback performs a rotation of
* the camera.
*
* @param x
*    the x coordinate of the mouse cursor,
* @param y
*    the y coordinate of the mouse cursor.
*/
void motionCB(int x, int y)
{
  // Convert event to equivalent SoEvent object
  int oiv_y = (height - 1) - y; // Glut seems to return native (inverted) Y values
  gpLocation2Event->setPosition( SbVec2s(x,oiv_y) );

  // Oops, GLUT does not currently allow this. :-(
  // Calling glutGetModifiers here causes a warning message from GLUT.
  // Implies constrained dragger behaviors will only work if the
  // shift/control key is already pressed when mouse button is pressed.
  // gpLocation2Event->setShiftDown( glutGetModifiers() & GLUT_ACTIVE_SHIFT );
  // gpLocation2Event->setCtrlDown ( glutGetModifiers() & GLUT_ACTIVE_CTRL  );

  // Allow OIV nodes (SoSelection, draggers, etc) to handle event
  sceneManager->processEvent( gpLocation2Event );

  // If events are not being handled by a node, rotate the camera
  if (mouseLBDown) {
    rotateCamera(posX - x, posY - y);
    posX = x;
    posY = y;
  }
}

/**
* Handles window visibility change events.
* GLUT calls this function automatically.
*
* Deactivates the render callback when the window is hidden or iconified.
*
* @param state
*    specifies whether the window is visible or not.
*/
void visibilityCB(int state)
{
  if (state == GLUT_NOT_VISIBLE)
    sceneManager->deactivate();
  else
    sceneManager->activate();
}

/**
* Handles menu events.
* GLUT calls this function automatically.
*
* @param value
*    the value associated to the selected menu item.
*/
void menuCB(int value)
{
  switch (value) {
    case 1:    // enable antialiasing
      sceneManager->setAntialiasing( 0.5f );
      glutChangeToMenuEntry(1, "Disable Antialiasing", 2);
      break;
    case 2:    // disable antialiasing
      sceneManager->setAntialiasing( 0.0f );
      glutChangeToMenuEntry(1, "Enable Antialiasing", 1);
      break;
  }

  sceneManager->scheduleRedraw();
}

/**
* This callback is performed when there are no other actions to perform.
* GLUT calls this function automatically.
*/
void idleCB()
{
  // Process OIV delay queue so scheduled redraws will happen.
  // If a redraw was requested inside Open Inventor (typically because a node
  // has been modified), a delay queue sensor was scheduled and will be
  // triggered by this call, resulting in a call to the sceneManager's callback,
  // which we implemented to call GLUT's request redraw method.
  SoDB::getSensorManager()->processDelayQueue( TRUE );

#ifdef USE_IVTUNE
  // Since the QApplication event loop is not started, the processEvents()
  // method must be called explicitly. This enables to nest the Qt event loop
  // into the Glut event loop.
  //
  // Note: Previously called processOneEvent(), but that method is no longer
  //       supported in Qt.
/*  if (qApp)
    qApp->processEvents();*/
#endif

}




