// Simple Open Inventor program
//
// Copyright (C) 1996-1998 Template Graphics Software, Inc.

// Overriding the Examiner viewer's createViewerButtons method.
//
// There is a bit of a trick to this because if you simply subclass
// SoXtExaminerViewer and override the method it will not get called...
//
// The reason (I believe) is that the SoXtExaminerViewer's constructor
// is (necessarily) called before ours and the createViewerButtons
// method is called as part of the execution of that constructor.  At
// that point our subclass object has not been constructed yet, so the
// call goes to the SoXtExaminerViewer's method.
//
// The trick lies in realizing that createViewerButtons is called as
// part of the "buildWidget" operation which can be deferred until the
// subclass is ready for it (see ToolMaker chapter 10) by passing an
// additional FALSE parameter to the parent class constructor.  Once
// our subclass' constructor is called we know our object exists and we
// can go ahead and invoke the buildWidget method.
//
// mmh 1-Jul-96
//
// Massively expanded to include:
//  - Overriding the viewer wheel strings
//    (there's a bit of trick to the right wheel string -- see below)
//  - Eliminating the zoom slider
//  - Eliminating one of the standard push buttons
//
// mmh 22-Jun-98

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

// Need this header file to create buttons
// (not included in Inventor SDK version 2.4, included with this example)
#include "SoWinBitmapButton.h"

// Window property constants are defined in this header
#include <Inventor/Win/viewers/SoWinViewerResource.h>

// Standard bitmap resource ids are defined in this header
#include <Inventor/Win/SoWinRes.h>

// Custom bitmap resource ids for this example are in this header
#include "resource.h"


//////////////////////////////////////////////////////////////////////
// Define a very simple custom viewer class
//
// Note that we copied the declaration of the constructor from the
// parent class (SoXtExaminerViewer) and we override processEvent.

const int MAX_BUTTONS = 10;

class MyViewer : public SoXtExaminerViewer
{
public:
  MyViewer(Widget parent = NULL, 
           const char *name = NULL, 
           SbBool buildInsideParent = TRUE, 
           SoXtFullViewer::BuildFlag flag = BUILD_ALL, 
           SoXtViewer::Type type = BROWSER);
  ~MyViewer();

protected:
  virtual Widget buildZoomSlider(Widget parent);
  virtual void   createViewerButtons(Widget parent);
  virtual void   pushButtonCB(Widget, int id, void *);
  virtual void   setCamera(SoCamera *cam);
  virtual void   setViewing(SbBool onOrOff);

private:
  // Because this is private in our parent class, we will need our own
  SoWinBitmapButton  *buttonList[MAX_BUTTONS];
};

// List of standard FullViewer push buttons
enum ViewerPushButtons {
  PICK_PUSH, 
  VIEW_PUSH, 
  HELP_PUSH, 
  HOME_PUSH, 
  SET_HOME_PUSH, 
  VIEW_ALL_PUSH, 
  SEEK_PUSH, 
  PUSH_NUM
};

// Custom push buttons for this example
const int XVIEW_PUSH = PUSH_NUM + 1;
const int YVIEW_PUSH = PUSH_NUM + 2;

//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////
// Custom viewer constructor.
//
// Note: We have to use this special syntax to make sure our parent
//       class constructor is called with the correct parameters.
//       Otherwise it would be called with all default param values.
//
// Standard version that does not work:
//
// MyViewer::MyViewer(Widget parent, 
//                    const char *name, 
//                    SbBool buildInsideParent, 
//                    SoXtFullViewer::BuildFlag flag, 
//                    SoXtViewer::Type type)
//          :SoXtExaminerViewer(parent, 
//                              name, 
//                              buildInsideParent, 
//                              flag, 
//                              type)
// {
// }
// Custom version that works...
//
MyViewer::MyViewer(Widget parent, 
                   const char *name, 
                   SbBool buildInsideParent, 
                   SoXtFullViewer::BuildFlag flag, 
                   SoXtViewer::Type type)
  :SoXtExaminerViewer(parent, 
                      name, 
                      buildInsideParent, 
                      flag, 
                      type, 
                      FALSE) // <== Delay building until we're ready!
{
  // Just to be safe, initialize the buttonList array
  for (int i = 0; i < MAX_BUTTONS; i++)
    buttonList[i] = NULL;

  // Now build the widget tree, 
  // and let SoWinComponent know about our base widget
  //
  // Note: Delay building until now because we don't want the
  //       createViewerButtons method to be called until our subclass
  //       object has been constructed and can intercept the call.

  Widget w = buildWidget(getParentWidget());
  setBaseWidget(w);

  // Reset the wheel labels
  // Do NOT assign static strings to the "leftWheelStr" etc member vars.
  // SoWin will attempt to free this memory in the viewer's destructor!!
  //
  // Note: Changing the rightWheelString under an Examiner viewer is
  //       problematic because SoWinExaminerViewer::setCamera sets this
  //       string depending on the camera type and setRightWheelString is
  //       not virtual.  We'll have to override setCamera... (see below).
  setLeftWheelString("Huey");
  setBottomWheelString("Dewey");
  setRightWheelString("Louie");
}

// Destructor -- Need to free memory alloc'd for "buttonList" because
//               this is our own private member variable.
//
// Note: SoWinFullViewer will free the wheel label strings and
//       the viewerButtonWidgets list.
//
MyViewer::~MyViewer()
{
  for (int i = 0; i < MAX_BUTTONS; i++) {
    if (buttonList[i] != NULL) {
      delete buttonList[i];
    }
  }
}

// Override of SoWinFullViewer::buildZoomSlider
//
Widget MyViewer::buildZoomSlider(Widget parent)
{
  // We don't want to see the zoom slider at all
  return NULL;
}

// Override of SoWinExaminerViewer::createViewerButtons
//
void MyViewer::createViewerButtons(Widget parent)
{
  // Strategies:
  //
  // 1) No buttons  - Just return from this method without doing
  //    anything and no buttons will be created.
  //
  // 2) Std buttons - Just call the parent viewer class method:
  //
  //        SoWinExaminerViewer::createViewerButtons(parent);
  //
  //    There is no reason to override this method just to do that, 
  //    but it is useful to try that first to confirm that the
  //    override is working correctly.
  //
  // 3) FullViewer buttons only - Pretty obvious, just call:
  //
  //        SoWinFullViewer::createViewerButtons(parent);
  //
  // 4) Subset of buttons - A little trickier.  We will have to
  //    duplicate the code from FullViewer that creates the std
  //    buttons (in order to leave some out), but the button id is
  //    stored with the button's window, so the remaining buttons
  //    will still do the right thing.  Suppose we want to leave out
  //    the "?" (Help) button -- just don't create the third button
  //    (id = 2).  See pushButtonCB below for list of id numbers.
  //
  //    Note: FullViewer::setViewing automatically updates the state
  //    of the Pick (arrow) and View (hand) buttons (if they exist)
  //    when the viewer changes between Pick and View modes as a
  //    result of a program call or the user pressing the ESC key.
  //    However, since the "buttonList" array is declared private, 
  //    FullViewer::setViewing won't think these buttons exist.  We
  //    will have to override setViewing and do the update ourselves.
  //
  //    Note: The maximum number of buttons is determined by the
  //    length of the "buttonList" array (10 in this example).
  //    "viewerButtonWidgets" is an SbPList so it expands as needed.
  //
  // 5) Create new buttons - Not much trickier than #4, but we will
  //    need one bitmap resource (24x24 pixels monochrome) for the
  //    image for each new button.  Note SoWinBitmapButton creates
  //    a "down" image automatically.  Open INVR240.DLL in DevStudio
  //    to see the standard button images.

  // Create the standard FullViewer buttons
  // Note: PUSH_NUM is an enum defined at top of this file
  int i;
  for (i = 0; i < PUSH_NUM; i++) {
    // ***** Example of strategy #4 *****
    // Skip creating the standard Inventor "?" (Help) button
    if (i == 2)
      continue;
    
    // Second arg to constructor is boolean "selectable", 
    // meaning whether button should lock down when clicked.
    // In the standard buttons, only true for Pick and View.
    buttonList[i] = new SoWinBitmapButton(parent, (i == 0 || i == 1));
    HWND w = buttonList[i]->getWidget();

    // The viewer buttons do not draw differently when they have focus
    buttonList[i]->showFocus(FALSE);

    // Store ptr to this SoWinFullViewer instance for use in window proc
    // (essentially same thing as what Motif code does with XmNuserData).
    // Also store info X handles with XtAddCallback.
    SetProp(w, PROP_THIS, (HANDLE)this);
    SetProp(w, PROP_CALLDATA, (HANDLE)(uintptr_t)i);

    // For now...
    // Also store a ptr to this instance of SoWinBitmapButton, so
    // we can call the correct draw function in btnWindowProc.
    SetProp(w, "SoWinBitmapButton", (HANDLE)buttonList[i]);

    // add this button to the list...
    viewerButtonWidgets->append(w);
  }

  // Remember how many buttons are the standard FullVwr buttons
  numFullVwrButtons = viewerButtonWidgets->getLength();

  // Set the standard button images
  // Note: For _WIN32 the bitmaps are stored in the SoWin resource DLL.
  //       This module must be in the path at runtime to get bitmaps.
  HMODULE hResDll = (HMODULE)SoWin::getResDllSafeHandle().getHandle();

  if (hResDll != NULL) {
    buttonList[PICK_PUSH]->setBitmap(hResDll, IDB_PUSH_PICK);
    buttonList[VIEW_PUSH]->setBitmap(hResDll, IDB_PUSH_VIEW);
    buttonList[HOME_PUSH]->setBitmap(hResDll, IDB_PUSH_HOME);
    buttonList[SET_HOME_PUSH]->setBitmap(hResDll, IDB_PUSH_SETHOME);
    buttonList[VIEW_ALL_PUSH]->setBitmap(hResDll, IDB_PUSH_VIEWALL);
    buttonList[SEEK_PUSH]->setBitmap(hResDll, IDB_PUSH_SEEK);

    // ***** Example ***** We're not creating the std help button
    //buttonList[HELP_PUSH]->setBitmap    (hResDll, IDB_PUSH_HELP);
  }

  // Show the pick/view state
  if (isViewing())
    buttonList[VIEW_PUSH]->select(TRUE);
  else
    buttonList[PICK_PUSH]->select(TRUE);

  // ***** Example of strategy #5 *****
  // Add our own custom "X View" and "Y View" buttons
  for (i = XVIEW_PUSH; i <= YVIEW_PUSH; i++) {
    buttonList[i] = new SoWinBitmapButton(parent, FALSE);
    HWND w = buttonList[i]->getWidget();
    buttonList[i]->showFocus(FALSE);
    SetProp(w, PROP_THIS, (HANDLE)this);
    SetProp(w, PROP_CALLDATA, (HANDLE)(uintptr_t)i);
    SetProp(w, "SoWinBitmapButton", (HANDLE)buttonList[i]);
    viewerButtonWidgets->append(w);
  }
  hResDll = GetModuleHandle(NULL);  // Want application's module handle
  buttonList[XVIEW_PUSH]->setBitmap(hResDll, IDB_PUSH_XVIEW);
  buttonList[YVIEW_PUSH]->setBitmap(hResDll, IDB_PUSH_YVIEW);
}

// Override of SoWinExaminerViewer::pushButtonCB
//
void MyViewer::pushButtonCB(Widget w, int id, void *)
{
  // If the button id falls in the range of standard FullViewer
  // buttons then kick it upstairs to the standard handler.
  //
  // If you just need to have one of your buttons invoke one of
  // the standard FullViewer behaviors, call the standard handler
  // with one of the following ids (of course these are public or
  // protected methods anyway, but it might be useful to know how
  // the buttons are implemented):
  //
  // id  0: setViewing(FALSE)
  //     1: setViewing(TRUE)
  //     2: openViewerHelpCard()
  //     3: resetToHomePosition()
  //     4: saveHomePosition()
  //     5: viewAll()
  //     6: setSeekMode(!isSeekMode())

  if (id <= 6) {  // Note this is not hard coded in SoWin!
    SoWinFullViewer::pushButtonCB(w, id, NULL);
    return;
  }

  // Else it must be the button appended by ExaminerViewer:
  //
  // id  7: toggleCameraType()
  //
  else if (id == 7)
    SoWinExaminerViewer::pushButtonCB(w, id, NULL);

  // Else maybe we have some push buttons of our own...
  else if (id == XVIEW_PUSH) {
    // Change to a view from the +X axis
    if (camera != NULL) {
      camera->orientation.setValue(SbVec3f(0, 1, 0), 1.570796326795f);
      viewAll();
    }
  }
  else if (id == YVIEW_PUSH) {
    // Change to a view from the +Y axis
    if (camera != NULL) {
      camera->orientation.setValue(SbVec3f(-1, 0, 0), 1.570796326795f);
      viewAll();
    }
  }
}

// Override of SoWinExaminerViewer::setCamera
//
void MyViewer::setCamera(SoCamera *cam)
{
  // We don't want to mess with the camera
  SoWinExaminerViewer::setCamera(cam);

  // We just want to prevent setCamera from changing this string!
  setRightWheelString("Louie");
}

// Override of SoWinExaminerViewer::setViewing
//
void MyViewer::setViewing(SbBool onOrOff)
{
  // Call parent class to do all the usual stuff
  SoWinExaminerViewer::setViewing(onOrOff);

  // Update the push buttons
  //
  // Note: We have to do this because "buttonList" is a private
  // member in our parent classes (I don't actually know why :-).
  if (buttonList[VIEW_PUSH])
    buttonList[VIEW_PUSH]->select(viewingFlag);
  if (buttonList[PICK_PUSH])
    buttonList[PICK_PUSH]->select(!viewingFlag);
}

//////////////////////////////////////////////////////////////////////
// Trivial Inventor program (build as console app)
//
void
main(int, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) 
    exit(1);

  SoSeparator *root = new SoSeparator;
  root->ref();
  SoMaterial *myMaterial = new SoMaterial;
  myMaterial->diffuseColor.setValue(1.0, 0.0, 0.0);
  root->addChild(myMaterial);
  root->addChild(new SoCone);

  // Create custom viewer
  MyViewer *myViewer = new MyViewer(myWindow);

  myViewer->setSceneGraph(root);
  myViewer->setTitle("Examiner Viewer");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  SoXt::finish();
}


