/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : MMH (MMM YYYY)
**=======================================================================*/

// Simple wellbore rendering using Open Inventor extrusion
//
// Notes:
// 1) Requires Open Inventor 4.0.6 or higher.
// 2) Requires separate data and colormap files.

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoTranslation.h>

#include <Inventor/nodes/SoExtrusion.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <math.h> // for M_PI



/////////////////////////////////////////////////////////////
//
// Class to store a simple "well bore" data set
//
// TODO: Data members should be private

class WellBore {

public:
  WellBore () : nPoints(0), points(NULL), dataMin(0),
                dataMax(1), data0(NULL), data1(NULL) {}

  ~WellBore()
    {
      delete points;
      delete data0;
      delete data1;
    }

  int nPoints;      // Number of data points
  SbVec3f *points;  // 3D coordinate of each data point
  float    dataMin; // Smallest data value
  float    dataMax; // Largest data value
  float   *data0;   // First set of data values
  float   *data1;   // (Optional) Second set of data values
};

/////////////////////////////////////////////////////////////
//
// Class to store a simple color map
//
// TODO: Data members should be private

class ColorMap {

public:
  ColorMap () : nColors(0), colors(NULL) {}

  ~ColorMap()
    { delete colors; }

  int   nColors;    // Number of color values in map
  SbColor *colors;  // Color values
};

/////////////////////////////////////////////////////////////
//
// Read a "well bore" data set from a simple file format
//
// Careful... this code is fragile...  only an example!

WellBore *
readWellBore(const char *filename)
{
  WellBore *pWell = new WellBore;

  FILE *fp = fopen(filename, "r");
  if (fp == NULL) {
    fprintf(stderr, "*** Unable to open data file '%s'.\n", filename);
    return NULL;
  }

  int nPoints = 0;

  const int BUFLEN = 256;
  char buffer[BUFLEN];
  char *p;

  while ((p = fgets(buffer, BUFLEN, fp))) {
    while (*p == ' ') p++; // skip whitespace
    if (*p == '#' || *p == '\0' || *p == '\n') // Comment or end-of-line
      continue;
    if (! strncmp(p, "nVertices", 9)) {
      p += 9;
      sscanf(p, "%d", &nPoints);
    }
    else if (! strncmp(p, "Coords", 6)) {
      pWell->nPoints = nPoints;
      pWell->points  = new SbVec3f[nPoints];
      for (int i = 0; i < nPoints; ++i) {
        float x, y, z;
        p = fgets(buffer, BUFLEN, fp);
        sscanf(p, "%g %g %g", &x, &y, &z);
        pWell->points[i].setValue(x, y, z);
      }
    }
    else if (! strncmp(p, "Data", 4)) {
      pWell->data0 = new float[nPoints];
      for (int i = 0; i < nPoints; ++i) {
        float val;
        p = fgets(buffer, BUFLEN, fp);
        sscanf(p, "%g", &val);
        pWell->data0[i] = val;
      }
    }
  }

  // We happen to know the range for the data in this example
  pWell->dataMin = 0;
  pWell->dataMax = 255;

  fclose(fp);
  printf("--- Loaded data file '%s': %d points.\n",
    filename, nPoints);

  return pWell;
}

/////////////////////////////////////////////////////////////
//
// Careful... this code is fragile...

ColorMap *
readColorMap(const char *filename)
{
  FILE *fp = fopen(filename, "r");
  if (fp == NULL) {
    fprintf(stderr, "*** Unable to open colormap file '%s'.\n", filename);
    return NULL;
  }

  ColorMap *pMap = NULL;

  const int BUFLEN = 256;
  char buffer[BUFLEN];
  char *p;

  while ((p = fgets(buffer, BUFLEN, fp))) {
    while (*p == ' ') p++; // skip whitespace
    if (*p == '#' || *p == '\0' || *p == '\n') // Comment or end-of-line
      continue;
    if (! strncmp(p, "nColors", 7)) {
      p += 7;
      int nColors;
      sscanf(p, "%d", &nColors);

      pMap = new ColorMap;
      pMap->nColors = nColors;
      pMap->colors = new SbColor[nColors];

      for (int i = 0; i < nColors; ++i) {
        float r, g, b;
        p = fgets(buffer, BUFLEN, fp);
        sscanf(p, "%g %g %g", &r, &g, &b);
        pMap->colors[i].setValue(r, g, b);
      }
    }
  }

  fclose(fp);
  printf("--- Loaded colormap file '%s': %d colors.\n",
    filename, (pMap ? pMap->nColors : 0));

  return pMap;
}

/////////////////////////////////////////////////////////////

SoSeparator *
makeExtrusion(WellBore *pWell, 
              ColorMap *pMap, 
              int   numSides = 12,
              float minValForColor = -1,
              float maxValForColor = -1,
              float scaleFactor = 1)
{
  // Group the attribute nodes and extrusion
  SoSeparator *pSep = new SoSeparator;

  // Extrusion will be considered "solid" to enable back-face culling.
  // Also set crease angle to "smooth" surface for more than 4 sides.
  SoShapeHints *pHints = new SoShapeHints;
  pHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  pHints->shapeType      = SoShapeHints::SOLID;
  pHints->creaseAngle    = (float)(M_PI / 2.1);
  pSep->addChild(pHints);

  SoExtrusion *pExt = new SoExtrusion;

  // Cross section (prescaled to diameter=1 to allow meaningful scaling)
  // 16 sides makes a reasonable cylinder (with smoothing).
  //  4 sides is useful for checking twist or to improve performance.

  // Compute x and z coordinates around circle
  if (numSides < 2)
    numSides = 2;
  int   side;
  float theta  = 0.0f;
  float dTheta = (float)(2.0 * M_PI / (double)numSides);
  const float eps = 1e-6f;

  pExt->crossSection.setNum(numSides + 1);

  for (side = 0; side < numSides; side++) {
    float x = 0.5f * sin(theta);
    float z = 0.5f * cos(theta);
    if (fabs(x) < eps) x = 0;
    if (fabs(z) < eps) z = 0;
    pExt->crossSection.set1Value(side, SbVec2f(x,z));
    theta += dTheta;
  }
  pExt->crossSection.set1Value(numSides, SbVec2f(0,0.5f)); // close loop

  // Coordinates of well bore define the spine
  pExt->spine.setValues(0, pWell->nPoints, pWell->points);

  // Get data range
  int   nPoints = pWell->nPoints;

  // First data set maps to color
  // (requires new SoExtrusion node)
  if (pWell->data0 && pMap) {

    // Assign color value to each spine point
    SoMaterialBinding *pBind = new SoMaterialBinding;
    pBind->value = SoMaterialBinding::PER_VERTEX;

    // Define colors (one per spine vertex)
    SoMaterial *pMat = new SoMaterial;
    pMat->diffuseColor.setNum(nPoints);

    // Number of colors in map
    int nColors = pMap->nColors;

    for (int i = 0; i < nPoints; ++i) {
      float data = pWell->data0[i];
      SbColor color;
      if (data <= minValForColor)
        color = pMap->colors[0];
      else if (data >= maxValForColor)
        color = pMap->colors[pMap->nColors - 1];
      else {
        float t = (data - minValForColor) / (maxValForColor - minValForColor);
        t *= nColors;
        int index = (int)(t + 0.5);
        color = pMap->colors[index];
      }
      pMat->diffuseColor.set1Value(i, color);
    }

    pSep->addChild(pBind);
    pSep->addChild(pMat );
  }

  // Second data set maps to radius (defines scale factor)
  if (pWell->data1) {

    pExt->scale.setNum(pWell->nPoints);

    for (int i = 0; i < pWell->nPoints; ++i) {
      float scale = pWell->data1[i];

      // Apply global scale factor
      scale *= scaleFactor;
      pExt->scale.set1Value(i, SbVec2f(scale,scale));
    }
  }
  // Else just use global scale factor for all points
  else {
    pExt->scale.setValue(SbVec2f(scaleFactor, scaleFactor));
  }

  pSep->addChild(pExt);
  return pSep;
}

/////////////////////////////////////////////////////////////

#ifdef _WIN32
SbBool 
RawEventCB(void *userData, MSG *msg)
{
  char buffer[256];
  
  if (msg->message == WM_KEYDOWN) {
    int key = (int)(msg->wParam);
    sprintf(buffer, "WM_KEYDOWN: %c (%d)\n", key, key);
    OutputDebugString(buffer);
    
    SoNode *scene = ((SoXtViewer*)userData)->getSceneGraph();
  }
  return FALSE;
}
#endif

#include <Inventor/actions/SoWriteAction.h>

/////////////////////////////////////////////////////////////

int
main(int /*argc*/, char **argv)
{
  // Initialize Inventor
  Widget myWindow = SoXt::init(argv[0]);

  // Lower epsilon value to avoid "twisting" artifacts
  SoPreferences::setFloat("OIV_EXTRUSION_EPSILON", 0.99f);

  // Try to read the files
  SbString DataDir0= SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Techniques/WellBore/data0.txt");
  SbString DataDir1= SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Techniques/WellBore/data1.txt");
  SbString colormapDir= SbFileHelper::expandString("$OIVHOME/examples/source/Inventor/Techniques/WellBore/colormap.txt");

  WellBore *pWell0 = readWellBore(DataDir0.toLatin1());
  WellBore *pWell1 = readWellBore(DataDir1.toLatin1());
  ColorMap *pCMap  = readColorMap(colormapDir.toLatin1());

  if (!pWell0 || !pWell1 || !pCMap) {
    fprintf(stderr, "Exiting -- unable to load data or colormap\n");
    return 0;
  }

  // Simulate having a second data set on the second bore
  pWell1->data1 = new float[pWell1->nPoints];
  memcpy(pWell1->data1, pWell1->data0, pWell1->nPoints * sizeof(float));

  // Create scene graph
  SoSeparator *pRoot = new SoSeparator;
  pRoot->ref();

  // We happen to know the bounds of the data volume these
  // well bores are associated with...
  float xmax = 300;
  float ymax = 455;
  float zmax =  91;

  // Simple bounding box
  {
    SoTranslation *pTrans = new SoTranslation;
    pTrans->translation.setValue(xmax/2, ymax/2, zmax/2);

    SoDrawStyle *pStyle = new SoDrawStyle;
    pStyle->style = SoDrawStyle::LINES;

    SoLightModel *pModel = new SoLightModel;
    pModel->model = SoLightModel::BASE_COLOR;

    SoCube *pCube = new SoCube;
    pCube->width  = xmax;
    pCube->height = ymax;
    pCube->depth  = zmax;

    SoSeparator *pBoxSep = new SoSeparator;
    pBoxSep->addChild(pTrans);
    pBoxSep->addChild(pStyle);
    pBoxSep->addChild(pModel);
    pBoxSep->addChild(pCube );

  pRoot->addChild(pBoxSep);
  }

  // Create well bore visualizations
  SoSeparator *pBoreSep;
  
  // Number of sides (12-16 gives a reasonable looking cylinder)
  const int   numSides = 12;

  // Range of data values that color map actually applies to
  const float dataMin  = 125;
  const float dataMax  = 200;

  // First bore only has one data set, so default bore will have
  // diameter of 1.  Scale it by 8 to be more visible.
  pBoreSep = makeExtrusion(pWell0, pCMap, numSides, dataMin, dataMax, 8);
  pRoot->addChild(pBoreSep);

  // Second bore has two data sets (actually both contain same values),
  // so bore diameter will be taken from second data set.  Scale it
  // down to be approximately in scale with the first bore.
  pBoreSep = makeExtrusion(pWell1, pCMap, numSides, dataMin, dataMax, 0.08f);
  pRoot->addChild(pBoreSep);

  // Create a viewer and set attributes
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setBackgroundColor(SbColor(0,.75,.75));
  myViewer->setSceneGraph(pRoot);
  myViewer->setTitle("WellBore");
  myViewer->show();

  // Loop forever
  SoXt::show(myWindow);
  SoXt::mainLoop();
  SoXt::finish();
  return 0;
}


