/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
//
// Main stuff for 3D text creator program
//
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/SoDB.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLinearProfile.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoProfileCoordinate2.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/sensors/SoSensor.h>
#include <Inventor/sys/SoGL.h>

#include <Xm/Xm.h>
#include <Xm/Text.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/PushBG.h>

#include "../common/InventorLogo.h"
#include "../common/LineManip.h"
#include "./TextGraph.h"
#include "../common/Useful.h"

//
// Some evil variables global to this file.  I should pass them around
// as paramaters or encapsulate them in a class, but this is easier.
// Especially since so much occurs in response to Xt callbacks, and
// passing arguments to a callback function means wrapping them all up
// in a structure and passing a pointer to that structure (which is a
// pain!).
//
static const char *DefaultString =
"Open Inventor -\n3D Programming\nFor Humans";
static LineManip2 *manip;

//
// This is called whenever the profile's coordinates change.  It has
// to change the profileCoordinate node which defines the text's
// extrusion.
//
static void
profileCallback(void *data, SoSensor *)
{
  SoCoordinate3 *coord = (SoCoordinate3 *)data;

  int n = coord->point.getNum();

  if (n == 0) return;

  SbVec2f *newV = new SbVec2f[n];
  for (int i = 0; i < n; i++) {
    newV[i].setValue(coord->point[i][0], coord->point[i][1]);
    profile->index.set1Value(i, i);
  }
  profileCoords->point.setValues(0, n, newV);
  profileCoords->point.deleteValues(n);  // Get rid of any extra
  profile->index.deleteValues(n);

  delete[] newV;
}
//
// Callback for 'About...' button
//
void
showAboutDialog(Widget, XtPointer, XtPointer)
{
  if (access("/usr/demos/Inventor/textomatic.about", R_OK) != 0)
    {
      system("xconfirm -t 'Sorry, could not find "
	     "/usr/demos/Inventor/textomatic.about' > /dev/null");
      return;
    }

  char command[100];
  sprintf(command, "showcase -v /usr/demos/Inventor/textomatic.about");

  int err = system(command);
  if (err) {
    system("xconfirm -t 'You must install showcase"
	   " for this function to work' > /dev/null");
    return;
  }
}	

//
// Called by the quit button
//
static void
quitCallback(Widget, XtPointer, XtPointer)
{
  exit(0);
}

void
enterProfileWindow(Widget, XtPointer, XEvent *, Boolean *cont)
{
  *cont = TRUE;	// Continue dispatching this event
}
void
leaveProfileWindow(Widget, XtPointer, XEvent *, Boolean *cont)
{
  manip->removeHilights();

  *cont = TRUE;	// Continue dispatching this event
}

//
// The default profile
//
static float
defaultCoords[2][3] =
{
  { -0.2F, 0.0F, 0.0F },
  {  0.2F, 0.0F, 0.0F },
};

//
// Grid for window:
//
static const char *GridString =
"#Inventor V2.0 ascii\n"
"Separator {"
"	PickStyle { style UNPICKABLE }"
"	LightModel { model BASE_COLOR }"
"	BaseColor { rgb 0.2 0.2 0.2 }"
"	Array {"
"		numElements1 7"
"		separation1 .1 0 0"
"		origin CENTER"
"		Coordinate3 { point [ 0 -1 0, 0 1 0 ] }"
"		LineSet { numVertices [ 2 ] }"
"	}"
"	Array {"
"		numElements1 7"
"		separation1 0 .1 0"
"		origin CENTER"
"		Coordinate3 { point [ -1 0 0, 1 0 0 ] }"
"		LineSet { numVertices [ 2 ] }"
"	}"
"	BaseColor { rgb 0.4 0.0 0.0 }"
"	Coordinate3 { point [ -1.0 0.0 0.0, 1.0 0.0 0.0,"
"			0.0 -1.0 0.0, 0.0 1.0 0.0 ]"
"	}"
"	LineSet { numVertices [ 2, 2 ] }"
"	BaseColor { rgb 0.3 0.3 0.3 }"
"	Transform {"
"		scaleFactor 0.025 0.025 0.025"
"		translation 0.0 -0.06 0.0"
"	}"
"	Font {"
"		size 2"
"		name \"Helvetica\""
"	}"
"	Array {"
"		numElements1 2"
"		separation1 20 0 0"
"		origin CENTER"
"		DEF switch Switch {"
"			whichChild -2"
"		Text3 { string \"Front\" justification LEFT parts ALL }"
"		Text3 { string \"Back\" justification RIGHT parts ALL }"
"		}"
"	}"
"}";

//
// Routine that creates the scene graph in the render area at the
// bottom of the window (where the text extrusion profile is defined).
//
SoSeparator *
createProfileSceneGraph()
{
  SoSeparator *root = new SoSeparator;
  root->ref();

  SoOrthographicCamera *c = new SoOrthographicCamera;
  c->nearDistance = 0.5F;
  c->height = 0.4F;
  c->aspectRatio = 4.0F/3.0F;
  root->addChild(c);

  SoLightModel *lm = new SoLightModel;
  lm->model = SoLightModel::BASE_COLOR;
  root->addChild(lm);

  // Axes and labels group
  SoInput in;
  in.setBuffer((void *)GridString, strlen(GridString));
  SoNode *node;	
  SoDB::read(&in, node);
  SoSeparator *g1 = (SoSeparator *)node;
  if (g1 != NULL)
    {
      root->addChild(g1);
      g1->boundingBoxCaching = SoSeparator::ON;
    }

  // Line manipulator group

  SoMaterial *m = new SoMaterial;
  m->diffuseColor.setValue(0.0F, 0.8F, 0.8F);
  root->addChild(m);

  manip = new LineManip2;
  manip->setHilightSize(0.01F);
  root->addChild(manip);

  SoCoordinate3 *coord = manip->getCoordinate3();
  SoNodeSensor *d = new SoNodeSensor(profileCallback, coord);
  d->attach(coord);
  coord->point.setValues(0, 2, defaultCoords);

  // Also initialize the profile to be the same in the main scene
  SbVec2f *defaultProfile = new SbVec2f[2];
  for (int i = 0; i < 2; i++) {
    defaultProfile[i].setValue(defaultCoords[i][0], defaultCoords[i][1]);
    profile->index.set1Value(i, i);
  }
  profileCoords->point.setValues(0, 2, defaultProfile);
  delete[] defaultProfile;

  root->unrefNoDelete();

  return root;
}


//////////////////////////////////////////////////////////////////////
//  -------------------
//  |                 |
//  |                 |
//  |      Main       |
//  |                 |
//  |                 |
//  |                 |
//  -------------------  <- MainHeight
//  |But-|Bevel  |Text|
//  | ton|       |    |
//  -------------------
//       ^       ^
//       Vert1   Vert2
//
////////////////////////////////////////////////////////////////////


// constants that define the UI layout
const int MainHeight = 79;
const int Vert1 = 20;
const int Vert2 = 60;

int
main(int /*argc*/, char **argv)
{
  Widget w = SoXt::init(argv[0]);
  if (w == NULL) exit(1);

  LineManip2::initClass();

  Arg resources[20];

  int n = 0;
  XtSetArg(resources[n], (char *)"width", 620); n++;
  XtSetArg(resources[n], (char *)"height", 620); n++;
  Widget form = XmCreateForm(w, (char *)"form", resources, n); n = 0;

  //
  // There are two 'tab groups' because keyboard focus must go to
  // either the text type-in, or to the profile render area (which
  // will respond to backspace-key events by deleting the last point
  // added).
  //
  XmAddTabGroup(form);

  SoXtExaminerViewer *examViewer = new SoXtExaminerViewer(form);
  examViewer->setSceneGraph(createTextSceneGraph());
  examViewer->setDecoration(FALSE);
  examViewer->setBorder(TRUE);
  examViewer->setFeedbackVisibility(FALSE);
  XtSetArg(resources[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(resources[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(resources[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(resources[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
  XtSetArg(resources[n], XmNbottomPosition, MainHeight); n++;
  XtSetValues(examViewer->getWidget(), resources, n); n = 0;
  examViewer->show();

  SoXtRenderArea *profile = new SoXtRenderArea(form);
  profile->setSceneGraph(createProfileSceneGraph());
  profile->setBorder(TRUE);
  XtSetArg(resources[n], XmNtopAttachment, XmATTACH_POSITION); n++;
  XtSetArg(resources[n], XmNtopPosition, MainHeight); n++;
  XtSetArg(resources[n], XmNleftAttachment, XmATTACH_POSITION); n++;
  XtSetArg(resources[n], XmNleftPosition, Vert1); n++;
  XtSetArg(resources[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(resources[n], XmNrightPosition, Vert2); n++;
  XtSetArg(resources[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(resources[n], XmNtraversalOn, TRUE); n++;
  XtSetValues(profile->getWidget(), resources, n); n = 0;
  XtAddEventHandler(profile->getWidget(), EnterWindowMask,
		    FALSE, enterProfileWindow, NULL);
  XtAddEventHandler(profile->getWidget(), LeaveWindowMask,
		    FALSE, leaveProfileWindow, NULL);
  profile->show();

  XtSetArg(resources[n], XmNvalue, DefaultString); ++n;
  XtSetArg(resources[n], XmNeditMode, XmMULTI_LINE_EDIT); ++n;
  XtSetArg(resources[n], XmNwordWrap, TRUE); ++n;
  XtSetArg(resources[n], XmNscrollHorizontal, FALSE); ++n;
  XtSetArg(resources[n], XmNtopAttachment, XmATTACH_POSITION); ++n;
  XtSetArg(resources[n], XmNtopPosition, MainHeight); ++n;
  XtSetArg(resources[n], XmNleftAttachment, XmATTACH_POSITION); ++n;
  XtSetArg(resources[n], XmNleftPosition, Vert2); ++n;
  XtSetArg(resources[n], XmNrightAttachment, XmATTACH_FORM); ++n;
  XtSetArg(resources[n], XmNbottomAttachment, XmATTACH_FORM); ++n;
  Widget textEdit = XmCreateScrolledText(form, (char *)"textEdit",
					 resources, n); n = 0;
  XtAddCallback(textEdit, XmNactivateCallback,
		updateText, (XtPointer)textEdit);
  XtAddCallback(textEdit, XmNvalueChangedCallback,
		delayUpdateText, (XtPointer)textEdit);
  XtManageChild(textEdit);
  updateText(textEdit, (XtPointer)textEdit, NULL);

  XtSetArg(resources[n], XmNnumColumns, 1); ++n;
  XtSetArg(resources[n], XmNorientation, XmVERTICAL); ++n;
  XtSetArg(resources[n], XmNpacking, XmPACK_COLUMN); ++n;
  XtSetArg(resources[n], XmNadjustLast, FALSE); ++n;
  XtSetArg(resources[n], XmNleftAttachment, XmATTACH_FORM); ++n;
  XtSetArg(resources[n], XmNrightAttachment, XmATTACH_POSITION); ++n;
  XtSetArg(resources[n], XmNrightPosition, Vert1); ++n;
  XtSetArg(resources[n], XmNbottomAttachment, XmATTACH_FORM); ++n;
  XtSetArg(resources[n], XmNbottomOffset, 2); ++n;
  XtSetArg(resources[n], XmNspacing, 0); ++n;
  XtSetArg(resources[n], XmNmarginHeight, 0); ++n;
  XtSetArg(resources[n], XmNmarginWidth, 17); ++n;
  Widget rc = XmCreateRowColumn(form, (char *)"buttons", resources, n);
  n = 0;

#define STRING(a) XmStringCreateSimple((char *)a)

  XtSetArg(resources[n], XmNlabelString, STRING("Copy")); ++n;
  Widget copyButton = XmCreatePushButtonGadget(rc, (char *)"copyButton",
					       resources, n); n = 0;
  XtAddCallback(copyButton, XmNactivateCallback,
		copyText, (XtPointer)rc);
  XtManageChild(copyButton);

  XtSetArg(resources[n], XmNlabelString, STRING("Edit Parts")); ++n;
  Widget editButton = XmCreatePushButtonGadget(rc, (char *)"editButton",
					       resources, n); n = 0;
  XtAddCallback(editButton, XmNactivateCallback,
		createPartEditor, (XtPointer)rc);
  XtManageChild(editButton);

  XtSetArg(resources[n], XmNlabelString, STRING("About...")); ++n;
  Widget aboutButton = XmCreatePushButtonGadget(rc, (char *)"about",
						resources, n); n = 0;
  XtAddCallback(aboutButton, XmNactivateCallback,
		showAboutDialog, NULL);
  XtManageChild(aboutButton);

  XtSetArg(resources[n], XmNlabelString, STRING("Quit")); ++n;
  Widget quitButton = XmCreatePushButtonGadget(rc, (char *)"quitButton",
					       resources, n); n = 0;
  XtAddCallback(quitButton, XmNactivateCallback,
		quitCallback, NULL);
  XtManageChild(quitButton);

  XtManageChild(rc);

  XtManageChild(form);

  SoXt::show(w);
  SoXt::mainLoop();
  SoXt::finish();
  return 0;
}


