
#include <QUrl>
#include <QFileDialog>
#include <QLayout>
#include <QMimeData>
#include <QDropEvent>
#include <Inventor/SoPreferences.h>
#include <Inventor/caches/SoCache.h>
#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <IvTune/SoIvTune.h>

#include <MainWindow.h>

bool MainWindow::shouldStop = false;

//------------------------------------------------------------------------------
MainWindow::MainWindow(QWidget* parent)
:QMainWindow(parent)
, m_viewerExaminer(NULL)
{
  // This is a convenient method from Qt to add the widgets to the interface.
  setupUi( this );

  m_fileOpenMenu->setShortcut( QKeySequence::Open );
  m_exitAction->setShortcut( QKeySequence::Quit );

  // We create a ViewerExaminer instance.
  m_viewerExaminer = new ViewerExaminer(NULL);

  // We set the ViewerExaminer instance as central widget to display it.
  this->setCentralWidget(m_viewerExaminer);

  // This env setting contains the path to OpenInventor
  SbString filepath = SoPreferences::getString("OIVHOME", "");
  filepath += "/data/models/chair.iv";

  // We set the scene graph from a file
  SoInput input;
  input.openFile(filepath);
  SoNode* m_sceneGraph;
  m_sceneGraph = SoDB::readAll(&input);
  m_viewerExaminer->setSceneGraph( m_sceneGraph );
  m_viewerExaminer->viewAll();

  setAcceptDrops(true);
 
  // Called when the user wants to open a file
  connect( m_fileOpenMenu, SIGNAL( triggered() ), this, SLOT( openFileAction() ) );

  // Ok we leave...
  connect( m_exitAction, SIGNAL( triggered() ), this, SLOT( close() ) );

  // launch IvTunevIewer to edit the scene graph.
  connect( m_ivtuneMenu, SIGNAL( triggered() ), this, SLOT( runIvTune() ) );

}

//------------------------------------------------------------------------------
void MainWindow::closeEvent( QCloseEvent* )
{
  MainWindow::shouldStop = true;
}

//------------------------------------------------------------------------------
void
MainWindow::openFileAction()
{
  QString defaultPath = QString::fromUtf16( (char16_t *) SoPreferences::getString("OIVHOME", "").toUtf16() );

  QString fileName = QFileDialog::getOpenFileName(this,
     tr("Open Iv file"), defaultPath, tr("IV Files (*.iv)"));

  if ( fileName.length() )
  {
    m_viewerExaminer->loadFile( fileName );
    if ( SoIvTune::isRunning() )
    {
      SoIvTune::stop();
      runIvTune();
    }
  }
}

//------------------------------------------------------------------------------
void
MainWindow::dropEvent( QDropEvent* event )
{
 QList<QUrl> urlList;
 
 if (event->mimeData()->hasUrls())
 {
   urlList = event->mimeData()->urls();
   
   // If we have one or more then we take the first one.
   if ( urlList.size() > 0)
   {
     // We need a local filesystem filename
     QString filename = urlList[0].toLocalFile(); 
     // Ok we can open it!
     if ( filename.length() )
     {
       m_viewerExaminer->loadFile( filename );
       if ( SoIvTune::isRunning() )
       {
         SoIvTune::stop();
         runIvTune();
       }
     }
   }
 }
 
 event->acceptProposedAction();
}

//------------------------------------------------------------------------------
void
MainWindow::dragEnterEvent( QDragEnterEvent* event )  
{
  // We accept any kind of event, we'll filter later...
  event->acceptProposedAction();
}

//------------------------------------------------------------------------------
void
MainWindow::runIvTune()
{
  SoIvTune::start( m_viewerExaminer->getRenderArea()->getSceneGraph() );
}
