/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : MMH (MMM YYYY)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCone.h>

#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoResetTransform.h>

#include <Inventor/nodes/SoCallback.h> 
#include <Inventor/elements/SoViewportRegionElement.h>

#include <Inventor/helpers/SbFileHelper.h>


#ifdef WIN32
#pragma warning( push )
#pragma warning( disable:4996 ) // Ignore stdio unsafe warnings
#endif 

////////////////////////////////////////////////////////////////////////
//
// Declarations and code to simulate a "pixel space camera"
//
// It's simulated because we're really using a standard orthographic
// camera plus a callback node that modifies the view volume to match
// the current pixel viewport dimensions.  In other words, each pixel
// space camera is 
//
// Actually creating a new kind of camera would be more elegant, but
// also much more complicated and involving a lot of stuff that's not
// directly relevant to this example.

//----------------------------------------------------------------------
// Information an instance of pixelSpaceCamera needs
//   - The ortho camera that will be modified
//   - Invert flags for X and Y
//     If FALSE (default), the range of values is 0..N-1 pixels.
//     Else the range is -(N-1)..0, which allows geometry to be
//     positioned relative to the right and/or top edge of the window.
//   - Last viewport (so we don't modify camera unless there is a change).
class pixelCameraInfo {
public:
  pixelCameraInfo() : 
    m_pCamera(NULL), 
    m_invertX(0), 
    m_invertY(0),
    m_lastViewport(0,0) {}

  SoOrthographicCamera *m_pCamera;      // Ortho camera we will modify
  SbBool                m_invertX;      // 0..N-1 if FALSE, else -(N-1)..0
  SbBool                m_invertY;      // ditto
  SbVec2i32             m_lastViewport; // 
};

//----------------------------------------------------------------------
// Callback function that modifies the camera
//
// Gets the current viewport from the traversal state and, if the
// viewport has changed, modifies the associated orthographic camera
// so we have a "pixel space" view volume in X and Y.
//
//   - userData : Callback instance info setup by application.
//                In this case should be ptr to a pixelCameraInfo object.
//   - pAction  : The action currently traversing the scene graph.
//                Could be a render action, bounding box action, etc.
//                In this case we do the same thing for all actions.
//
void pixelCameraCB( void *userData, SoAction *pAction )
{
  // Get the current traversal state
  SoState *pState = pAction->getState();
  if (!pState)
    return;

  // Get the current viewport from the traversal state
  const SbViewportRegion &vpRegion = SoViewportRegionElement::get( pState );
  const SbVec2i32 viewport = vpRegion.getViewportSizePixels_i32();

  // Get pixel space info object
  pixelCameraInfo *pInfo = (pixelCameraInfo *)userData;

  // Return if viewport same as last time (no need to change camera)
  if (viewport == pInfo->m_lastViewport)
    return;

  // Save viewport and compute aspect ratio
  pInfo->m_lastViewport = viewport;
  int vpWidth, vpHeight;
  viewport.getValue( vpWidth, vpHeight );
  float vpAspect = (float)vpWidth / vpHeight;

  // First disable notification on the camera node
  // (we're already in the middle of a traversal)
  // 
  // Set the view volume height to number of pixels in Y.
  // Set the aspect ratio the same as the viewport
  // (effectively setting view volume width to pixels in X).
  SoOrthographicCamera *pCamera = pInfo->m_pCamera;
  pCamera->enableNotify( FALSE );
  pCamera->height = (float)vpHeight;
  pCamera->aspectRatio = vpAspect;

  // Compute an offset that will make the view volume 0..N-1
  // (or the reverse if inverted).
  //
  // Remember that Inventor first computes a view volume centered around
  // 0,0.  I.e. -N/2 to N/2, where N is the width or the height.
  // Then it translates the view volume by pCamera->position.  So...
  //  - If the viewport width is even, for example 400, we need the left
  //    edge at zero, the center at 199.5 and the right edge at 399.
  //  - If the viewport width is odd, for example 399, we need the left
  //    edge at zero, the center at 199 and the right edge at 398.
  //
  // If the invert flag is set, that axis will be -(N-1)..0 instead.
  // This allows positioning geometry relative to the right/top edge.
  float xRadius = 0.5f * (vpWidth - 1);
  float yRadius = 0.5f * (vpHeight - 1);
  if (pInfo->m_invertX)
    xRadius = -xRadius;
  if (pInfo->m_invertY)
    yRadius = -yRadius;
  pCamera->position = SbVec3f( xRadius, yRadius, 2 );
  pCamera->enableNotify( TRUE );

#if defined(_DEBUG)
  fprintf( stderr, "--- Camera 0x%p updated, vp = %d %d\n", pCamera, vpWidth, vpHeight );
#endif
}

//----------------------------------------------------------------------
// Convenience function to set up a simulated "pixel space" camera
//
//   - pParent : Group node to add the camera under
//   - invertX : X will be 0..N-1 if False, else -(N-1)..0
//   - invertY : Y will be 0..N-1 if False, else -(N-1)..0
//
void addPixelSpaceCamera( SoGroup *pParent, 
                          SbBool invertX = 0, SbBool invertY = 0 )
{
  // Create an orthographic camera
  SoOrthographicCamera *p2dCam = new SoOrthographicCamera();

  // Set the viewportMapping to LEAVE_ALONE because we're going to
  // make the view volume aspect match the viewport ourselves.
  p2dCam->viewportMapping = SoCamera::LEAVE_ALONE;

  // We're going to put the camera position at +2 in Z, so these
  // near and far distances make the view volume 2 units deep in
  // Z centered around zero.  Obviously we're assuming all the
  // 2D annotation geometry really is flat!
  p2dCam->nearDistance = 1;
  p2dCam->farDistance  = 3;

  // Setup info for pixel space camera callback
  pixelCameraInfo *pInfo = new pixelCameraInfo();
  pInfo->m_pCamera = p2dCam;
  pInfo->m_invertX = invertX;
  pInfo->m_invertY = invertY;

  // Create callback node to modify the camera.
  // Set it to call pixelCameraCB with pInfo
  SoCallback *p2dCamCB = new SoCallback();
  p2dCamCB->setCallback( pixelCameraCB, (void*)pInfo );

  // Add callback and camera to scene graph (callback before camera)
  pParent->addChild( p2dCamCB );
  pParent->addChild( p2dCam   );
}

////////////////////////////////////////////////////////////////////////

void
main(int argc, char **argv)
{
  SbString filename = SbFileHelper::expandString( "$OIVHOME" ) + "/examples/source/Inventor/BigImageRender/test.iv";
  if (argc > 1)
    filename = argv[1];
  FILE *fp = fopen( filename.toLatin1 (), "r" );
  if (fp == NULL) {
    printf( "Unable to open '%s'\n", filename.toLatin1() );
    filename.makeNull();
  }
  else
    fclose( fp );

  // Initialize Open Inventor
  Widget myWindow = SoXt::init(argv[0]);
  if (myWindow == NULL) exit(1);

  // Root node of scene graph
  // 3D (application) and 2D (annotation) scene graphs below this.
  SoSeparator *pRoot = new SoSeparator();
  pRoot->ref();

  // Try to read the file in 3D Root separator
  SoSeparator *p3dSep = NULL;
  if (! filename.isEmpty()) {
    SoInput in;
    in.openFile( filename.toLatin1 () );
    p3dSep = SoDB::readAll( &in );
    in.closeFile();
  }

  // If that failed, create our favorite geometry
  if (p3dSep == NULL) {
    SoMaterial  *pMatl = new SoMaterial();
    SoCone      *pCone = new SoCone();

    pMatl->diffuseColor.setValue( 1,0,0 );

    p3dSep = new SoSeparator();
    p3dSep->addChild( pMatl );
    p3dSep->addChild( pCone );
  }

  // Create 3D (usually perspective) camera.
  // We'll add this above the 3D scene root to make it easy to clear out
  // the 3D scene (e.g. removeAllChildren) and add a new one.
  SoPerspectiveCamera *p3dCam = new SoPerspectiveCamera();
  pRoot->addChild( p3dCam );

  // Add the 3D scene to the scene graph.
  pRoot->addChild( p3dSep );

  // ----- 2D Annotation Setup ---

  // Font settings and material for 2D annotations.
  SoFont *pFont1 = new SoFont();
  pFont1->name = "Arial";
  pFont1->size = 20;
  pFont1->renderStyle = SoFont::TEXTURE;
  SoMaterial *pMatl1 = new SoMaterial();
  pMatl1->diffuseColor = SbColor(1,1,1);

  // Create 2D scene root and build basic 2D scene
  SoSeparator *p2dSep   = new SoSeparator();
  pRoot->addChild( p2dSep );
  p2dSep->addChild( pFont1 );
  p2dSep->addChild( pMatl1 );

  // Create 2D text annotation in upper left corner of window.
  //
  // We'll put this under its own separator so the camera and
  // translation don't affect anything else.
  //
  // First we'll create a "pixel space" camera with the Y axis inverted
  // so we can position geometry relative to the top edge of the window.
  //
  // Next we need a translation node to position the text 5 pixels in
  // from the left and 20 pixels down from the top. (20 pixels down
  // because font height is 20 meaning text is slightly less.)
  SoSeparator *pTextSep1 = new SoSeparator();
    addPixelSpaceCamera( pTextSep1, FALSE, TRUE ); // Y is inverted
    SoTranslation *pTran1 = new SoTranslation();
      pTran1->translation = SbVec3f( 5, -20, 0 ); // Upper left corner
      pTextSep1->addChild( pTran1 );
    SoText2 *pText1 = new SoText2();
      pText1->string = "Upper Left";
      pTextSep1->addChild( pText1 );
  p2dSep->addChild( pTextSep1 );

  // Create 2D text annotation in lower left corner of window.
  // Same structure as previous, except neither axis is inverted.
  SoSeparator *pTextSep2 = new SoSeparator();
    addPixelSpaceCamera( pTextSep2, FALSE, FALSE ); // Neither X nor Y is inverted
    SoTranslation *pTran2 = new SoTranslation();
      pTran2->translation = SbVec3f( 5, 5, 0 ); // Lower left corner
      pTextSep2->addChild( pTran2 );
    SoText2 *pText2 = new SoText2();
      pText2->string = "Lower Left";
      pTextSep2->addChild( pText2 );
  p2dSep->addChild( pTextSep2 );

  // Create 2D text annotation in lower right corner of window.
  // Same structure but X axis is inverted to allow positioning
  // relative to the right edge of the window, and also the
  // text justification is set to RIGHT so we can position the
  // text without knowing exactly how wide it is.
  SoSeparator *pTextSep3 = new SoSeparator();
    addPixelSpaceCamera( pTextSep3, TRUE, FALSE ); // X is inverted
    SoTranslation *pTran3 = new SoTranslation();
      pTran3->translation = SbVec3f( -5, 5, 0 ); // Lower right corner
      pTextSep3->addChild( pTran3 );
    SoText2 *pText3 = new SoText2();
      pText3->string = "Lower Right";
      pText3->justification = SoText2::RIGHT;
      pTextSep3->addChild( pText3 );
  p2dSep->addChild( pTextSep3 );

  // Create 2D text annotation in upper right corner of window.
  // Same structure as previous except both axes are inverted.
  SoSeparator *pTextSep4 = new SoSeparator();
    addPixelSpaceCamera( pTextSep4, TRUE, TRUE ); // Both axes inverted
    SoTranslation *pTran4 = new SoTranslation();
      pTran4->translation = SbVec3f( -5, -20, 0 ); // Upper right corner
      pTextSep4->addChild( pTran4 );
    SoText2 *pText4 = new SoText2();
      pText4->string = "Upper Right";
      pText4->justification = SoText2::RIGHT;
      pTextSep4->addChild( pText4 );
  p2dSep->addChild( pTextSep4 );

  // Finally... We don't want the extent of the annotations to affect
  // a "viewAll" of the application geometry, so we'll reset the
  // bounding box to empty after traversing the 2D annotation stuff.
  SoResetTransform *pResetBbox = new SoResetTransform();
  pResetBbox->whatToReset = SoResetTransform::BBOX;
  p2dSep->addChild( pResetBbox );

  // Create the viewer and set some properties
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setBackgroundColor( SbColor(0,.6f,.6f) );
  myViewer->setTitle( "2D Annotation" );
  myViewer->show();

  // Load the scene graph and adjust the 3D camera to view entire scene.
  // Have to do this "manually" because we have our own camera.
  // Don't forget to save this as the Home position!
  myViewer->setSceneGraph( pRoot );
  myViewer->viewAll();
  myViewer->saveHomePosition();

  // Loop, then cleanup
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  pRoot->unref();
  SoXt::finish();
}

#ifdef WIN32
#pragma warning( pop )
#endif
