/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : MMH (MMM YYYY)
**=======================================================================*/

/////////////////////////////////////////////////////////////////////
// Annotation example - Gnomon (aka compass)
//
// Original: Mike Heck, Thermo Fisher Scientific
// Modified: Mike Heck
//           Replaced OpenGL calls with modern Open Inventor nodes.
//           Replaced SoResetTransform with SoBBox.
//
// This annotation is called a "gnomon" (pronounced NO-mon) or 
// just a "compass". Its purpose is to continuously display the 3D 
// orientation of the scene. This helps users understand what they 
// are looking at.  When the scene exists in an arbitary coordinate
// system, for example in computer aided engineering, the gnomon might
// be drawn as a small set of XYZ axes. When the scene is geo-referenced,
// for example in seismic interpretation, the gnomon might look like 
// part of a real world compass. We will only assume the gnomon is 
// defined by some 3D geometry.  Typically we want the gnomon to appear
// in one of the corners of the screen, to always appear on top of the
// scene graphics and to automatically rotate when the user changes
// the camera orientation. Typically the user cannot interact directly
// with the gnomon.
//
// There are three main parts to the solution:
// 1. The gnomon geometry
//    This can be any shape. Here it is a simple 3D axis built from
//    cylinders and cones.
// 2. The gnomon scene graph
//    This scene graph contains the gnomon geometry of course, but
//    also a switch to turn it off, some nodes to set and clear the
//    viewport and a callback node to update the gnomon camera when
//    the scene camera orientation changes.
// 3. The gnomon callback function
//    This function gets the current view orientation using the view
//    matrix from the traversal state list. This is much safer than,
//    for example, attaching a sensor to the scene camera because
//    the viewer may delete and replace that camera.
//
// Notes:
// - By default the gnomon scene graph uses an SoDepthBuffer node
//   to clear the depth buffer inside the gnomon's viewport.
//   This ensures that the gnomon is rendered on top of the scene.
//   BUT!
//   This is somewhat dangerous because it could affect rendering of
//   other parts of the scene graph. You can reduce the danger, by
//   making the gnomon the last thing in the scene graph.  But objects
//   that are semi-transparent may still be rendered after the gnomon
//   when the transparency mode is "_DELAYED".
//
/////////////////////////////////////////////////////////////////////

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

// Open Inventor nodes we will need
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoDepthBuffer.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoResetTransform.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoViewport.h>
#include <Inventor/nodes/SoViewportClipping.h>

// Open Inventor traversal state elements we will need
#include <Inventor/elements/SoViewingMatrixElement.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <iostream>
using namespace std;

////////////////////////////////////////////////////////////////////////
//
// Gnomon parameters
static const float GNOMON_CAMDIST = 3.5;
static const int   GNOMON_WIDTH   = 100; // pixels
static const int   GNOMON_HEIGHT  = GNOMON_WIDTH;

////////////////////////////////////////////////////////////////////////
//
// Define the gnomon geometry (simple axes in this case)
//
// We could define these programmatically, but using a memory stream
// avoids including headers for nodes that we don't actually use in our 
// program. We could define these in a .iv file, but the memory stream
// avoids the danger of not having the file available at run time.
//
// For convenience we define the gnomon geometry in a -1..1 space
// because this is the default view volume for Open Inventor cameras.
//
// Note that, to be safe, we reset some attributes that might be 
// inherited from the scene graph and are not appropriate for our axes.
// But to be safe you should also keep the actual scene under its own
// Separator, so its attributes won't affect the gnomon.

char *gnomonGeometry = "\
#Inventor V2.1 ascii\n\
\
DEF GnomonGeom Separator { \
  PickStyle { style UNPICKABLE } \
  DrawStyle { style FILLED } \
  LightModel { model PHONG } \
  MaterialBinding { value OVERALL } \
  Complexity { value .2 } \
  ShapeHints { vertexOrdering COUNTERCLOCKWISE shapeType SOLID } \
  Font { name \"Arial : Bold\" size 15 } \
  Separator { \
    Material { \
      diffuseColor    [ 0.5 0 0 ] \
      emissiveColor   [ 0.5 0 0 ] \
    } \
    RotationXYZ { axis Z angle -1.570796327 } \
    Cylinder { height 2 radius .06 } \
    Translation { translation 0 1 0 } \
    Cone { bottomRadius .18 height .3 } \
    Translation { translation 0 .16 0 } \
    Text2 { string \"X\" } \
  } \
  Separator { \
    Material { \
      diffuseColor    [ 0 0.5 0 ] \
      emissiveColor   [ 0 0.5 0 ] \
    } \
    Cylinder { height 2 radius .06 } \
    Translation { translation 0 1 0 } \
    Cone { bottomRadius .18 height .3 } \
    Translation { translation 0 .16 0 } \
    Text2 { string \"Y\" } \
  } \
  Material { \
    diffuseColor    [ 0 0 0.5 ] \
    emissiveColor   [ 0 0 0.5 ] \
  } \
  RotationXYZ { axis X angle 1.570796327 } \
  Cylinder { height 2 radius .06 } \
  Translation { translation 0 1 0 } \
  Cone { bottomRadius .18 height .3 } \
  Translation { translation 0 .16 0 } \
  Text2 { string \"Z\" } \
} ";


////////////////////////////////////////////////////////////////////////
//
// Gnomon callback function
//
// This function will be called (from our Callback node) just before
// the gnomon camera is traversed.  It does some drawing setup and
// modifies the gnomon camera, as follows:
//
// 1) Modify OpenGL viewport to position the gnomon in the window.
// 2) Clear the OpenGL depth buffer in the region where the gnomon
//    will be drawn (to ensure gnomon is "on top").
// 3) Modify the gnomon camera to:
//    a) Have the same orientation as the 3D scene camera.
//    b) View the gnomon geometry at a consistent size.
//
void gnomonCB( void *userData, SoAction *action )
{
  // Don't do anything if this is not a render traversal
  if (action->isOfType(SoGLRenderAction::getClassTypeId())) {

    SoState *state = action->getState();

    // Get the current camera rotation from the viewing matrix.
    //
    // Note that the viewing matrix is applied to geometry, so it's
    // actually the inverse of the matrix we need.  The getTransform
    // method will return the rotation in variable cameraRotation.
    //
    // We could get the camera orientation by hiding a pointer to the 
    // camera, but this approach is very "fragile".  For example, if the 
    // user changes the camera type by clicking the viewer button, the
    // viewer destroys the current camera and creates a new one.
    SbMatrix viewMat = SoViewingMatrixElement::get( state );
    SbVec3f tran, scale;
    SbRotation cameraRotation, orient;
    viewMat.inverse().getTransform( tran, cameraRotation, scale, orient );

    // Get ptr to the gnomon's camera
    SoCamera *camera = (SoCamera*)userData;

    // Disable notification because we're already traversing
    camera->enableNotify( FALSE );

    // Set the new orientation for the gnomon camera
    camera->orientation = cameraRotation;

    // Get the current "focal distance" (distance to pt-of-rotation)
    float distance = camera->focalDistance.getValue();

    // Reposition the camera so it's looking at the pt-of-rotation
    // (which is the center of the gnomon or 0,0,0 in this case).
    //
    // We'll use an algorithm similar to what the examiner viewer uses
    // to spin the camera around the pt-of-rotation.
    // First get the rotation as a matrix.  Conveniently SbMatrix has an
    // assignment operator that takes an SbRotation value.
    // Next extract the view direction vector from the rotation matrix.
    // Now move the camera radius units along the direction vector.
    SbMatrix mx;
    mx = cameraRotation;
    SbVec3f direction( -mx[2][0], -mx[2][1], -mx[2][2] );
    camera->position = SbVec3f(0,0,0) - distance * direction;
  }
}


////////////////////////////////////////////////////////////////////////
//
// Create the gnomon scene graph
//
SoSeparator* makeGnomon()
{
  SoSeparator         *pRoot   = new SoSeparator(1);
  SoSwitch            *pSwitch = new SoSwitch(2);
  SoCallback          *pCallb  = new SoCallback;
  SoPerspectiveCamera *pCam    = new SoPerspectiveCamera;
  SoResetTransform    *pReset  = new SoResetTransform;
  pRoot->ref();
  pRoot->setName( "Gnomon" );

  // Switch node will allow us turn the gnomon on and off
  pSwitch->setName( "GnomonSwitch" );
  pRoot->addChild( pSwitch );

  // Specify that the gnomon has no bounding box.
  // We don't want it to affect the viewer's viewAll operation.
  SoBBox* pBBox = new SoBBox;
  pBBox->mode = SoBBox::NO_BOUNDING_BOX;
  pSwitch->addChild( pBBox );

  // Specify viewport (screen region) that gnomon will be rendered into
  SoViewport* pViewport = new SoViewport;
  pViewport->size.setValue( (float)GNOMON_WIDTH, (float)GNOMON_HEIGHT );
  pSwitch->addChild( pViewport );

  // Specify screen region that depth buffer clear will affect
  SoViewportClipping* pViewportClipping = new SoViewportClipping;
  pViewportClipping->size.setValue( (float)GNOMON_WIDTH, (float)GNOMON_HEIGHT );
  pSwitch->addChild( pViewportClipping );

  // Clear the depth buffer to ensure gnomon is on top of the scene.
  // NOTE!
  // This is somewhat dangerous because it could affect rendering of
  // other parts of the scene graph. You can reduce the danger, by
  // making the gnomon the last thing in the scene graph.  But objects
  // that are semi-transparent may still be rendered after the gnomon
  // when the transparency mode is "_DELAYED".
  SoDepthBuffer* pDepthBuffer = new SoDepthBuffer;
  pDepthBuffer->clearBuffer = TRUE;
  pSwitch->addChild( pDepthBuffer );

  // Callback node allows us to modify the gnomon camera during traversal
  pCallb->setCallback( gnomonCB, (void*)pCam );

  SoSeparator* psep = new SoSeparator;
  psep->addChild(pCallb);
  pSwitch->addChild(psep);

  // Setup gnomon camera (viewAll might not work here because we want
  // the point of rotation to be the origin of the axes (i.e. 0,0,0),
  // not the geometric center of the gnomon geometry.
  pCam->setName( "GnomonCam" );
  pCam->position = SbVec3f( 0,0,GNOMON_CAMDIST );
  pCam->nearDistance  = 1;
  pCam->farDistance   = 6;
  pCam->focalDistance = GNOMON_CAMDIST;
  pSwitch->addChild( pCam );

  // Read the gnomon geometry from a memory buffer
  SoInput in;
  in.setBuffer((void *)gnomonGeometry, (size_t) strlen(gnomonGeometry));
  SoNode *node;
  SbBool ok = SoDB::read(&in, node);
  if (ok && node != NULL) {
    pSwitch->addChild(node);
    pSwitch->whichChild = SO_SWITCH_ALL;
  }

  pRoot->unrefNoDelete();
  return pRoot;
}

////////////////////////////////////////////////////////////////////////
//
// Application main function
void
main(int argc, char **argv)
{
  char *filename = "test.iv";
  if (argc > 1)
    filename = argv[1];
  if (! SbFileHelper::isAccessible( filename )) {
    cerr << "Unable to open '" << filename << "'" << endl;
    filename = NULL;    // Indicate we do not have a file
  }

  // Initialize Inventor
  HWND myWindow = SoWin::init(argv[0]);

  // If we have an input file, try to read it
  SoSeparator *pScene = NULL;
  if (filename != NULL) {
    SoInput in;
    in.openFile( filename );
    pScene = SoDB::readAll( &in );
    in.closeFile();
  }
  else {
    // Else create a simple scene graph
    pScene            = new SoSeparator;
    SoMaterial *pMatl = new SoMaterial;
    SoCone     *pCone = new SoCone;

    pMatl->diffuseColor.setValue( 1,0,0 );
    pScene->addChild( pMatl );
    pScene->addChild( pCone );
  }
  pScene->setName( "_3D_Scene" );

  // Keep the scene under its own separator so it
  // won't affect the appearance of the gnomon.
  SoSeparator *pRoot = new SoSeparator;
  pRoot->ref();
  pRoot->setName( "SceneRoot" );
  pRoot->addChild( pScene );

  // Create and initialize viewer
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setBackgroundColor( SbColor(.2f,.2f,.2f) );
  myViewer->setSceneGraph( pRoot );
  myViewer->setTitle( "Gnomon Example" );
  myViewer->show();

  // Add gnomon to scene graph
  // Specifically the gnomon subgraph must be inserted after the
  // camera that the viewer is controlling, because it uses the
  // viewing matrix inherited through the scene graph.  There are
  // other ways to do this, e.g. passing a pointer to the scene
  // camera to the callback, but this is fragile and breaks for
  // example if the user switches to an orthographic camera.
  //
  // Note that we need to make this addChild call *after* the
  // initial setScenegraph call, to ensure that the viewer will
  // control the scene camera.  If the actual scene does not
  // contain a camera (typical), then the viewer would find the
  // camera in the gnomon subgraph and control that one!
  SoSeparator *pGnomon = makeGnomon();
  pRoot->addChild( pGnomon );

  // Loop then cleanup
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  pRoot->unref();
  SoXt::finish();
}
