/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
//
//  ivclock - this views and operates a clock, whose geometry is
//  specified in an input file.
//
//  See the README file in this directory for a complete explanation.
//

#include <X11/Intrinsic.h>
#include <math.h>

#include <Inventor/SoDB.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/engines/SoTimeCounter.h>
#include <Inventor/engines/SoCounter.h>
#include <Inventor/engines/SoCalculator.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoWWWInline.h>

#include <Inventor/actions/SoWriteAction.h>



class ClockHands {
  public:
    SoRotationXYZ   *hour;
    SoRotationXYZ   *minute;
    SoRotationXYZ   *second;
};

static
void getCurrentTime( int &h, int &m, int &s )
{
    time_t clock;
    struct tm *timeofday;

    time(&clock);
    timeofday = (struct tm *)localtime(&clock);
    h = timeofday->tm_hour;
    if (h >= 12) h -= 12;
    m = timeofday->tm_min;
    s = timeofday->tm_sec;

}


// Find the clock hand separators.
// If the geometry does not include a minute hand and an hour hand
// return FALSE.
// Attach the engines to make the hands rotate.
static SbBool
setupHands(SoGroup *root)
{
    SbBool foundSecond = FALSE;
    int currentHour, currentMinute, currentSecond;

    // Engines to time the ticks of the clock hands
    SoTimeCounter *minuteTimer, *secondTimer = nullptr;
    SoCounter *minuteCounter, *hourCounter;

    // Engines to convert the clock ticks into radian angles of rotation
    SoCalculator *secondRadians = new SoCalculator; 
    SoCalculator *minuteRadians = new SoCalculator; 
    SoCalculator *hourRadians = new SoCalculator; 

    // Rotation of the hands
    ClockHands	*rotation = new ClockHands;
    rotation->hour = NULL;
    rotation->minute = NULL;
    rotation->second = NULL;

    // Get current time of day to set the initial time
    getCurrentTime(currentHour, currentMinute, currentSecond);
    
    // Search for Names which denote the hands of the clock,
    // then insert these rotations into the clock geometry and
    // attach engines to update the rotations.

    SoSeparator *s;


    // Second hand (optional)
    s = (SoSeparator *)root->getByName("SecondHand");
    if (s != NULL && s->isOfType(SoSeparator::getClassTypeId())) {

	rotation->second = new SoRotationXYZ;
	rotation->second->axis = SoRotationXYZ::Z;
	s->insertChild(rotation->second, 0);
        // Second hand ticks from 0 to 59 in one minute (60 seconds)
	secondTimer = new SoTimeCounter;
        secondTimer->min = 0;
        secondTimer->max = 59;
        secondTimer->frequency = 1./60.;
        secondTimer->reset = currentSecond;
        secondRadians->a.connectFrom(&secondTimer->output);
        secondRadians->expression = "oa=-a*M_PI/30.0";
        rotation->second->angle.connectFrom(&secondRadians->oa);
	foundSecond = TRUE;
   }



    // Hour hand 
    s = (SoSeparator *)root->getByName("HourHand");
    if (s != NULL && s->isOfType(SoSeparator::getClassTypeId())) {
	rotation->hour = new SoRotationXYZ;
	rotation->hour->axis = SoRotationXYZ::Z;
	s->insertChild(rotation->hour, 0);

	hourCounter = new SoCounter;
	hourCounter->min = 0;
	hourCounter->max = 11;
        hourCounter->reset = currentHour;
        hourRadians->a.connectFrom(&hourCounter->output);
        hourRadians->expression = "oa=-((a+b/60.)*M_PI/6.0)";
        rotation->hour->angle.connectFrom(&hourRadians->oa);
    } else {
	return (FALSE); // The hour hand is required
    }



    // Minute hand 
    s = (SoSeparator *)root->getByName("MinuteHand");
    if (s != NULL && s->isOfType(SoSeparator::getClassTypeId())) {
	rotation->minute = new SoRotationXYZ;
	rotation->minute->axis = SoRotationXYZ::Z;
	s->insertChild(rotation->minute, 0);

 	// If the clock has a second hand:
	// use the second timer to figure out when to tick the minutes
    if (foundSecond && secondTimer) {
	    minuteCounter = new SoCounter;
	    minuteCounter->min = 0;
	    minuteCounter->max = 59;
            minuteCounter->reset = currentMinute;
	    minuteCounter->trigger.connectFrom(&secondTimer->syncOut);
            minuteRadians->a.connectFrom(&minuteCounter->output);

	    hourCounter->trigger.connectFrom(&minuteCounter->syncOut);
            hourRadians->b.connectFrom(&minuteCounter->output);
	} 
	// If the clock doesn't have a second hand:
        // use a minute timer
	else {
	    minuteTimer = new SoTimeCounter;
            minuteTimer->min = 0;
            minuteTimer->max = 59;
            minuteTimer->frequency = 1./3600.;
            minuteTimer->reset = currentMinute;
            minuteRadians->a.connectFrom(&minuteTimer->output);

	    hourCounter->trigger.connectFrom(&minuteTimer->syncOut);
            hourRadians->b.connectFrom(&minuteTimer->output);
	}

        minuteRadians->expression = "oa=-a*M_PI/30.0";
        rotation->minute->angle.connectFrom(&minuteRadians->oa);
    } else {
	return (FALSE);  // The minute hand is required
    }

    return (TRUE);
}

int
main(int argc, char **argv)
{

    SbString filename = "$OIVHOME/examples/source/Inventor/clock/clockData.iv";

    if (argc != 2) {
	fprintf(stderr, "NOTE: You can specify your own geometry file.\n");
	fprintf(stderr, "Run: %s inputFile\n",  argv[0]);
	fprintf(stderr, "Running with the default geometry, %s\n", filename.toLatin1());
    }
    else filename = argv[1];
    
    printf("\nUse left mouse to rotate the view.\n");
    printf("Middle mouse to pan the view.\n");
    printf("Left and middle together to zoom the view\n");
    printf("Right mouse for a popup menu showing more viewing features\n");

    
    Widget mainWindow = SoXt::init("Time");
    if (mainWindow == NULL)
	exit (1);



    // Read in the scene graph
    SoInput in;
    if (!in.openFile(filename)) {
        fprintf(stderr, "Error - could not open %s.\n", filename.toLatin1());
	exit(1);
    }

    SoWWWInline::setReadAsSoFile(TRUE);

    SoSeparator *root = SoDB::readAll(&in);
    if (root == NULL) {
        fprintf(stderr, "Error - could not read %s.\n", filename.toLatin1());
	exit(1);
    }
    root->ref();

    if (! setupHands(root)) {
        fprintf(stderr, "Error - cannot use %s for the clock.\n", 
			filename.toLatin1());
        exit(1);
    }


//    SoWriteAction myAction;
//
 //   myAction.getOutput()->openFile("myclock.iv");
  //  myAction.getOutput()->setBinary(FALSE);
   // myAction.apply(root);
    //myAction.getOutput()->closeFile();


    // Build and initialize the Inventor render area widget
    SoXtExaminerViewer *examiner = new SoXtExaminerViewer(mainWindow);
    examiner->setSize(SbVec2s(100, 100));
    examiner->setDecoration(FALSE);
    examiner->setFeedbackVisibility(FALSE);
    examiner->setSceneGraph(root);
    examiner->setTitle("Inventor Time");

    examiner->show();
    examiner->viewAll();

    SoXt::show(mainWindow);
    SoXt::mainLoop();
}

