/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
//
// A class containing some utility routines that I find useful.
//

#include <Inventor/SoDB.h>
#include <Inventor/SoLists.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoSeparator.h>

#include "Useful.h"

//
// Read a file given a filename and return a separator containing all
// of the stuff in the file.
//
SoSeparator *
Useful::readFile(const char *filename)
{
    SoInput in;
    if (filename != NULL) {
	if (in.openFile(filename) == FALSE) {
	    fprintf(stderr, "Could not open file %s\n", filename);
	    return NULL;
	}
    }
    return readFile(in);
}
//
// Read a file given a file pointer...
//
SoSeparator *
Useful::readFile(FILE *fp)
{
    SoInput in;
    in.setFilePointer(fp);
    return readFile(in);
}
//
// And read a file given an SoInput.  This is used by the other
// readFile routines.
//
SoSeparator *
Useful::readFile(SoInput &in)
{
    SoSeparator *graph = new SoSeparator;
    graph->ref();

    //
    // Keep on reading until there are no more nodes to read
    //
    SoNode *root;
    do {
	int read_ok = SoDB::read(&in, root);

	if (!read_ok) {
	    fprintf(stderr, "Error reading file\n");
	    graph->unref();
	    return NULL;
	}
	else if (root != NULL) graph->addChild(root);

    } while (root != NULL);
    in.closeFile();

    //
    // Try to avoid creating extra separators; if this scene graph
    // already has exactly one separator at the top, use it.  This
    // will avoid an explosion of separators at the top of scenes that
    // would otherwise occur if we automatically created a new
    // separator every time a scene graph was read.
    //
    if (graph->getNumChildren() == 1 && 
		graph->getChild(0)->isOfType(
		SoSeparator::getClassTypeId())) {
	SoSeparator *result = (SoSeparator *)graph->getChild(0);
	result->ref();	// Note the order here!
	graph->unref();

	result->unrefNoDelete();
	return result;
    }

    graph->unrefNoDelete();
    return graph;
}

//
// I use this routine to make a good guess at which material or
// texture or material binding or other property affects a given
// shape.  For example, the last material on the path to an object
// will be that object's material(s), unless ignore flags are set on
// that material's fields.
//
SoNode *
Useful::searchLastType(SoPath *p, SoType t)
{
    SoSearchAction sa;
    sa.setType(t);
    sa.setInterest(SoSearchAction::LAST);
    sa.apply(p);
    SoPath *outPath = sa.getPath();

    SoNode *result = NULL;
    if (outPath != NULL && (outPath->getLength() > 0) )
        result = outPath->getTail();

    return result; 
}

