/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
//
// Print information about the Structure of a Nodekit class.
//

#include <stdio.h>
#include <dlfcn.h>
#include <getopt.h>
#include <Inventor/SoDB.h>
#include <Inventor/nodekits/SoBaseKit.h>
#include <Inventor/Xt/SoXt.h>

static void
print_usage(const char *progname)
{
  (void)fprintf(stderr, "\nThis program prints a diagram and a table of the structure of a nodekit class.\n");
  (void)fprintf(stderr, "See the README for details about the diagrams and tables.\n\n");
  (void)fprintf(stderr, "Usage: %s [-h] className\n\n", progname);
  (void)fprintf(stderr, "       -h : This message (help)\n");
  (void)fprintf(stderr, "className : The name of any node class");
  (void)fprintf(stderr, "derived from SoBaseKit.\n");
  (void)fprintf(stderr, "            This includes all dragger classes as well.\n");
  exit(99);
}

static void
parse_args(int argc, char **argv)
{
  int err = 0;	// Flag: error in options?
  int c;
    
  while ((c = getopt(argc, argv, "h")) != -1) {
    switch(c) {
    case 'h':	// Help
    default:
      err = 1;
      break;
    }
  }

  if (err) {
    print_usage(argv[0]);
  }
}

int main(int argc, char **argv)
{
  SoXt::init(argv[0]);

  // Parse arguments

  parse_args(argc, argv);
  char *classString;

  if (optind != (argc - 1))
    print_usage(argv[0]);
  else
    classString = strdup(argv[optind]);

  // 'fromName' takes the name of the Class as it appears in file.
  // For the built-in Inventor types, this differs from the class name
  // because it has no 'So' prefix.
  // Therefore, if the class name begins with 'So', our first try will be
  // with the prefix removed.
  SbBool startsWithSo = FALSE;
  if ( !strncmp( classString, "So", 2 ) ) {
    startsWithSo = TRUE;
  }

  char *lookupString;
  if (startsWithSo == TRUE)
    lookupString = classString + 2;
  else
    lookupString = classString;

  SoType classType = SoType::fromName(lookupString);

  // If we got a bad type and the name began with "So" we will type
  // a message and try again with the full name.
  if (classType == SoType::badType() && startsWithSo) {
    fprintf(stderr,"The className you gave has a prefix of \"So\"\n");
    fprintf(stderr,"I tried to remove the prefix and look it up but\n");
    fprintf(stderr,"found no such class. Now I'll try again with the\n");
    fprintf(stderr,"full string you gave me\n");
    classType = SoType::fromName(classString);
  }

  if (classType == SoType::badType()) {
    fprintf(stderr, "ERROR: The given className is not a valid\n");
    fprintf(stderr, "       node type. Message retrieved \n");
    fprintf(stderr, "       from dlerror() follows:\n%s\n", dlerror());
    exit(0);
  }
  else if ( ! classType.isDerivedFrom(SoBaseKit::getClassTypeId())) {
    fprintf(stderr, "ERROR: The given className is not a subClass\n");
    fprintf(stderr, "       of SoBaseKit.\n");
    exit(0);
  }

  SoBaseKit *myKit = (SoBaseKit *) classType.createInstance();

  // Print the diagram and the table.
  myKit->printDiagram();
  fprintf(stdout, "\n\n");
  myKit->printTable();

  return 0;
}

