///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*----------------------------------------------------------------------------------------
* Example program.
* Purpose : Demonstrate how to use CPU data combining feature
* author : Jerome Hummel
* date :   November 2005
* Updaded by Pascal Estrade (Sep 2014)
*
* Modification : Pascal Estrade
* Modication Data : September 2014
*
* This program combines two volumes - data volume1 and "mask" volume2
* The mask volume is used to selectively make voxels completely
* transparent, effectively clipping volume1 at voxel precision.
*
* To achieve this effect we define the mask volume to contain only the
* values zero and one.  We create an SoDataCompositor node (or custom
* compositor) that combines the two volumes by multiplying the data
* values.  In other words, for each voxel:
*     result_voxel = volume1_voxel * volume2_voxel
* Any voxel where volume2 contains zero will be forced to zero, while
* all voxels where volume2 contains one will be unchanged.
* Finally we make sure that value zero maps to completely transparent
* in the transfer function.
*
* The program can be built two different ways:
*
* 1. If USE_CUSTOM_COMPOSITOR is defined, then a custom compositor
*    class is derived from SoDataCompositor.  In this case the custom
*    compositor implements one of the predefined algorithms so we can
*    easily confirm that the result is correct.
*
* 2. Else a standard SoDataCompositor node is created and its
*    preDefCompositor field is set to SoDataCompositor::MULTIPLY.
*
*----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/SoPreferences.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoDataCompositor.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

const char* FILENAME = "$OIVHOME/examples/data/Medical/files/head.ldm";

// Comment out following line to use the predefined compositor node
#define USE_CUSTOM_COMPOSITOR 1

#if USE_CUSTOM_COMPOSITOR 
#include "MedicalCPUDataCompositor.h"
#endif

////////////////////////////////////////////////////////////////////////
// main function

int main(int argc, char **argv)
{
    // Create the window
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow) return 0;

    SoVolumeRendering::init();
    InventorMedical::init();

#if USE_CUSTOM_COMPOSITOR 
    MedicalCPUDataCompositor::initClass();
    SbString envVar = SbString("OIV_USER_LIBS");
    SbString pathToCustomNode = "$OIVHOME/examples/bin/$OIVARCH/Medical/MedicalCPUDataCompositor";
    SoPreferences::setString(envVar, pathToCustomNode);
#endif

    // Assemble the scene graph
    SoRef<SoSeparator> root = new SoSeparator;

    // Camera (must have explicit camera to use Gnomon node)
    root->addChild( new SoPerspectiveCamera() );
    
    // MUST use a multidata separator when combining volumes
    SoMultiDataSeparator *mds = new SoMultiDataSeparator;
      root->addChild(mds);

    // Add some specular reflection to the appearance
    SoMaterial *material = new SoMaterial;
      material->ambientColor.setValue(.0f,.0f,.0f);
      material->diffuseColor.setValue(1,1,1);
      material->transparency.setValue(0.0f);
      material->specularColor.setValue(1,1,1);
      material->shininess.setValue(0.5);
      mds->addChild( material );

    // Change complexity when the dataset move.
    SoInteractiveComplexity *intComplexity = new SoInteractiveComplexity();
      intComplexity->fieldSettings.set1Value(0,"SoComplexity value 0.3 0.9");
      intComplexity->fieldSettings.set1Value(1,"SoVolumeRender samplingAlignment BOUNDARY_ALIGNED BOUNDARY_ALIGNED");
      intComplexity->fieldSettings.set1Value(2,"SoVolumeRenderingQuality preIntegrated FALSE TRUE");
      intComplexity->fieldSettings.set1Value(3,"SoVolumeRender interpolation LINEAR CUBIC");
      intComplexity->refinementDelay = 0;
      mds->addChild( intComplexity );

    mds->addChild( new SoComplexity() );

    // CPU compositor:
    SoDataCompositor* dataCompositor = new SoDataCompositor;
    dataCompositor->dataType = SoDataCompositor::UNSIGNED_BYTE;
    dataCompositor->preDefCompositor = SoDataCompositor::MULTIPLY;

    // Add the data compositor node
    // Either a predefined compositor or a custom compositor
#if USE_CUSTOM_COMPOSITOR
    MedicalCPUDataCompositor *composer = new MedicalCPUDataCompositor;
    // Enable this to experiment generation of RGBA dataset on the fly. 
    // composer->rgbaMode.setValue(TRUE);
    mds->addChild( composer );
#else
    mds->addChild( dataCompositor );
#endif
    const char *filename = (argc >= 2 ? argv[1] : FILENAME);

    // Load volume1 then get size and dimension
    SoVolumeData* pVolData1 = new SoVolumeData;
    pVolData1->fileName     = filename;
    pVolData1->dataSetId    = 1;
    SbBox3f size            = pVolData1->extent.getValue();
    SbVec3i32 vol1dim       = pVolData1->data.getSize();

    // Compute values for volume2 (mask volume)
    int testXdim = vol1dim[0];
    int testYdim = vol1dim[1];
    int testZdim = vol1dim[2];
    int numTestBytes = testXdim * testYdim * testZdim;
    unsigned char *testData = new unsigned char[numTestBytes];
    int counter = 0;
    for (int k = 0; k < testZdim; k++){
        for (int j = 0; j < testYdim; j++){
            for (int i = 0; i < testXdim; i++){
                double phase = sin(4*3.1*j/testYdim) + 4*i*i/(testZdim*testZdim);//same for each given j
                float waveFactor = 23.;
                float t = char(127 + 127*sin((testZdim - k)/waveFactor + phase ));//now make it vary on k
                testData[counter++] = (t > -10 && t < 50) ? 0 : 1;
            }
        }
    }
    mds->addChild( pVolData1 );

    // Create volume2
    //
    // Note: Must have unique dataSetId, but same extent and tile size as volume1
    SoVolumeData* pVolData2 = new SoVolumeData();
      pVolData2->data.setValue(SbVec3i32(testXdim, testYdim, testZdim), SbDataType::UNSIGNED_BYTE,
          0, testData, SoSFArray3D::NO_COPY);  
      pVolData2->dataSetId = 2;
      pVolData2->extent = size;
      pVolData2->ldmResourceParameters.getValue()->tileDimension.setValue(pVolData1->ldmResourceParameters.getValue()->tileDimension.getValue());
      mds->addChild( pVolData2 );

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction* pTransFunc = new SoTransferFunction;
      pTransFunc->predefColorMap = SoTransferFunction::GREY;
      pTransFunc->minValue = 40;
      pTransFunc->maxValue = 250;
      mds->addChild( pTransFunc );

    // Property node
    SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality;
      pVRVolQuality->preIntegrated = TRUE;
      pVRVolQuality->deferredLighting = TRUE;
      pVRVolQuality->ambientOcclusion = TRUE;
      mds->addChild( pVRVolQuality );

    // Node in charge of drawing the volume
    SoVolumeRender* pVolRender = new SoVolumeRender;
      pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
      mds->addChild( pVolRender );

    // Display bounding box of volume
    root->addChild( MedicalHelper::createBoundingBox( pVolData1->extent.getValue() ));

    // Define OIV logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation marker
    root->addChild( new Gnomon() );

    // Set up viewer:
    SoXtExaminerViewer* viewer = new SoXtExaminerViewer(myWindow);
    viewer->setSceneGraph(root.ptr());
    viewer->setTitle("CPU Data Masking");
    viewer->setDecoration(FALSE);
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, viewer->getCamera() );
    viewer->viewAll();
    viewer->saveHomePosition();
    viewer->show();

    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;
    delete [] testData;
    root = NULL;
#if USE_CUSTOM_COMPOSITOR 
    MedicalCPUDataCompositor::exitClass();
#endif
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}
