///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

// Custom data compositor class.
// Used in the CPUDataCompose demo.

#define BUILDING_DLL 1

#include "MedicalCPUDataCompositor.h"

#include <Inventor/devices/SoCpuBufferObject.h>

#include <Inventor/SbElapsedTime.h> 

SO_NODE_SOURCE( MedicalCPUDataCompositor )

//-----------------------------------------------------------------------------
MedicalCPUDataCompositor::MedicalCPUDataCompositor()
{
  SO_NODE_CONSTRUCTOR( MedicalCPUDataCompositor );

  //Automatically set preDefCompositor field to NONE
  preDefCompositor = SoDataCompositor::NONE;
}

//-----------------------------------------------------------------------------
MedicalCPUDataCompositor::~MedicalCPUDataCompositor()
{
}

//-----------------------------------------------------------------------------
void
MedicalCPUDataCompositor::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );
  SO__NODE_INIT_CLASS(MedicalCPUDataCompositor, "MedicalCPUDataCompositor", SoDataCompositor);
}

//-----------------------------------------------------------------------------
void
MedicalCPUDataCompositor::exitClass()
{
  SO__NODE_EXIT_CLASS(MedicalCPUDataCompositor);
}

//-----------------------------------------------------------------------------
// Implement custom compose method
void
MedicalCPUDataCompositor::compose(int /*numDataSet*/, const SbVec3i32& /*tileDimension*/,
               int* /*vdid*/, SoBufferObject** /*inputBuffer*/, SoDataCompositor::DataType* /*dataTypes*/,
               SoBufferObject* /*outputBuffer*/)
{
  SoDebugError::post("MedicalCPUDataCompositor::compose","This demo only implement compose method for converted input");
  return;
}

void 
MedicalCPUDataCompositor::compose( int numVolumeData, const SbVec3i32& tileDimension,
                       int* vdid, SoBufferObject** inputBufferObject, SoBufferObject* outputBufferObject )
{
  if ( rgbaMode.getValue() == FALSE )
  {
    // Check that we are using the right type of data as
    if (dataType.getValue()!=SbDataType::UNSIGNED_BYTE)
    {
      SoDebugError::post("MedicalCPUDataCompositor::compose","Only implemented for UNSIGNED_BYTE data type");
      return;
    }
    composeInCPU( numVolumeData, tileDimension,vdid, inputBufferObject, outputBufferObject );
  }
  else
  {
    composeRGBAInCPU( numVolumeData, tileDimension,vdid, inputBufferObject, outputBufferObject );
  }
}

//-----------------------------------------------------------------------------
// Implement custom compose method
void 
MedicalCPUDataCompositor::composeRGBAInCPU( int numVolumeData, const SbVec3i32& tileDimension,
                       int* /*vdid*/, SoBufferObject** inputBufferObject, SoBufferObject* outputBufferObject )
{
  SbElapsedTime timer;

  // Map the output buffer on CPU to get a classical pointer
  SoCpuBufferObject* cpuObj = new SoCpuBufferObject;
  cpuObj->ref();

  SoCpuBufferObject* cpuInObj = new SoCpuBufferObject;
  cpuInObj->ref();

  outputBufferObject->map(cpuObj,SoCpuBufferObject::SET);
  void *outputBuffer = cpuObj->map(SoCpuBufferObject::SET);

  const size_t nbElem = static_cast<size_t>(tileDimension[0]) * tileDimension[1] * tileDimension[2];

  //loop on each datum of the tile and do RGBA combining
  for(int n = 0; n < numVolumeData; n++ )
  {
    // each volume is colored in a different color
    unsigned int color = ((0xff) << ((n+1)*4) ) ;
    // set alpha to 0
    color = color & 0x00;

    // map the current input buffer inside CPU
    inputBufferObject[n]->map(cpuInObj,SoCpuBufferObject::READ_ONLY);
    void *inputBuffer = cpuInObj->map(SoCpuBufferObject::READ_ONLY);
    {
      unsigned int *pOut = (unsigned int*)(outputBuffer);
      unsigned int *pIn  = (unsigned int*)(inputBuffer);
      
      timer.reset();
      {
        for(size_t ii = 0; ii < nbElem; ++ii )
        {
          if (n==0)
          {
            unsigned int inValue = *(pIn++);
            if ( inValue <40 || inValue>250)
              *(pOut++) = 0;
            else
              *(pOut++) = 0xffffffff;
          }
          else
          {
            *(pOut++) *= *(pIn++);
          }
        }
      }
     
      //double elapsed = timer.getElapsed();
      //printf( "CPU: Compute time: %fs\n", elapsed );

    }
    // unmap the current input buffer
    cpuInObj->unmap();
    inputBufferObject[n]->unmap(cpuInObj);
  }
  cpuInObj->unref();

  // unmap output buffer object
  cpuObj->unmap();
  outputBufferObject->unmap(cpuObj);
  cpuObj->unref();
}

//-----------------------------------------------------------------------------
// Implement custom compose method
void 
MedicalCPUDataCompositor::composeInCPU( int numVolumeData, const SbVec3i32& tileDimension,
                       int* /*vdid*/, SoBufferObject** inputBufferObject, SoBufferObject* outputBufferObject )
{
  SbElapsedTime timer;

  // Map the output buffer on CPU to get a classical pointer
  SoCpuBufferObject* cpuObj = new SoCpuBufferObject;
  cpuObj->ref();

  SoCpuBufferObject* cpuInObj = new SoCpuBufferObject;
  cpuInObj->ref();


  const size_t nbElem = static_cast<size_t>(tileDimension[0]) * tileDimension[1] * tileDimension[2];

  // copy first input buffer inside outputBuffer
  outputBufferObject->memcpy( inputBufferObject[0] );

  outputBufferObject->map(cpuObj,SoCpuBufferObject::SET);
  void *outputBuffer = cpuObj->map(SoCpuBufferObject::SET);

  //loop on each datum of the tile and do multiplication
  for(int n = 1; n < numVolumeData; n++ )
  {
    // map the current input buffer inside CPU
    inputBufferObject[n]->map(cpuInObj,SoCpuBufferObject::READ_ONLY);
    void *inputBuffer = cpuInObj->map(SoCpuBufferObject::READ_ONLY);
    {
      unsigned char *pOut = (unsigned char*)(outputBuffer);
      unsigned char *pIn  = (unsigned char*)(inputBuffer);
      
      timer.reset();
      {
        for(size_t ii = 0; ii < nbElem; ++ii )
          *(pOut++) *= *(pIn++);

      }
     
      //double elapsed = timer.getElapsed();
      //printf( "CPU: Compute time: %fs\n", elapsed );

    }
    // unmap the current input buffer
    cpuInObj->unmap();
    inputBufferObject[n]->unmap(cpuInObj);
  }

  cpuInObj->unref();

  // unmap output buffer object
  cpuObj->unmap();
  outputBufferObject->unmap(cpuObj);
  cpuObj->unref();
}
